% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rjaf.R
\name{rjaf}
\alias{rjaf}
\title{Regularized Joint Assignment Forest with Treatment Arm Clustering}
\usage{
rjaf(
  data.trainest,
  data.heldout,
  y,
  id,
  trt,
  vars,
  prob,
  ntrt = 5,
  nvar = 3,
  lambda1 = 0.5,
  lambda2 = 0.5,
  ipw = TRUE,
  nodesize = 5,
  ntree = 1000,
  prop.train = 0.5,
  eps = 0.1,
  resid = TRUE,
  clus.tree.growing = FALSE,
  clus.outcome.avg = FALSE,
  clus.max = 10,
  reg = TRUE,
  impute = TRUE,
  setseed = FALSE,
  seed = 1,
  nfold = 5
)
}
\arguments{
\item{data.trainest}{input data used for training and estimation, where each
row corresponds to an individual and columns contain information on treatments,
covariates, probabilities of treatment assignment, and observed outcomes.}

\item{data.heldout}{input data used for validation with covariates
and optional counterfactual outcomes.}

\item{y}{a character string denoting the column name of outcomes.}

\item{id}{a character string denoting the column name of individual IDs. If missing,
a column of integers named "id" will be added to \code{data.trainest} and \code{data.heldout}.}

\item{trt}{a character string denoting the column name of treatments.}

\item{vars}{a vector of character strings denoting the column names of covariates.}

\item{prob}{a character string denoting the column name of probabilities of
treatment assignment. If missing, a column named "prob" will be added to \code{data.trainest} and
\code{data.heldout} indicating simple random treatment assignment.}

\item{ntrt}{number of treatments randomly sampled at each split. It should be
at most equal to the number of unique treatments available. The default value is 5.}

\item{nvar}{number of covariates randomly sampled at each split. It should be
at most equal to the number of unique covariates available. The default value is 3.}

\item{lambda1}{regularization parameter for shrinking arm-wise within-leaf average
outcomes towards the overall within-leaf average outcome during recursive
partitioning. The default value is 0.5.}

\item{lambda2}{regularization parameter for shrinking arm-wise within-leaf average
outcomes towards the overall within-leaf average outcome during outcome estimation.
It is only valid when \code{reg} is \code{TRUE}. The default value is 0.5.}

\item{ipw}{a logical indicator of inverse probability weighting when calculating
leaf-wise weighted averages based on Wu and Gagnon-Bartsch (2018). The default value is \code{TRUE}.}

\item{nodesize}{minimum number of observations in a terminal node. The default value is 5.}

\item{ntree}{number of trees to grow in the forest. This should not be set to
too small a number. The default value is 1000.}

\item{prop.train}{proportion of data used for training in \code{data.trainest}.
The default value is 0.5.}

\item{eps}{threshold for minimal welfare gain in terms of the empirical standard
deviation of the overall outcome \code{y}. The default value is 0.1.}

\item{resid}{a logical indicator of arbitrary residualization. If \code{TRUE},
residualization is implemented to reduce the variance of the outcome.
The default value is \code{TRUE}.}

\item{clus.tree.growing}{a logical indicator of clustering for tree growing.
The default value is \code{FALSE}.}

\item{clus.outcome.avg}{a logical indicator of clustering for tree bagging.
If \code{TRUE}, the average outcome is calculated across treatment clusters
determined by the k-means. The default value is \code{FALSE}. This option is deprecated.}

\item{clus.max}{the maximum number of clusters for k-means. It should be
greater than 1 and at most equal to the number of unique treatments.
The default value is 10.}

\item{reg}{a logical indicator of regularization when calculating the arm-wise
within-leaf average outcome.}

\item{impute}{a logical indicator of imputation. If \code{TRUE}, the within-leaf
average outcome is used to impute the arm-wise within-leaf average outcome
when the arm has no observation. If \code{FALSE}, the within-leaf average outcome
is set to zero when the arm has no observation. The default value is \code{TRUE}.}

\item{setseed}{a logical indicator. If \code{TRUE}, a seed is set through the
argument \code{seed} below and passed to the function \code{rjaf_cpp}.
The default value is \code{FALSE}.}

\item{seed}{an integer used as a random seed if \code{setseed=TRUE}.
The default value is 1.}

\item{nfold}{the number of folds used for cross-validation in outcome
residualization and k-means clustering. The default value is 5.}
}
\value{
If \code{clus.tree.growing} and \code{clus.outcome.avg} are \code{TRUE}, \code{rjaf}
returns a list of two objects: a tibble named as \code{res} consisting of individual
IDs, cluster identifiers, and predicted outcomes, and a data frame named as
\code{clustering} consisting of cluster identifiers, probabilities of being assigned
to the clusters, and treatment arms. Otherwise, \code{rjaf}  returns a list of two tibbles
named \code{res} and \code{counterfactuals}. \code{res} consists of individual IDs (\code{id}),
optimal treatment arms identified by the algorithm (\code{trt.rjaf}), treatment
clusters (\code{clus.rjaf}) if \code{clus.tree.growing} is \code{TRUE}, and predicted optimal outcomes (\code{Y.rjaf}).
If counterfactual outcomes are also present, they will be included
in \code{res} along with the column of predicted outcomes (\code{Y.cf}). \code{counterfactuals} consists of
counterfactual estimates of every available treatment.
}
\description{
This algorithm trains a joint forest model to estimate the optimal treatment assignment
by pooling information across treatment arms.
}
\details{
It first obtains an assignment forest by bagging trees as in Kallus (2017) with
covariate and treatment arm randomization for each tree
and estimating "honest" and regularized estimates of the treatment-specific counterfactual outcomes
on the training sample following Wager and Athey (2018).

Like Bonhomme and Manresa (2015), it uses a clustering of treatment arms when
constructing the assignment trees. It employs a k-means algorithm for
clustering the K treatment arms into M treatment groups
based on the K predictions for each of the n units in the training sample.

After clustering, it then repeats the assignment-forest algorithm on the full training data
with M+1 (including control) "arms" (where data from the original arms are combined by groups)
to obtain an ensemble of trees.

It obtains final regularized predictions and assignments, where it estimates
regularized averages separately by the original treatment arms \eqn{k \in \{0,\ldots,K\}}
and obtain the corresponding assignment.
}
\examples{
library(dplyr)
library(MASS)
sim.data <- function(n, K, gamma, sigma, prob=rep(1,K+1)/(K+1)) {
   # K: number of treatment arms
  options(stringsAsFactors=FALSE)
  data <- left_join(data.frame(id=1:n,
                               trt=sample(0:K, n, replace=TRUE, prob),
                               mvrnorm(n, rep(0,3), diag(3))),
                    data.frame(trt=0:K, prob), by="trt")
  data <- mutate(data, tmp1=10+20*(X1>0)-20*(X2>0)-40*(X1>0&X2>0),
                 tmp2=gamma*(2*(X3>0)-1)/(K-1),
                 tmp3=-10*X1^2,
                 Y=tmp1+tmp2*(trt>0)*(2*trt-K-1)+tmp3*(trt==0)+rnorm(n,0,sigma))
  # Y: observed outcomes
  Y.cf <- data.frame(sapply(0:K, function(t) # counterfactual outcomes
    mutate(data, Y=tmp1+tmp2*(t>0)*(2*t-K-1)+tmp3*(t==0))$Y))
  names(Y.cf) <- paste0("Y",0:K)
  return(mutate(bind_cols(dplyr::select(data, -c(tmp1,tmp2,tmp3)), Y.cf),
                across(c(id, trt), as.character)))
}

n <- 200; K <- 3; gamma <- 10; sigma <- 10
Example_data <- sim.data(n, K, gamma, sigma)
Example_trainest <- Example_data \%>\% slice_sample(n = floor(0.5 * nrow(Example_data)))
Example_heldout <- Example_data \%>\% filter(!id \%in\% Example_trainest$id)
id <- "id"; y <- "Y"; trt <- "trt"
vars <- paste0("X", 1:3)
forest.reg <- rjaf(Example_trainest, Example_heldout, y, id, trt, vars, ntrt = 4, ntree = 100,
                   clus.tree.growing = FALSE)

}
\references{
Bonhomme, Stéphane and Elena Manresa (2015). Grouped Patterns of Heterogeneity in Panel Data. Econometrica, 83: 1147-1184.
\cr

Kallus, Nathan (2017). Recursive Partitioning for Personalization using Observational Data. In Precup, Doina and Yee Whye Teh, editors,
Proceedings of the 34th International Conference on Machine Learning, volume 70 of Proceedings of Machine Learning Research, pages 1789–1798. PMLR.
\cr

Wager, Stefan and Susan Athey (2018). Estimation and inference of heterogeneous treatment effects
using random forests. Journal of the American Statistical Association, 113(523):1228–1242.
\cr

Wu, Edward and Johann A Gagnon-Bartsch (2018). The LOOP Estimator: Adjusting
for Covariates in Randomized Experiments. Evaluation Review, 42(4):458–488.
\cr
}
