% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_restorable_constraint.R
\name{add_restorable_constraint}
\alias{add_restorable_constraint}
\title{Add constraint to specify the available amount of surface for restoration}
\usage{
add_restorable_constraint(
  problem,
  min_restore,
  max_restore,
  min_proportion = 1,
  unit = "ha"
)
}
\arguments{
\item{problem}{\code{\link[=restopt_problem]{restopt_problem()}} Restoration problem object.}

\item{min_restore}{\code{numeric} Minimum allowed area to restore in the solution.}

\item{max_restore}{\code{numeric} Maximum allowed area to restore in the solution}

\item{min_proportion}{\code{float} Minimum habitat proportion to consider a cell
as restored.}

\item{unit}{\code{unit} object or a \code{character} that can be coerced to an area
unit (see \code{unit} package), or "cells" for number of cells from the original
habitat raster). If the input habitat raster does not use a projected
coordinate system, only "cells" is available.}
}
\value{
An updated restoration problem (\code{\link[=restopt_problem]{restopt_problem()}}) object.
}
\description{
Add constraint to a restoration problem (\code{\link[=restopt_problem]{restopt_problem()}}) object
to specify specify the available amount of surface for restoration
}
\details{
Given the \code{restorable_habitat} input raster in \link{restopt_problem},
this constraint ensures that the total amount of restorable habitat in
selected planning units is at least \code{min_restore} and at most \code{max_restore}.
The unit of \code{min_restore} and \code{max_restore} can be either in a surface unit handled
by the \code{unit} package, or in number of cells from the original habitat input raster
("cells"). The \code{min_proportion} parameter is a numeric between 0 and 1, and
correspond to the minimum proportion of habitat area that needs to be restored
in the planning unit to consider the planning unit as restored. This proportion
is relative to the area of a planning unit, which is computed automatically
from the input habitat raster. Note that planning unit area is considered
uniform, and the distortion is not corrected. It could be using the \code{cellSize}
function of the \code{terra} package, but this function is currently pretty slow
for large rasters. If your problem is at regional scale, the distortion
should be negligible. However, at larger scales, the best is to use an
equal-area projected coordinate system.

Note that when a solution is found, the "maximum restorable habitat" is
displayed, this value does not correspond to the \code{max_restore} parameter,
but to the total area that can be restored in the selected planning units.
The \code{max_restore} parameter is actually an upper bound of the minimum habitat
that needs to be restored to reach the \code{min_proportion} of habitat in every
selected planning units.
}
\examples{
\donttest{
# load data
habitat_data <- rast(
  system.file("extdata", "habitat_hi_res.tif", package = "restoptr")
)

# create problem
p <- restopt_problem(
    existing_habitat = habitat_data,
    aggregation_factor = 16,
    habitat_threshold = 0.7
  ) \%>\%
  add_restorable_constraint(
    min_restore = 200,
    max_restore = 300,
    min_proportion = 0.7
  ) \%>\%
  add_compactness_constraint(5, unit = "cells")

# print problem
print(p)

# plot preprocessed data
plot(rast(list(get_existing_habitat(p),
               get_restorable_habitat(p),
               get_locked_out_areas(p))), nc = 3)

# Solve problem
s <- solve(p)
# plot solution
plot(s)
}

}
\seealso{
Other constraints: 
\code{\link{add_available_areas_constraint}()},
\code{\link{add_compactness_constraint}()},
\code{\link{add_components_constraint}()},
\code{\link{add_connected_constraint}()},
\code{\link{add_locked_out_constraint}()},
\code{\link{add_min_iic_constraint}()},
\code{\link{add_min_mesh_constraint}()}
}
\concept{constraints}
