
#' Activate a Project
#'
#' Use `activate()` to write the infrastructure needed to ensure that
#' newly-launched \R projects will load the project's private library on launch,
#' alongside any other project-specific state recorded for the project.
#'
#' @inherit renv-params
#'
#' @family renv
#'
#' @export
#'
#' @examples
#' \dontrun{
#'
#' # activate the current project
#' renv::activate()
#'
#' # activate a separate project
#' renv::activate("~/projects/analysis")
#'
#' }
activate <- function(project = NULL) {
  renv_consent_check()
  renv_scope_error_handler()
  project <- project %||% renv_project()
  renv_activate_impl(project, NULL, FALSE, FALSE)
  invisible(project)
}

renv_activate_impl <- function(project, version, restart, quiet) {

  # prepare renv infrastructure
  renv_infrastructure_write(project, version)

  # try to load the project
  load(project, quiet = quiet)

  # ensure renv is installed
  if (!renv_testing())
    renv_bootstrap_self(project = project)

  # restart session
  if (restart)
    renv_request_restart(project, reason = "renv activated")

}

#' Deactivate a Project
#'
#' Use `deactivate()` to remove the infrastructure used by `renv` to activate
#' projects for newly-launched \R sessions. In particular, this implies removing
#' the requisite code from the project `.Rprofile` that automatically activates
#' the project when new \R sessions are launched in the project directory.
#'
#' @inherit renv-params
#'
#' @family renv
#'
#' @export
#'
#' @examples
#' \dontrun{
#'
#' # deactivate the currently-activated project
#' renv::deactivate()
#'
#' }
deactivate <- function(project = NULL) {
  renv_scope_error_handler()
  project <- project %||% renv_project()

  renv_shims_deactivate()
  renv_sandbox_deactivate()

  renv_infrastructure_remove_rprofile(project)
  renv_envvars_restore()
  renv_libpaths_restore()

  renv_request_restart(project, reason = "renv deactivated")
  invisible(project)
}

#' Load a Project
#'
#' Load a project.
#'
#' Normally, this is done automatically on session startup by the infrastructure
#' generated by [activate()] -- users should not need to call this function
#' directly.
#'
#' @inherit renv-params
#'
#' @param quiet Boolean; be quiet during load?
#'
#' @export
#'
#' @examples
#' \dontrun{
#'
#' # load a project -- note that this is normally done automatically
#' # when the R session is started in an renv project after calling
#' # renv::activate()
#' renv::load()
#'
#' }
load <- function(project = NULL, quiet = FALSE) {
  renv_scope_error_handler()
  project <- project %||% renv_project()

  if (quiet)
    renv_scope_options(renv.verbose = FALSE)

  renv_envvars_save()

  # load a minimal amount of state when testing
  if (renv_testing()) {
    renv_load_libpaths(project)
    return(invisible(project))
  }

  renv_load_path(project)
  renv_load_shims(project)
  renv_load_renviron(project)
  renv_load_settings(project)
  renv_load_project(project)
  renv_load_sandbox(project)
  renv_load_libpaths(project)
  renv_load_profile(project)

  lockfile <- renv_lockfile_load(project)
  if (length(lockfile)) {
    renv_load_r(project, lockfile$R)
    renv_load_python(project, lockfile$Python)
  }

  renv_infrastructure_write_rbuildignore(project)
  renv_infrastructure_write_gitignore(project)

  renv_load_finish(project)

  invisible(project)
}

renv_activate_version <- function(project) {

  # try to get version from activate.R
  methods <- list(
    renv_activate_version_lockfile,
    renv_activate_version_activate,
    renv_activate_version_default
  )

  for (method in methods) {
    version <- catch(method(project))
    if (is.character(version))
      return(version)
  }

  fmt <- "failed to determine renv version for project '%s'"
  stopf(fmt, aliased_path(project))

}

renv_activate_version_activate <- function(project) {

  activate <- file.path(project, "renv/activate.R")
  if (!file.exists(activate))
    return(NULL)

  contents <- readLines(activate, warn = FALSE)
  line <- grep("^\\s*version", contents, value = TRUE)
  parsed <- parse(text = line)[[1]]
  parsed[[3]]

}

renv_activate_version_lockfile <- function(project) {

  path <- renv_lockfile_path(project)
  if (!file.exists(path))
    return(NULL)

  lockfile <- renv_lockfile_read(path)
  lockfile$Packages[["renv"]]$Version %||% lockfile$renv$Version

}

renv_activate_version_default <- function(project) {
  renv_namespace_version("renv")
}
