% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redist_smc.R
\name{redist_smc}
\alias{redist_smc}
\title{SMC Redistricting Sampler (McCartan and Imai 2023)}
\usage{
redist_smc(
  map,
  nsims,
  counties = NULL,
  compactness = 1,
  constraints = list(),
  resample = TRUE,
  runs = 1L,
  ncores = 0L,
  init_particles = NULL,
  n_steps = NULL,
  adapt_k_thresh = 0.99,
  seq_alpha = 0.5,
  truncate = (compactness != 1),
  trunc_fn = redist_quantile_trunc,
  pop_temper = 0,
  final_infl = 1,
  est_label_mult = 1,
  ref_name = NULL,
  verbose = FALSE,
  silent = FALSE
)
}
\arguments{
\item{map}{A \code{\link[=redist_map]{redist_map()}} object.}

\item{nsims}{The number of samples to draw.}

\item{counties}{A vector containing county (or other administrative or
geographic unit) labels for each unit, which may be integers ranging from 1
to the number of counties, or a factor or character vector.  If provided,
the algorithm will only generate maps which split up to \code{ndists-1}
counties. Even there are fewer counties than \code{ndists - 1}, the spanning
trees will change the results of the simulations. There is no strength
parameter associated with this constraint. To adjust the number of county
splits further, or to constrain a second type of administrative split,
consider using \code{add_constr_splits()}, \code{add_constr_multisplits()}, and
\code{add_constr_total_splits()}.}

\item{compactness}{Controls the compactness of the generated districts, with
higher values preferring more compact districts. Must be nonnegative. See
the 'Details' section for more information, and computational
considerations.}

\item{constraints}{A \code{\link[=redist_constr]{redist_constr()}} object or a list containing
information on sampling constraints. See \link{constraints} for more information.}

\item{resample}{Whether to perform a final resampling step so that the
generated plans can be used immediately.  Set this to \code{FALSE} to
perform direct importance sampling estimates, or to adjust the weights
manually.}

\item{runs}{How many independent parallel runs to conduct. Each run will
have \code{nsims} simulations. Multiple runs allows for estimation of simulation
standard errors. Output will only be shown for the first run. For
compatibility with MCMC methods, runs are identified with the \code{chain}
column in the output.}

\item{ncores}{How many cores to use to parallelize plan generation within each
run. The default, 0, will use the number of available cores on the machine
as long as \code{nsims} and the number of units is large enough. If \code{runs>1}
you will need to set this manually. If more than one core is used, the
sampler output will not be fully reproducible with \code{set.seed()}. If full
reproducibility is desired, set \code{ncores=1}.}

\item{init_particles}{A matrix of partial plans to begin sampling from. For
advanced use only.  The matrix must have \code{nsims} columns and a row for
every precinct. It is important to ensure that the existing districts meet
contiguity and population constraints, or there may be major issues when
sampling.}

\item{n_steps}{How many steps to run the SMC algorithm for.
Each step splits off a new district. Defaults to all remaining districts.
If fewer than the number of remaining splits, reference plans are disabled.}

\item{adapt_k_thresh}{The threshold value used in the heuristic to select a
value \code{k_i} for each splitting iteration. Higher values are more accurate
but may require more computation. Set to 1 for the most conservative
sampling. Must be between 0 and 1.}

\item{seq_alpha}{The amount to adjust the weights by at each resampling step;
higher values prefer exploitation, while lower values prefer exploration.
Must be between 0 and 1.}

\item{truncate}{Whether to truncate the importance sampling weights at the
final step by \code{trunc_fn}.  Recommended if \code{compactness} is not 1.
Truncation only applied if \code{resample=TRUE}.}

\item{trunc_fn}{A function which takes in a vector of weights and returns a
truncated vector. If the \link[loo:loo]{loo} package is installed (strongly
recommended), will default to Pareto-smoothed Importance Sampling (PSIS)
rather than naive truncation.}

\item{pop_temper}{The strength of the automatic population tempering. Try
values of 0.01-0.05 to start if the algorithm gets stuck on the final few
splits.}

\item{final_infl}{A multiplier for the population constraint on the final
iteration. Used to loosen the constraint when the sampler is getting stuck
on the final split. \code{pop_temper} should be tried first, since using
\code{final_infl} will actually change the target distribution.}

\item{est_label_mult}{A multiplier for the number of importance samples to
use in estimating the number of ways to sequentially label the districts.
Lower values increase speed at the cost of accuracy.  Only applied when
there are more than 13 districts.}

\item{ref_name}{a name for the existing plan, which will be added as a
reference plan, or \code{FALSE} to not include the initial plan in the
output. Defaults to the column name of the existing plan.}

\item{verbose}{Whether to print out intermediate information while sampling.
Recommended.}

\item{silent}{Whether to suppress all diagnostic information.}
}
\value{
\code{redist_smc} returns a \link{redist_plans} object containing the simulated
plans.
}
\description{
\code{redist_smc} uses a Sequential Monte Carlo algorithm (McCartan and Imai 2023)
to generate representative samples of congressional or legislative
redistricting plans according to contiguity, population, compactness, and
administrative boundary constraints.
}
\details{
This function draws samples from a specific target measure controlled by
the \code{map}, \code{compactness}, and \code{constraints} parameters.

Key to ensuring good performance is monitoring the efficiency of the resampling
process at each SMC stage.  Unless \code{silent=FALSE}, this function will print
out the effective sample size of each resampling step to allow the user to
monitor the efficiency.  If \code{verbose=TRUE} the function will also print
out information on the \eqn{k_i} values automatically chosen and the
acceptance rate (based on the population constraint) at each step.
Users should also check diagnostics of the sample by running
\code{summary.redist_plans()}.

Higher values of \code{compactness} sample more compact districts;
setting this parameter to 1 is computationally efficient and generates nicely
compact districts.  Values of other than 1 may lead to highly variable
importance sampling weights.  In these cases, these weights are by default
truncated using \code{\link[=redist_quantile_trunc]{redist_quantile_trunc()}} to stabilize the resulting
estimates, but if truncation is used, a specific truncation function should
probably be chosen by the user.
}
\examples{
\donttest{
data(fl25)

fl_map <- redist_map(fl25, ndists = 3, pop_tol = 0.1)

sampled_basic <- redist_smc(fl_map, 5000)

constr <- redist_constr(fl_map)
constr <- add_constr_incumbency(constr, strength = 100, incumbents = c(3, 6, 25))
sampled_constr <- redist_smc(fl_map, 5000, constraints = constr)

# Multiple parallel independent runs
redist_smc(fl_map, 1000, runs = 2)

# One run with multiple cores
redist_smc(fl_map, 1000, ncores = 2)
}

}
\references{
McCartan, C., & Imai, K. (2023). Sequential Monte Carlo for Sampling
Balanced and Compact Redistricting Plans. \emph{Annals of Applied Statistics} 17(4).
Available at \doi{10.1214/23-AOAS1763}.
}
\concept{simulate}
