% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RecoSys.R
\name{train}
\alias{train}
\title{Training a Recommender Model}
\arguments{
\item{r}{Object returned by \code{\link{Reco}}().}

\item{train_data}{An object of class "DataSource" that describes the source
of training data, typically returned by function
\code{\link{data_file}()} or \code{\link{data_memory}()}.}

\item{out_model}{Path to the model file that will be created.}

\item{opts}{A number of parameters and options for the model training.
See section \strong{Parameters and Options} for details.}
}
\description{
This method is a member function of class "\code{RecoSys}"
that trains a recommender model. It will read from a training data source and
create a model file at the specified location. The model file contains
necessary information for prediction.

The common usage of this method is
\preformatted{r = Reco()
r$train(train_data, out_model = file.path(tempdir(), "model.txt"),
        opts = list())}
}
\section{Parameters and Options}{

The \code{opts} argument is a list that can supply any of the following parameters:

\describe{
\item{\code{loss}}{Character string, the loss function. Default is "l2", see below for details.}
\item{\code{dim}}{Integer, the number of latent factors. Default is 10.}
\item{\code{costp_l1}}{Numeric, L1 regularization parameter for user factors. Default is 0.}
\item{\code{costp_l2}}{Numeric, L2 regularization parameter for user factors. Default is 0.1.}
\item{\code{costq_l1}}{Numeric, L1 regularization parameter for item factors. Default is 0.}
\item{\code{costq_l2}}{Numeric, L2 regularization parameter for item factors. Default is 0.1.}
\item{\code{lrate}}{Numeric, the learning rate, which can be thought
                    of as the step size in gradient descent. Default is 0.1.}
\item{\code{niter}}{Integer, the number of iterations. Default is 20.}
\item{\code{nthread}}{Integer, the number of threads for parallel
                      computing. Default is 1.}
\item{\code{nbin}}{Integer, the number of bins. Must be greater than \code{nthread}.
                   Default is 20.}
\item{\code{nmf}}{Logical, whether to perform non-negative matrix factorization.
                  Default is \code{FALSE}.}
\item{\code{verbose}}{Logical, whether to show detailed information. Default is
                      \code{TRUE}.}
}

The \code{loss} option may take the following values:

For real-valued matrix factorization,

\describe{
\item{\code{"l2"}}{Squared error (L2-norm)}
\item{\code{"l1"}}{Absolute error (L1-norm)}
\item{\code{"kl"}}{Generalized KL-divergence}
}

For binary matrix factorization,

\describe{
\item{\code{"log"}}{Logarithmic error}
\item{\code{"squared_hinge"}}{Squared hinge loss}
\item{\code{"hinge"}}{Hinge loss}
}

For one-class matrix factorization,

\describe{
\item{\code{"row_log"}}{Row-oriented pair-wise logarithmic loss}
\item{\code{"col_log"}}{Column-oriented pair-wise logarithmic loss}
}
}

\examples{
## Training model from a data file
train_set = system.file("dat", "smalltrain.txt", package = "recosystem")
r = Reco()
set.seed(123) # This is a randomized algorithm
r$train(data_file(train_set),
        opts = list(dim = 20, costp_l2 = 0.01, costq_l2 = 0.01, nthread = 1)
)

## Training model from data in memory
train_df = read.table(train_set, sep = " ", header = FALSE)
set.seed(123)
r$train(data_memory(train_df[, 1], train_df[, 2], train_df[, 3]),
        opts = list(dim = 20, costp_l2 = 0.01, costq_l2 = 0.01, nthread = 1)
)

}
\references{
W.-S. Chin, Y. Zhuang, Y.-C. Juan, and C.-J. Lin.
A Fast Parallel Stochastic Gradient Method for Matrix Factorization in Shared Memory Systems.
ACM TIST, 2015.

W.-S. Chin, Y. Zhuang, Y.-C. Juan, and C.-J. Lin.
A Learning-rate Schedule for Stochastic Gradient Methods to Matrix Factorization.
PAKDD, 2015.

W.-S. Chin, B.-W. Yuan, M.-Y. Yang, Y. Zhuang, Y.-C. Juan, and C.-J. Lin.
LIBMF: A Library for Parallel Matrix Factorization in Shared-memory Systems.
Technical report, 2015.
}
\seealso{
\code{$\link{tune}()}, \code{$\link{output}()}, \code{$\link{predict}()}
}
\author{
Yixuan Qiu <\url{http://statr.me}>
}
