% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyse.R
\name{analyse}
\alias{analyse}
\title{Analyse Multiple Imputed Datasets}
\usage{
analyse(
  imputations,
  fun = ancova,
  delta = NULL,
  ...,
  ncores = 1,
  .validate = TRUE
)
}
\arguments{
\item{imputations}{An \code{imputations} object as created by \code{\link[=impute]{impute()}}.}

\item{fun}{An analysis function to be applied to each imputed dataset. See details.}

\item{delta}{A \code{data.frame} containing the delta transformation to be applied to the imputed
datasets prior to running \code{fun}. See details.}

\item{...}{Additional arguments passed onto \code{fun}.}

\item{ncores}{The number of parallel processes to use when running this function. Can also be a
cluster object created by \code{\link[=make_rbmi_cluster]{make_rbmi_cluster()}}. See the parallisation section below.}

\item{.validate}{Should \code{inputations} be checked to ensure it conforms to the required format
(default = \code{TRUE}) ? Can gain a small performance increase if this is set to \code{FALSE} when
analysing a large number of samples.}
}
\description{
This function takes multiple imputed datasets (as generated by
the \code{\link[=impute]{impute()}} function) and runs an analysis function on
each of them.
}
\details{
This function works by performing the following steps:
\enumerate{
\item Extract a dataset from the \code{imputations} object.
\item Apply any delta adjustments as specified by the \code{delta} argument.
\item Run the analysis function \code{fun} on the dataset.
\item Repeat steps 1-3 across all of the datasets inside the \code{imputations}
object.
\item Collect and return all of the analysis results.
}

The analysis function \code{fun} must take a \code{data.frame} as its first
argument. All other options to \code{\link[=analyse]{analyse()}} are passed onto \code{fun}
via \code{...}.
\code{fun} must return a named list with each element itself being a
list containing a single
numeric element called \code{est} (or additionally \code{se} and \code{df} if
you had originally specified \code{\link[=method_bayes]{method_bayes()}} or \code{\link[=method_approxbayes]{method_approxbayes()}})
i.e.:

\if{html}{\out{<div class="sourceCode">}}\preformatted{myfun <- function(dat, ...) \{
    mod_1 <- lm(data = dat, outcome ~ group)
    mod_2 <- lm(data = dat, outcome ~ group + covar)
    x <- list(
        trt_1 = list(
            est = coef(mod_1)[[group]],
            se = sqrt(vcov(mod_1)[group, group]),
            df = df.residual(mod_1)
        ),
        trt_2 = list(
            est = coef(mod_2)[[group]],
            se = sqrt(vcov(mod_2)[group, group]),
            df = df.residual(mod_2)
        )
     )
     return(x)
 \}
}\if{html}{\out{</div>}}

Please note that the \code{vars$subjid} column (as defined in the original call to
\code{\link[=draws]{draws()}}) will be scrambled in the data.frames that are provided to \code{fun}.
This is to say they will not contain the original subject values and as such
any hard coding of subject ids is strictly to be avoided.

By default \code{fun} is the \code{\link[=ancova]{ancova()}} function.
Please note that this function
requires that a \code{vars} object, as created by \code{\link[=set_vars]{set_vars()}}, is provided via
the \code{vars} argument e.g. \code{analyse(imputeObj, vars = set_vars(...))}. Please
see the documentation for \code{\link[=ancova]{ancova()}} for full details.
Please also note that the theoretical justification for the conditional mean imputation
method (\code{method = method_condmean()} in \code{\link[=draws]{draws()}}) relies on the fact that ANCOVA is
a linear transformation of the outcomes.
Thus care is required when applying alternative analysis functions in this setting.

The \code{delta} argument can be used to specify offsets to be applied
to the outcome variable in the imputed datasets prior to the analysis.
This is typically used for sensitivity or tipping point analyses. The
delta dataset must contain columns \code{vars$subjid}, \code{vars$visit} (as specified
in the original call to \code{\link[=draws]{draws()}}) and \code{delta}. Essentially this \code{data.frame}
is merged onto the imputed dataset by \code{vars$subjid} and \code{vars$visit} and then
the outcome variable is modified by:

\if{html}{\out{<div class="sourceCode">}}\preformatted{imputed_data[[vars$outcome]] <- imputed_data[[vars$outcome]] + imputed_data[["delta"]]
}\if{html}{\out{</div>}}

Please note that in order to provide maximum flexibility, the \code{delta} argument
can be used to modify any/all outcome values including those that were not
imputed. Care must be taken when defining offsets. It is recommend that you
use the helper function \code{\link[=delta_template]{delta_template()}} to define the delta datasets as
this provides utility variables such as \code{is_missing} which can be used to identify
exactly which visits have been imputed.
}
\section{Parallelisation}{

To speed up the evaluation of \code{analyse()} you can use the \code{ncores} argument to enable parallelisation.
Simply providing an integer will get \code{rbmi} to automatically spawn that many background processes
to parallelise across. If you are using a custom analysis function then you need to ensure
that any libraries or global objects required by your function are available in the
sub-processes. To do this you need to use the \code{\link[=make_rbmi_cluster]{make_rbmi_cluster()}} function for example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{my_custom_fun <- function(...) <some analysis code>
cl <- make_rbmi_cluster(
    4,
    objects = list("my_custom_fun" = my_custom_fun),
    packages = c("dplyr", "nlme")
)
analyse(
    imputations = imputeObj,
    fun = my_custom_fun,
    ncores = cl
)
parallel::stopCluster(cl)
}\if{html}{\out{</div>}}

Note that there is significant overhead both with setting up the sub-processes and with
transferring data back-and-forth between the main process and the sub-processes. As such
parallelisation of the \code{analyse()} function tends to only be worth it when you have
\verb{> 2000} samples generated by \code{\link[=draws]{draws()}}. Conversely using parallelisation if your samples
are smaller than this may lead to longer run times than just running it sequentially.

It is important to note that the implementation of parallel processing within \code{\link[=analyse]{analyse()}} has
been optimised around the assumption that the parallel processes will be spawned on the same
machine and not a remote cluster. One such optimisation is that the required data is saved to
a temporary file on the local disk from which it is then read into each sub-process. This is
done to avoid the overhead of transferring the data over the network. Our assumption is that
if you are at the stage where you need to be parallelising your analysis over a remote cluster
then you would likely be better off parallelising across multiple \code{rbmi} runs rather than within
a single \code{rbmi} run.

Finally, if you are doing a tipping point analysis you can get a reasonable performance
improvement by re-using the cluster between each call to \code{analyse()} e.g.

\if{html}{\out{<div class="sourceCode">}}\preformatted{cl <- make_rbmi_cluster(4)
ana_1 <- analyse(
    imputations = imputeObj,
    delta = delta_plan_1,
    ncores = cl
)
ana_2 <- analyse(
    imputations = imputeObj,
    delta = delta_plan_2,
    ncores = cl
)
ana_3 <- analyse(
    imputations = imputeObj,
    delta = delta_plan_3,
    ncores = cl
)
parallel::clusterStop(cl)
}\if{html}{\out{</div>}}
}

\examples{
\dontrun{
vars <- set_vars(
    subjid = "subjid",
    visit = "visit",
    outcome = "outcome",
    group = "group",
    covariates = c("sex", "age", "sex*age")
)

analyse(
    imputations = imputeObj,
    vars = vars
)

deltadf <- data.frame(
    subjid = c("Pt1", "Pt1", "Pt2"),
    visit = c("Visit_1", "Visit_2", "Visit_2"),
    delta = c( 5, 9, -10)
)

analyse(
    imputations = imputeObj,
    delta = deltadf,
    vars = vars
)
}
}
\seealso{
\code{\link[=extract_imputed_dfs]{extract_imputed_dfs()}} for manually extracting imputed
datasets.

\code{\link[=delta_template]{delta_template()}} for creating delta data.frames.

\code{\link[=ancova]{ancova()}} for the default analysis function.
}
