% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bison.R
\name{bison}
\alias{bison}
\title{Search for and collect data from the USGS Bison API.}
\usage{
bison(
  species = NULL,
  type = "scientific_name",
  tsn = NULL,
  start = 0,
  count = 25,
  countyFips = NULL,
  county = NULL,
  state = NULL,
  aoi = NULL,
  aoibbox = NULL,
  params = NULL,
  ...
)
}
\arguments{
\item{species}{(character) A species name.}

\item{type}{(character) Type, one of scientific_name or common_name.}

\item{tsn}{(numeric) Specifies the TSN to query by. If you supply a tsn it
doesn't make sense to supply a species name as well. Example:162003.}

\item{start}{(numeric) Record to start at. Default: 0. See "Pagination"
in Details.}

\item{count}{(numeric) Number of records to return. Default: 25.
See "Pagination" in Details.}

\item{countyFips}{(character) Specifies the county fips code to
geographically constrain the search to one county. Character must be
supplied as a number starting with zero may lose the zero. Eg: "49015".}

\item{county}{(character) County name. As codes are a pain in the ass, you
can put in the county name here instead of specifying a countyFips entry,
and bison will attempt to look up the countyFips code. (character)}

\item{state}{(character) Specifies the state name to geographically
constrain the search. Example: Tennessee.}

\item{aoi}{(character) Specifies a WKT (Well-Known Text) polygon to
geographically  constrain the search.Eg.: c(-111.06 38.84,
-110.80 39.377,
-110.20 39.17,
-110.20 38.90,
-110.63 38.67,
-111.06 38.84),
which calls up the occurrences within the specified area. Check out
the Wikipedia page here \url{http://en.wikipedia.org/wiki/Well-known_text}
for an in depth look at the options, terminology, etc.}

\item{aoibbox}{(character) Specifies a four-sided bounding box to
geographically constrain the search (using format: minx,miny,maxx,maxy).
The coordinates are Spherical Mercator with a datum of WGS84. Example:
-111.31,38.81,-110.57,39.21}

\item{params}{(character) String of parameters, one of providerID,
resourceID, basisOfRecord, catalogNumber, year, computedStateFips,
hierarchy_homonym_string, TSNs, recordedBy, occurrenceID, collectorNumber,
provider, ownerInstitutionCollectionCode, eventDate, providedScientificName,
scientificName, ITISscientificName, providedCommonName, ITIScommonName,
kingdom, ITIStsn, centroid, higherGeographyID, computedCountyFips,
providedCounty, calculatedCounty, stateProvince, calculatedState,
countryCode. See examples.}

\item{...}{Further args passed on to \code{\link[crul:HttpClient]{crul::HttpClient()}}. See examples.}
}
\description{
Search for and collect data from the USGS Bison API.
}
\section{Pagination}{

\code{bison()} paginates internally for you on the \code{count} parameter, so that
for example, if you request 2000 records, then we'll do two requests to
get all those records. If you request for example 50 records, then we
just do one request.
}

\examples{
\dontrun{
bison(species="Bison bison", count=50)

# lots of results
res <- bison(species="Bison bison", count=2000)
res$summary
NROW(res$points)

out <- bison(species="Helianthus annuus", count=300)
out$summary # see summary
out$counties # see county data
out$states # see state data
bisonmap(out, tomap = "points")
bisonmap(out, tomap = "county")
bisonmap(out, tomap = "state")

# Search for a common name
bison(species="Tufted Titmouse", type="common_name")

# Constrain search to a certain county, 49015 is Emery County in Utah
bison(species="Helianthus annuus", countyFips = "49015")

# Constrain search to a certain county, specifying county name instead of 
# code
bison(species="Helianthus annuus", county = "Los Angeles")
# bison(species="Helianthus annuus", county = "Los")

# Constrain search to a certain aoi, which turns out to be Emery County, 
# Utah as well
bison(species="Helianthus annuus",
 aoi = "POLYGON((-111.06360117772908 38.84001566645886,
                 -110.80542246679359 39.37707771107983,
                 -110.20117441992392 39.17722368276862,
                 -110.20666758398464 38.90844075244811,
                 -110.63513438085685 38.67724220095734,
                 -111.06360117772908 38.84001566645886))")

# Constrain search to a certain aoibbox, which, you guessed it, is also 
# Emery Co., Utah
bison(species="Helianthus annuus", aoibbox = '-111.31,38.81,-110.57,39.21')

# Taxonomic serial number
bison(tsn = 162003)
## If you don't have tsn's, search for a taxonomic serial number
library('taxize')
poa_tsn <- get_tsn('Poa annua')
bison(tsn = poa_tsn)

# Curl debugging and some of these examples aren't 
# that useful, but are given for demonstration purposes
## get curl verbose output to see what's going on with your request
bison(tsn = 162003, count=1, verbose = TRUE)
## set a timeout so that the call stops after time x, compare 1st to 2nd call
# bison(tsn=162003, count=1, timeout_ms = 1)
## set cookies
bison(tsn=162003, count=1, cookie = "a=1;b=2")
## user agent and verbose 
bison(tsn=162003, count=1, useragent = "rbison", 
  verbose = TRUE)

# Params - the params function accepts a number of search terms
## Find the provider with ID 318.
bison(params='providerID:("318")')
## Find all resources with id of '318,1902' OR '318,9151', with values 
## separated by spaces.
bison(params='resourceID:("318,1902" "318,9151")')
## Criterion may be combined using the semicolon (';') character, which 
## translates to a logical AND operator. Note that field names and values 
## are case sensitive.
bison(params='providerID:("408" "432");resourceID:("14027")')
## Search by basisOfRecord, for specimen types in this case
bison(params='basisOfRecord:(specimen)')
## Search by computedStateFips, 01 for Alabama
bison(params='computedStateFips:01')
## Search by ITIStsn
bison(params='ITIStsn:162003')
## Search by countryCode
bison(params='countryCode:US')
## Search by ITIScommonName
bison(params='ITIScommonName:"Canada goose"')
}
}
\references{
\url{https://bison.usgs.gov/#opensearch}
}
\seealso{
\code{\link[=bison_solr]{bison_solr()}} \code{\link[=bison_tax]{bison_tax()}}
}
