% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fractional.computations.R
\name{spde.matern.loglike}
\alias{spde.matern.loglike}
\title{Parameter-based log-likelihood for a latent Gaussian Matern SPDE model
using a rational SPDE approximation}
\usage{
spde.matern.loglike(
  object,
  Y,
  A,
  sigma.e,
  mu = 0,
  user_nu = NULL,
  user_kappa = NULL,
  user_tau = NULL,
  user_theta = NULL,
  user_m = NULL,
  pivot = TRUE
)
}
\arguments{
\item{object}{The rational SPDE approximation,
computed using \code{\link[=spde.matern.operators]{spde.matern.operators()}}}

\item{Y}{The observations, either a vector or a matrix where
the columns correspond to independent replicates of observations.}

\item{A}{An observation matrix that links the measurement location to the
finite element basis.}

\item{sigma.e}{IF non-null, the standard deviation of the measurement noise will be kept fixed in
the returned likelihood.}

\item{mu}{Expectation vector of the latent field (default = 0).}

\item{user_nu}{If non-null, the shape parameter will be kept fixed in the returned likelihood.}

\item{user_kappa}{If non-null, updates the range parameter.}

\item{user_tau}{If non-null, updates the parameter tau.}

\item{user_theta}{If non-null, updates the parameter theta (that connects tau and kappa to the model matrices in \code{object}).}

\item{user_m}{If non-null, update the order of the rational approximation,
which needs to be a positive integer.}

\item{pivot}{Should pivoting be used for the Cholesky decompositions?
Default is TRUE}
}
\value{
The log-likelihood value.
}
\description{
This function evaluates the log-likelihood function for observations
of a Gaussian process defined as the solution to the SPDE
\deqn{(\kappa(s) - \Delta)^\beta (\tau(s)u(s)) = W.}
}
\details{
The observations are assumed to be generated as
\eqn{Y_i = u(s_i) + \epsilon_i}{Y_i = u(s_i) + \epsilon_i}, where
\eqn{\epsilon_i}{\epsilon_i} are
iid mean-zero Gaussian variables. The latent model is approximated using a
rational approximation of the fractional SPDE model.
}
\examples{
# this example illustrates how the function can be used for maximum
# likelihood estimation
# Sample a Gaussian Matern process on R using a rational approximation
sigma.e <- 0.1
n.rep <- 10
n.obs <- 100
n.x <- 51
# create mass and stiffness matrices for a FEM discretization
x <- seq(from = 0, to = 1, length.out = n.x)
fem <- rSPDE.fem1d(x)
tau <- rep(0.5, n.x)
nu <- 0.8
kappa <- rep(1, n.x)
# compute rational approximation
op <- spde.matern.operators(
  kappa = kappa, tau = tau, nu = nu,
  G = fem$G, C = fem$C, d = 1
)
# Sample the model
u <- simulate(op, n.rep)
# Create some data
obs.loc <- runif(n = n.obs, min = 0, max = 1)
A <- rSPDE.A1d(x, obs.loc)
noise <- rnorm(n.obs * n.rep)
dim(noise) <- c(n.obs, n.rep)
Y <- as.matrix(A \%*\% u + sigma.e * noise)
# define negative likelihood function for optimization using matern.loglike
mlik <- function(theta) {
  return(-spde.matern.loglike(op, Y, A, sigma.e = exp(theta[4]),
                                 user_nu = exp(theta[3]),
                                 user_kappa = exp(theta[2]),
                                 user_tau = exp(theta[1]),
                                 pivot = TRUE))
}
#' #The parameters can now be estimated by minimizing mlik with optim
\donttest{
# Choose some reasonable starting values depending on the size of the domain
theta0 <- log(c(1 / sqrt(var(c(Y))), sqrt(8), 0.9, 0.01))
# run estimation and display the results
theta <- optim(theta0, mlik)
print(data.frame(
  tau = c(tau[1], exp(theta$par[1])), kappa = c(kappa[1], exp(theta$par[2])),
  nu = c(nu, exp(theta$par[3])), sigma.e = c(sigma.e, exp(theta$par[4])),
  row.names = c("Truth", "Estimates")
))
}
}
\seealso{
\code{\link[=matern.loglike]{matern.loglike()}}, \code{\link[=rSPDE.loglike]{rSPDE.loglike()}}.
}
