#' Generate the QR code as an svg file
#'
#' Create the QR code using [qr_code()] and save it as an svg file.
#' @param qrcode a `qr_code` object as generated by `qr_code`.
#' @param filename Where to store the filename.
#'   Silently overwrites existing files.
#'   Tries to create the path, when it doesn't exist.
#' @param size size of the svg file in pixels.
#' @param foreground Stroke and fill colour for the foreground.
#'   Use a valid [CSS colour](https://www.w3schools.com/colors/).
#'   Defaults to `"black"`.
#' @param background Fill colour for the background.
#'   Use a valid [CSS colour](https://www.w3schools.com/colors/).
#'   Defaults to `"white"`.
#' @param show Open the file after creating it.
#'   Defaults to `TRUE` on [interactive()] sessions, otherwise `FALSE`.
#' @return invisible `NULL`
#' @examples
#' code <- qr_code("HELLO WORLD")
#' generate_svg(
#'   qrcode = code, filename = tempfile(fileext = ".svg"), show = FALSE
#' )
#' @export
#' @importFrom assertthat assert_that
#' @importFrom utils browseURL
#' @author Thierry Onkelinx
#' @family qr
generate_svg <- function(
  qrcode, filename, size = 100, foreground = "black", background = "white",
  show = interactive()
) {
  assert_that(inherits(qrcode, "qr_code"))
  heading <- c(
    "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>",
    sprintf(
      paste(
        "<svg xmlns=\"http://www.w3.org/2000/svg\"",
        "xmlns:xlink=\"http://www.w3.org/1999/xlink\"",
        "height=\"%1$i\" width=\"%1$i\">"
      ),
      size
    ),
    sprintf("  <g id=\"qrcode:%s\">", attr(qrcode, "string")),
    sprintf(
      paste(
        "    <rect x=\"0\" y=\"0\" width=\"%1$ipx\" height=\"%1$ipx\"",
        "style=\"fill:%2$s;\"/>"
      ),
      size, background
    )
  )
  footing <- c("  </g>", "</svg>")
  pixel <- size / ncol(qrcode)
  top_left <- (which(qrcode, arr.ind = TRUE) - 1) * pixel
  svg_data <- sprintf(
    paste(
      "    <rect x=\"%.2f\" y=\"%.2f\" width=\"%3$0.2f\" height=\"%3$0.2f\"",
      "stroke=\"%4$s\" stroke-width=\"0.2\" stroke-linecap=\"butt\"",
      "fill=\"%4$s\"/>"
    ),
    top_left[, 2], top_left[, 1], pixel, foreground
  )
  dir.create(dirname(filename), showWarnings = FALSE, recursive = TRUE)
  writeLines(c(heading, svg_data, footing), filename)
  if (show) {
    browseURL(filename)
  }
  return(invisible(NULL))
}
