% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/etc_utils_autoplot.R, R/g_autoplot.R
\name{autoplot}
\alias{autoplot}
\alias{autoplot.sscurves}
\alias{autoplot.mscurves}
\alias{autoplot.smcurves}
\alias{autoplot.mmcurves}
\alias{autoplot.sspoints}
\alias{autoplot.mspoints}
\alias{autoplot.smpoints}
\alias{autoplot.mmpoints}
\title{Plot performance evaluation measures with ggplot2}
\usage{
\method{autoplot}{sscurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{mscurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{smcurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{mmcurves}(object, curvetype = c("ROC", "PRC"), ...)

\method{autoplot}{sspoints}(object,
  curvetype = .get_metric_names("basic"), ...)

\method{autoplot}{mspoints}(object,
  curvetype = .get_metric_names("basic"), ...)

\method{autoplot}{smpoints}(object,
  curvetype = .get_metric_names("basic"), ...)

\method{autoplot}{mmpoints}(object,
  curvetype = .get_metric_names("basic"), ...)
}
\arguments{
\item{object}{An \code{S3} object generated by \code{\link{evalmod}}.
  The \code{autoplot} function accepts the following code{S3} objects for two
  different modes, "rocprc" and "basic".

\enumerate{

  \item ROC and Precision-Recall curves (\code{mode = "rocprc"})

  \tabular{lll}{
    \strong{\code{S3} object}
    \tab \strong{# of models}
    \tab \strong{# of test datasets} \cr

    sscurves \tab single   \tab single   \cr
    mscurves \tab multiple \tab single   \cr
    smcurves \tab single   \tab multiple \cr
    mmcurves \tab multiple \tab multiple
  }

  \item Basic evaluation measures (\code{mode = "basic"})

  \tabular{lll}{
    \strong{\code{S3} object}
    \tab \strong{# of models}
    \tab \strong{# of test datasets} \cr

    sspoints \tab single   \tab single   \cr
    mspoints \tab multiple \tab single   \cr
    smpoints \tab single   \tab multiple \cr
    mmpoints \tab multiple \tab multiple
  }
}

See the \strong{Value} section of \code{\link{evalmod}} for more details.}

\item{curvetype}{A character vector with the following curve types.
\enumerate{

  \item ROC and Precision-Recall curves (mode = "rocprc")
    \tabular{ll}{
      \strong{curvetype}
      \tab \strong{description} \cr

      ROC \tab ROC curve \cr
      PRC \tab Precision-Recall curve
    }
    Multiple \code{curvetype} can be combined, such as
    \code{c("ROC", "PRC")}.

  \item Basic evaluation measures (mode = "basic")
    \tabular{ll}{
      \strong{curvetype}
      \tab \strong{description} \cr

      error \tab Normalized ranks vs. error rate \cr
      accuracy \tab Normalized ranks vs. accuracy \cr
      specificity \tab Normalized ranks vs. specificity \cr
      sensitivity \tab Normalized ranks vs. sensitivity \cr
      precision \tab Normalized ranks vs. precision \cr
      mcc \tab Normalized ranks vs. Matthews correlation coefficient \cr
      fscore \tab Normalized ranks vs. F-score
   }
   Multiple \code{curvetype} can be combined, such as
   \code{c("precision", "sensitivity")}.
}}

\item{...}{Following additional arguments can be specified.

\describe{
  \item{type}{
    A character to specify the line type as follows.
    \describe{
      \item{"l"}{lines}
      \item{"p"}{points}
      \item{"b"}{both lines and points}
    }
  }
  \item{show_cb}{
    A Boolean value to specify whether point-wise confidence
    bounds are drawn. It is effective only when \code{calc_avg} of the
   \code{\link{evalmod}} function is set to \code{TRUE} .
  }
  \item{raw_curves}{
    A Boolean value to specify whether raw curves are
    shown instead of the average curve. It is effective only
    when \code{raw_curves} of the \code{\link{evalmod}} function is set to
    \code{TRUE}.
  }
  \item{show_legend}{
    A Boolean value to specify whether the legend is shown.
  }
  \item{ret_grob}{
    A logical value to indicate whether
    \code{autoplot} returns a \code{grob} object. The \code{grob} object
    is internally generated by \code{\link[gridExtra]{arrangeGrob}}.
    The \code{\link[grid]{grid.draw}} function takes a \code{grob} object and
    shows a plot. It is effective only when a multiple-panel plot is
    generated, for example, when \code{curvetype} is \code{c("ROC", "PRC")}.
  }
  \item{reduce_points}{
    A Boolean value to decide whether the points should be reduced
    when \code{mode = "rocprc"}. The points are reduced according to
    \code{x_bins} of the \code{\link{evalmod}} function.
    The default values is \code{TRUE}.
  }
}}
}
\value{
The \code{autoplot} function returns a \code{ggplot} object
  for a single-panel plot and a frame-grob object for a multiple-panel plot.
}
\description{
The \code{autoplot} function plots performance evaluation measures
  by using \pkg{ggplot2} instead of the general R plot.
}
\examples{
\dontrun{

## Load libraries
library(ggplot2)
library(grid)

##################################################
### Single model & single test dataset
###

## Load a dataset with 10 positives and 10 negatives
data(P10N10)

## Generate an sscurve object that contains ROC and Precision-Recall curves
sscurves <- evalmod(scores = P10N10$scores, labels = P10N10$labels)

## Plot both ROC and Precision-Recall curves
autoplot(sscurves)

## Reduced/Full supporting points
sampss <- create_sim_samples(1, 50000, 50000)
evalss <- evalmod(scores = sampss$scores, labels = sampss$labels)

# Reduced supporting point
system.time(autoplot(evalss))

# Full supporting points
system.time(autoplot(evalss, reduce_points = FALSE))

## Get a grob object for multiple plots
pp1 <- autoplot(sscurves, ret_grob = TRUE)
plot.new()
grid.draw(pp1)

## A ROC curve
autoplot(sscurves, curvetype = "ROC")

## A Precision-Recall curve
autoplot(sscurves, curvetype = "PRC")

## Generate an sspoints object that contains basic evaluation measures
sspoints <- evalmod(mode = "basic", scores = P10N10$scores,
                    labels = P10N10$labels)

## Normalized ranks vs. basic evaluation measures
autoplot(sspoints)

## Normalized ranks vs. precision
autoplot(sspoints, curvetype = "precision")


##################################################
### Multiple models & single test dataset
###

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(1, 100, 100, "all")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]])

## Generate an mscurve object that contains ROC and Precision-Recall curves
mscurves <- evalmod(mdat)

## ROC and Precision-Recall curves
autoplot(mscurves)

## Reduced/Full supporting points
sampms <- create_sim_samples(5, 50000, 50000)
evalms <- evalmod(scores = sampms$scores, labels = sampms$labels)

# Reduced supporting point
system.time(autoplot(evalms))

# Full supporting points
system.time(autoplot(evalms, reduce_points = FALSE))

## Hide the legend
autoplot(mscurves, show_legend = FALSE)

## Generate an mspoints object that contains basic evaluation measures
mspoints <- evalmod(mdat, mode = "basic")

## Normalized ranks vs. basic evaluation measures
autoplot(mspoints)

## Hide the legend
autoplot(mspoints, show_legend = FALSE)


##################################################
### Single model & multiple test datasets
###

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(10, 100, 100, "good_er")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]],
               dsids = samps[["dsids"]])

## Generate an smcurve object that contains ROC and Precision-Recall curves
smcurves <- evalmod(mdat, raw_curves = TRUE)

## Average ROC and Precision-Recall curves
autoplot(smcurves, raw_curves = FALSE)

## Hide confidence bounds
autoplot(smcurves, raw_curves = FALSE, show_cb = FALSE)

## Raw ROC and Precision-Recall curves
autoplot(smcurves, raw_curves = TRUE, show_cb = FALSE)

## Reduced/Full supporting points
sampsm <- create_sim_samples(4, 5000, 5000)
mdatsm <- mmdata(sampsm$scores, sampsm$labels, expd_first = "dsids")
evalsm <- evalmod(mdatsm, raw_curves = TRUE)

# Reduced supporting point
system.time(autoplot(evalsm, raw_curves = TRUE))

# Full supporting points
system.time(autoplot(evalsm, raw_curves = TRUE, reduce_points = FALSE))

## Generate an smpoints object that contains basic evaluation measures
smpoints <- evalmod(mdat, mode = "basic")

## Normalized ranks vs. average basic evaluation measures
autoplot(smpoints)


##################################################
### Multiple models & multiple test datasets
###

## Create sample datasets with 100 positives and 100 negatives
samps <- create_sim_samples(10, 100, 100, "all")
mdat <- mmdata(samps[["scores"]], samps[["labels"]],
               modnames = samps[["modnames"]],
               dsids = samps[["dsids"]])

## Generate an mscurve object that contains ROC and Precision-Recall curves
mmcurves <- evalmod(mdat, raw_curves = TRUE)

## Average ROC and Precision-Recall curves
autoplot(mmcurves, raw_curves = FALSE)

## Show confidence bounds
autoplot(mmcurves, raw_curves = FALSE, show_cb = TRUE)

## Raw ROC and Precision-Recall curves
autoplot(mmcurves, raw_curves = TRUE)

## Reduced/Full supporting points
sampmm <- create_sim_samples(4, 5000, 5000)
mdatmm <- mmdata(sampmm$scores, sampmm$labels, modnames = c("m1", "m2"),
                 dsids = c(1, 2), expd_first = "modnames")
evalmm <- evalmod(mdatmm, raw_curves = TRUE)

# Reduced supporting point
system.time(autoplot(evalmm, raw_curves = TRUE))

# Full supporting points
system.time(autoplot(evalmm, raw_curves = TRUE, reduce_points = FALSE))

## Generate an mmpoints object that contains basic evaluation measures
mmpoints <- evalmod(mdat, mode = "basic")

## Normalized ranks vs. average basic evaluation measures
autoplot(mmpoints)


##################################################
### N-fold cross validation datasets
###

## Load test data
data(M2N50F5)

## Speficy nessesary columns to create mdat
cvdat <- mmdata(nfold_df = M2N50F5, score_cols = c(1, 2),
                lab_col = 3, fold_col = 4,
                modnames = c("m1", "m2"), dsids = 1:5)

## Generate an mmcurve object that contains ROC and Precision-Recall curves
cvcurves <- evalmod(cvdat)

## Average ROC and Precision-Recall curves
autoplot(cvcurves)

## Show confidence bounds
autoplot(cvcurves, show_cb = TRUE)

## Generate an mmpoints object that contains basic evaluation measures
cvpoints <- evalmod(cvdat, mode = "basic")

## Normalized ranks vs. average basic evaluation measures
autoplot(cvpoints)

}

}
\seealso{
\code{\link{evalmod}} for generating an \code{S3} object.
  \code{\link{fortify}} for converting a curves and points object
  to a data frame.  \code{\link{plot}} for plotting the equivalent curves
  with the general R plot.
}
