% Generated by roxygen2 (4.0.1): do not edit by hand
\name{mlg}
\alias{mlg}
\alias{mlg.crosspop}
\alias{mlg.id}
\alias{mlg.table}
\alias{mlg.vector}
\title{Create counts, vectors, and matrices of multilocus genotypes.}
\usage{
mlg(pop, quiet = FALSE)

mlg.table(pop, sublist = "ALL", blacklist = NULL, mlgsub = NULL,
  bar = TRUE, total = FALSE, quiet = FALSE)

mlg.vector(pop)

mlg.crosspop(pop, sublist = "ALL", blacklist = NULL, mlgsub = NULL,
  indexreturn = FALSE, df = FALSE, quiet = FALSE)

mlg.id(pop)
}
\arguments{
\item{pop}{a \code{\linkS4class{genind}} or \code{\linkS4class{genclone}} object.}

\item{sublist}{a \code{vector} of population names or indices that the user
wishes to keep. Default to "ALL".}

\item{blacklist}{a \code{vector} of population names or indices that the user
wishes to discard. Default to \code{NULL}.}

\item{mlgsub}{a \code{vector} of multilocus genotype indices with which to
subset \code{mlg.table} and \code{mlg.crosspop}. NOTE: The resulting table
from \code{mlg.table} will only contain countries with those MLGs}

\item{quiet}{\code{Logical}. If FALSE, progress of functions will be printed
to the screen.}

\item{bar}{\code{logical} If \code{TRUE}, a bar graph for each population
will be displayed showing the relative abundance of each MLG within the
population.}

\item{indexreturn}{\code{logical} If \code{TRUE}, a vector will be returned
to index the columns of \code{mlg.table}.}

\item{df}{\code{logical} If \code{TRUE}, return a data frame containing the
counts of the MLGs and what countries they are in. Useful for making graphs
with \code{\link{ggplot}}.}

\item{total}{\code{logical} If \code{TRUE}, a row containing the sum of all
represented MLGs is appended to the matrix produced by mlg.table.}
}
\value{
\subsection{mlg}{
an integer describing the number of multilocus genotypes observed.
}
\subsection{mlg.table}{
a matrix with columns indicating unique multilocus genotypes and rows
indicating populations.
}
\subsection{mlg.vector}{
a numeric vector naming the multilocus genotype of each individual in
  the dataset.
}
\subsection{mlg.crosspop}{
\itemize{
\item{default}{ a \code{list} where each element contains a named integer vector representing the number of individuals represented from each population in that MLG}
\item{\code{indexreturn = TRUE}}{ a \code{vector} of integers defining the multilocus genotypes that have individuals crossing populations}
\item{\code{df = TRUE}}{ A long form data frame with the columns: MLG, Population, Count. Useful for graphing with ggplot2}
}
}
\subsection{mlg.id}{
a list of multilocus genotypes with the associated individual names per MLG.
}
}
\description{
Create counts, vectors, and matrices of multilocus genotypes.
}
\note{
The resulting matrix of \code{mlg.table} can be used for analysis with
the \code{\link{vegan}} package.

mlg.vector will recalculate the mlg vector for
  \code{\linkS4class{genind}} objects and will return the contents of the mlg
  slot in \code{\linkS4class{genclone}} objects. This means that MLGs will be
  different for subsetted \code{\linkS4class{genind}} objects.
}
\examples{
# Load the data set
data(Aeut)

# Investigate the number of multilocus genotypes.
amlg <- mlg(Aeut)
amlg # 119

# show the multilocus genotype vector
avec <- mlg.vector(Aeut)
avec

# Get a table
atab <- mlg.table(Aeut, bar = FALSE)
atab

# See where multilocus genotypes cross populations
acrs <- mlg.crosspop(Aeut) # MLG.59: (2 inds) Athena Mt. Vernon

# See which individuals belong to each MLG
aid <- mlg.id(Aeut)
aid["59"] # individuals 159 and 57

\dontrun{

# A simple example. 10 individuals, 5 genotypes.
mat1 <- matrix(ncol=5, 25:1)
mat1 <- rbind(mat1, mat1)
mat <- matrix(nrow=10, ncol=5, paste(mat1,mat1,sep="/"))
mat.gid <- df2genind(mat, sep="/")
mlg(mat.gid)
mlg.vector(mat.gid)
mlg.table(mat.gid)

# Now for a more complicated example.
# Data set of 1903 samples of the H3N2 flu virus genotyped at 125 SNP loci.
data(H3N2)
mlg(H3N2, quiet=FALSE)

H.vec <- mlg.vector(H3N2)

# Changing the population vector to indicate the years of each epidemic.
pop(H3N2) <- other(H3N2)$x$country
H.tab <- mlg.table(H3N2, bar=FALSE, total=TRUE)

# Show which genotypes exist accross populations in the entire dataset.
res <- mlg.crosspop(H3N2, quiet=FALSE)

# Let's say we want to visualize the multilocus genotype distribution for the
# USA and Russia
mlg.table(H3N2, sublist=c("USA", "Russia"), bar=TRUE)

# An exercise in subsetting the output of mlg.table and mlg.vector.
# First, get the indices of each MLG duplicated across populations.
inds <- mlg.crosspop(H3N2, quiet=FALSE, indexreturn=TRUE)

# Since the columns of the table from mlg.table are equal to the number of
# MLGs, we can subset with just the columns.
H.sub <- H.tab[, inds]

# We can also do the same by using the mlgsub flag.
H.sub <- mlg.table(H3N2, mlgsub=inds)

# We can subset the original data set using the output of mlg.vector to
# analyze only the MLGs that are duplicated across populations.
new.H <- H3N2[H.vec \%in\% inds, ]

}
}
\author{
Zhian N. Kamvar
}
\seealso{
\code{\link[vegan]{diversity}} \code{\link{popsub}}
}

