% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reac.R
\name{reac}
\alias{reac}
\title{Calculate reactivity and first-timestep attenuation}
\usage{
reac(A, vector = "n", bound = NULL, return.N = FALSE)
}
\arguments{
\item{A}{a square, non-negative numeric matrix of any dimension}

\item{vector}{(optional) a numeric vector or one-column matrix describing 
the age/stage distribution used to calculate a 'case-specific' reactivity/
first-timestep attenuation}

\item{bound}{(optional) specifies whether an upper or lower bound should be 
calculated (see details).}

\item{return.N}{(optional) if \code{TRUE}, returns population size in the 
first time interval (including effects of asymptotic growth and initial 
population size), alongside standardised reactivity/first-timestep attenuation.}
}
\value{
If \code{vector="n"}, the upper bound on reactivity of \code{A} if 
\code{bound="upper"} and the lower bound on first-timestep attenuation of 
\code{A} if \code{bound="lower"}.\cr
If \code{vector} is specified, the 'case-specific' reactivity or first-timestep 
attenuation of the model.

If \code{return.N=TRUE}, a list with components:
\describe{
\item{reac}{the bound on or case-specific reactivity or first-timestep 
attenuation}
\item{N}{the population size at the first timestep, including the effects 
of initial population size and asymptotic growth.}
}
}
\description{
Calculate reactivity (first-timestep amplification) and first-timestep 
attenuation for a population matrix projection model.
}
\details{
\code{reac} returns a standardised measure of first-timestep 
amplification or attenuation, discounting the effects of both initial 
population size and asymoptotic growth (Stott et al. 2011).

If \code{vector}="n" then either \code{bound="upper"} or \code{bound="lower"}
must be specified, which calculate the upper or lower bound on first-timestep
amplification and attenuation (i.e. the largest and smallest values that 
reactivity and first-timestep attenuation may take) respectively.
Specifying \code{vector} overrides calculation of a bound, and will yield 
a 'case-specific' reactivity/first-timestep attenuation.

If \code{return.N=T} then the function also returns realised population size 
(including the effects of asymptotic growth and initial population size).

\code{reac} works with imprimitive and irreducible matrices, but 
returns a warning in these cases.

NOTE: \code{reac} replaces \code{reactivity} and \code{firststepatt} as of 
version 1.0-0. Although semantically 'reactivity' and 'first-timestep 
attenuation' are different (the former is an amplification in the first timestep
and the latter an attenuation in the first timestep), as a population matrix 
projection model EITHER amplifies OR attenuates in the first timestep, it made 
no sense to have two separate functions to calculate one thing 
(transient dynamics in the first timestep).
}
\examples{
  # Create a 3x3 PPM
  ( A <- matrix(c(0,1,2,0.5,0.1,0,0,0.6,0.6), byrow=TRUE, ncol=3) )

  # Create initial stage structures
  ( initial1 <- c(1,3,2) )
  ( initial2 <- c(3,1,1) )

  # Calculate the upper bound on reactivity of A
  reac(A, bound="upper")

  # Calculate the lower bound on first-timestep attenuation of A
  reac(A, bound="lower")

  # Calculate case-specific reactivity of A
  # when projected using specific demographic structure
  # that amplifies
  reac(A, vector=initial1)

  # Calculate case-specific reactivity of A
  # and initial1 and return realised population size
  reac(A, vector=initial1, return.N=TRUE)

  # Calculate case-specific first-timestep attenuation of A 
  # when projected using a specific demographic structure that
  #attenuates
  reac(A, vector=initial2)

  # Calculate case-specific first-timestep attenuation of A 
  # and initial2 and return realised population size
  reac(A, vector=initial2, return.N=TRUE)#'

}
\references{
\itemize{
 \item Neubert & Caswell (1997) Ecology, 78, 653-665.
 \item Stott et al. (2011) Ecol. Lett., 14, 959-970.
 \item Townley & Hodgson (2008) J. Appl. Ecol., 45, 1836-1839.
}
}
\seealso{
Other TransientIndices: \code{\link{Kreiss}},
  \code{\link{inertia}}, \code{\link{maxamp}},
  \code{\link{maxatt}}
}
\concept{
transient dynamics amplification unstable instability structure
}
