##' Ebola outbreak, West Africa, 2014-2016
##'
##' Data and models for the 2014--2016 outbreak of Ebola virus disease in West Africa.
##'
##' The data include monthly case counts and death reports derived from WHO situation reports, as reported by the U.S. CDC.
##' The models are described in King et al. (2015).
##'
##' The data-cleaning script is included in the R source code file \file{ebola.R}.
##'
##' @rdname ebola
##' @name ebola
##' @aliases ebolaWA2014
##' @docType data
##' @include pomp.R
##' @family pomp datasets
##' @family pomp examples
##'
##' @references
##'
##' \King2015
##'
##' \WHO2014
##'
##' \He2010
##'
##' @example examples/ebola.R
##'
NULL

##' @rdname ebola
##' @param country ISO symbol for the country (GIN=Guinea, LBR=Liberia, SLE=Sierra Leone).
##' @param data if NULL, the situation report data (WHO Ebola Response Team 2014) for the appropriate country or region will be used.
##' Providing a dataset here will override this behavior.
##' @param nstageE integer; number of incubation stages.
##' @param incubation_period,infectious_period mean duration (in days) of the incubation and infectious periods.
##' @param R0 basic reproduction ratio
##' @param cfr case fatality rate
##' @param rho case reporting efficiency
##' @param k dispersion parameter (negative binomial \code{size} parameter)
##' @param index_case number of cases on day 0 (2014-04-01)
##' @param timestep duration (in days) of Euler timestep for the simulations.
##'
##' @section Model structure:
##' The default incubation period is supposed to be Gamma distributed with shape parameter \code{nstageE} and mean 11.4 days and the case-fatality ratio (`cfr`) is taken to be 0.7 (cf. WHO Ebola Response Team 2014).
##' The discrete-time formula is used to calculate the corresponding \code{alpha} (cf. He et al. 2010).
##'
##' The observation model is a hierarchical model for cases and deaths:
##'   \deqn{p(R_t, D_t| C_t) = p(R_t | C_t) p(D_t | C_t, R_t).}
##' Here, \eqn{p(R_t | C_t)} is negative binomial with mean \eqn{\rho C_t} and dispersion parameter \eqn{1/k};
##' \eqn{p(D_t | C_t, R_t)} is binomial with size \eqn{R_t} and probability equal to the case fatality rate \code{cfr}.
##' @export
ebolaModel <- function (
  country=c("GIN", "LBR", "SLE"),
  data = NULL,
  timestep = 1/8, nstageE = 3L,
  R0 = 1.4, rho = 0.2, cfr = 0.7, k = 0, index_case = 10,
  incubation_period = 11.4, infectious_period = 7
) {

  ## Population sizes in Guinea, Liberia, and Sierra Leone (census 2014)
  populations <- c(GIN=10628972,LBR=4092310,SLE=6190280)

  pomp::ebolaWA2014 -> dat
  dat$date <- as.Date(as.character(dat$date))
  dat$country <- as.character(dat$country)
  dat$day <- julian(dat$date,origin=as.Date("2014-04-01"))

  ctry <- match.arg(country)
  pop <- unname(populations[ctry])
  nstageE <- as.integer(nstageE)
  dt <- as.numeric(timestep)

  globs <- paste0("static int nstageE = ",nstageE,";");

  theta <- c(
    N=pop,R0=1.4,
    rho=rho,cfr=cfr,k=k,
    alpha=-1/(nstageE*dt)*log(1-nstageE*dt/incubation_period),
    gamma=-log(1-dt/infectious_period)/dt,
    S_0=1-index_case/pop,
    E_0=index_case/pop/2-5e-9,
    I_0=index_case/pop/2-5e-9,
    R_0=1e-8
  )

  if (is.null(data)) {
    dat <- subset(dat,country==ctry,select=c("day","cases","deaths"))
  } else {
    dat <- data
  }

  dObs <- Csnippet("
    double f, g;
    if (k > 0.0) {
      f = dnbinom_mu(nearbyint(cases),1.0/k,rho*N_EI,give_log);
      g = dnbinom_mu(nearbyint(deaths),1.0/k,rho*cfr*N_IR,give_log);
    } else {
      f = dpois(nearbyint(cases),rho*N_EI,give_log);
      g = dpois(nearbyint(deaths),rho*cfr*N_IR,give_log);
    }
    lik = (give_log) ? f+g : f*g;
    "
  )

  rObs <- Csnippet("
    if (k > 0) {
      cases = rnbinom_mu(1.0/k,rho*N_EI);
      deaths = rnbinom_mu(1.0/k,rho*cfr*N_IR);
    } else {
      cases = rpois(rho*N_EI);
      deaths = rpois(rho*cfr*N_IR);
    }
    "
  )

  rInit <- Csnippet("
    double m = N/(S_0+E_0+I_0+R_0);
    double *E = &E1;
    int j;
    S = nearbyint(m*S_0);
    for (j = 0; j < nstageE; j++) E[j] = nearbyint(m*E_0/nstageE);
    I = nearbyint(m*I_0);
    R = nearbyint(m*R_0);
    N_EI = 0;
    N_IR = 0;
    "
  )

  rSim <- Csnippet("
    double lambda, beta;
    double *E = &E1;
    beta = R0 * gamma; // Transmission rate
    lambda = beta * I / N; // Force of infection
    int i;

    // Transitions
    // From class S
    double transS = rbinom(S, 1.0 - exp(- lambda * dt)); // No of infections
    // From class E
    double transE[nstageE]; // No of transitions between classes E
    for(i = 0; i < nstageE; i++){
      transE[i] = rbinom(E[i], 1.0 - exp(- nstageE * alpha * dt));
    }
    // From class I
    double transI = rbinom(I, 1.0 - exp(- gamma * dt)); // No of transitions I->R

    // Balance the equations
    S -= transS;
    E[0] += transS - transE[0];
    for(i=1; i < nstageE; i++) {
      E[i] += transE[i-1] - transE[i];
    }
    I += transE[nstageE - 1] - transI;
    R += transI;
    N_EI += transE[nstageE - 1]; // No of transitions from E to I
    N_IR += transI; // No of transitions from I to R
    "
  )

  skel <- Csnippet("
    double lambda, beta;
    const double *E = &E1;
    double *DE = &DE1;
    beta = R0 * gamma; // Transmission rate
    lambda = beta * I / N; // Force of infection
    int i;

    // Balance the equations
    DS = - lambda * S;
    DE[0] = lambda * S - nstageE * alpha * E[0];
    for (i=1; i < nstageE; i++)
      DE[i] = nstageE * alpha * (E[i-1]-E[i]);
    DI = nstageE * alpha * E[nstageE-1] - gamma * I;
    DR = gamma * I;
    DN_EI = nstageE * alpha * E[nstageE-1];
    DN_IR = gamma * I;
    "
  )

  pomp(
    data=dat,
    times="day",
    t0=0,
    params=theta,
    globals=globs,
    rinit=rInit,
    rprocess=discrete_time(rSim,delta.t=timestep),
    dmeasure=dObs,
    rmeasure=rObs,
    skeleton=vectorfield(skel),
    partrans=parameter_trans(
      log=c("N","R0","alpha","gamma","k"),
      logit=c("rho","cfr"),
      barycentric=c("S_0","E_0","I_0","R_0")
    ),
    statenames=c("S",sprintf("E%d",1:nstageE),"I","R","N_EI","N_IR"),
    paramnames=c("N","R0","alpha","gamma","rho","k","cfr",
      "S_0","E_0","I_0","R_0"),
    accumvars=c("N_EI","N_IR")
  )

}

## This codes below processes case count and death data from the 2014-2016 Ebola
## outbreak in West Africa, as provided by the WHO via US CDC.
## In particular, the following are purportedly the
## "Number of Cases and Deaths in Guinea, Liberia, and Sierra Leone
## during the 2014-2016 West Africa Ebola Outbreak."
## These data were accessed from
## https://www.cdc.gov/vhf/ebola/history/2014-2016-outbreak/case-counts.html
## on 2019-03-09 and are included below.
##
## Variables are:
## - WHO report date
## - Total Cases, Guinea
## - Total Deaths, Guinea
## - Total Cases, Liberia
## - Total Deaths, Liberia
## - Total Cases, Sierra Leone
## - Total Deaths, Sierra Leone
## - Total Cases
## - Total Deaths
##
## In this script, these data are processed to produce monthly incidence
## and deaths for each of the three countries.
##

## BEGIN DATA-CLEANING CODE
## "date,cases_GIN,deaths_GIN,cases_LBR,deaths_LBR,cases_SLE,deaths_SLE,cases_total,deaths_total
## 4/13/2016,3814,2544,10678,4810,14124,3956,28616,11310
## 3/30/2016,3811,2543,10675,4809,14124,3956,28610,11308
## 3/23/2016,3809,2540,10675,4809,14124,3956,28608,11305
## 3/3/2016,3804,2536,10675,4809,14124,3956,28603,11301
## 2/17/2016,3804,2536,10675,4809,14124,3956,28603,11301
## 2/10/2016,3804,2536,10675,4809,14124,3956,28603,11301
## 2/3/2016,3804,2536,10675,4809,14124,3956,28603,11301
## 1/27/2016,3804,2536,10675,4809,14124,3956,28603,11301
## 1/20/2016,3804,2536,10675,4809,14123,3956,28602,11301
## 1/13/2016,3804,2536,10675,4809,14122,3955,28601,11300
## 1/6/2016,3804,2536,10675,4809,14122,3955,28601,11300
## 12/30/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/29/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/28/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/25/2015,3805,2536,10675,4809,14122,3955,28602,11300
## 12/24/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/23/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/22/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/21/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/18/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/17/2015,3805,2536,10675,4809,14122,3955,28602,11300
## 12/16/2015,3807,2536,10675,4809,14122,3955,28604,11300
## 12/15/2015,3807,2536,10675,4809,14122,3955,28604,11300
## 12/14/2015,3806,2536,10675,4809,14122,3955,28603,11300
## 12/11/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/10/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/9/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/8/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/7/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/4/2015,3805,2536,10675,4809,14122,3955,28602,11300
## 12/3/2015,3805,2536,10675,4809,14122,3955,28602,11300
## 12/2/2015,3804,2536,10675,4809,14122,3955,28601,11300
## 12/1/2015,3804,2536,10675,4808,14122,3955,28601,11299
## 11/30/2015,3804,2536,10675,4808,14122,3955,28601,11299
## 11/27/2015,3804,2536,10675,4808,14122,3955,28601,11299
## 11/26/2015,3804,2536,10675,4808,14122,3955,28601,11299
## 11/25/2015,3804,2536,10675,4808,14122,3955,28601,11299
## 11/24/2015,3804,2536,10672,4808,14122,3955,28598,11299
## 11/23/2015,3804,2536,10672,4808,14122,3955,28598,11299
## 11/20/2015,3804,2536,10672,4808,14122,3955,28598,11299
## 11/19/2015,3806,2536,10672,4808,14122,3955,28600,11299
## 11/18/2015,3804,2536,10672,4808,14122,3955,28598,11299
## 11/17/2015,3404,2536,10672,4808,14122,3955,28598,11299
## 11/16/2015,3805,2536,10672,4808,14122,3955,28599,11299
## 11/12/2015,3805,2536,10672,4808,14122,3955,28599,11299
## 11/10/2015,3805,2536,10672,4808,14122,3955,28599,11299
## 11/9/2015,3806,2536,10672,4808,14122,3955,28600,11299
## 11/6/2015,3805,2536,10672,4808,14109,3955,28586,11299
## 11/5/2015,3805,2536,10672,4808,14104,3955,28581,11299
## 11/3/2015,3810,2536,10672,4808,14089,3955,28571,11299
## 11/2/2015,3808,2536,10672,4808,14078,3955,28558,11299
## 10/30/2015,3808,2536,10672,4808,14067,3955,28547,11299
## 10/29/2015,3808,2536,10672,4808,14066,3955,28546,11299
## 10/27/2015,3806,2535,10672,4808,14061,3955,28539,11298
## 10/26/2015,3804,2535,10672,4808,14052,3955,28528,11298
## 10/23/2015,3801,2535,10672,4808,14038,3955,28511,11298
## 10/22/2015,3801,2535,10672,4808,14031,3955,28504,11298
## 10/20/2015,3803,2535,10672,4808,14001,3955,28476,11298
## 10/19/2015,3806,2535,10672,4808,13999,3955,28513,11298
## 10/16/2015,3804,2535,10672,4808,13992,3955,28468,11298
## 10/15/2015,3802,2534,10672,4808,13992,3955,28466,11297
## 10/13/2015,3800,2534,10672,4808,13982,3955,28454,11297
## 10/12/2015,3798,2534,10672,4808,13978,3955,28448,11297
## 10/9/2015,3800,2534,10672,4808,13957,3955,28429,11297
## 10/8/2015,3799,2534,10672,4808,13956,3955,28427,11297
## 10/7/2015,3804,2534,10672,4808,13945,3955,28421,11297
## 10/5/2015,3804,2534,10672,4808,13941,3955,28417,11297
## 10/2/2015,3809,2533,10672,4808,13931,3955,28412,11296
## 10/1/2015,3808,2533,10672,4808,13928,3955,28408,11296
## 9/29/2015,3805,2533,10672,4808,13911,3955,28388,11296
## 9/28/2015,3805,2533,10672,4808,13894,3955,28371,11296
## 9/24/2015,3801,2533,10672,4808,13846,3955,28319,11296
## 9/22/2015,3800,2532,10672,4808,13823,3955,28295,11295
## 9/21/2015,3797,2532,10672,4808,13811,3955,28280,11295
## 9/18/2015,3794,2531,10672,4808,13785,3955,28251,11294
## 9/17/2015,3794,2530,10672,4808,13573,3955,28245,11293
## 9/15/2015,3792,2530,10672,4808,13756,3953,28220,11291
## 9/14/2015,3793,2530,10672,4808,13747,3953,28212,11291
## 9/11/2015,3791,2530,10672,4808,13701,3953,28164,11291
## 9/10/2015,3791,2530,10672,4808,13697,3953,28160,11291
## 9/8/2015,3792,2530,10672,4808,13683,3953,28147,11291
## 9/4/2015,3792,2529,10672,4808,13639,3953,28103,11290
## 9/3/2015,3792,2529,10672,4808,13638,3953,28102,11290
## 9/1/2015,3792,2529,10672,4808,13609,3953,28073,11290
## 8/31/2015,3790,2528,10672,4808,13603,3953,28065,11289
## 8/28/2015,3797,2528,10672,4808,13586,3952,28055,11288
## 8/27/2015,3797,2528,10672,4808,13582,3952,28051,11288
## 8/25/2015,3792,2527,10672,4808,13541,3952,28005,11287
## 8/24/2015,3790,2527,10672,4808,13538,3952,28000,11287
## 8/21/2015,3795,2526,10672,4808,13533,3952,28000,11286
## 8/20/2015,3792,2526,10672,4808,13518,3952,27982,11286
## 8/18/2015,3786,2524,10672,4808,13494,3952,27952,11284
## 8/17/2015,3790,2524,10672,4808,13489,3952,27951,11284
## 8/14/2015,3791,2524,10672,4808,13485,3952,27948,11284
## 8/13/2015,3792,2524,10672,4808,13484,3951,27948,11283
## 8/11/2015,3787,2524,10672,4808,13470,3951,27929,11283
## 8/10/2015,3791,2524,10672,4808,13465,3951,27928,11283
## 8/7/2015,3795,2524,10672,4808,13428,3951,27895,11283
## 8/6/2015,3792,2524,10672,4808,13426,3951,27890,11283
## 8/5/2015,3784,2522,10672,4808,13406,3951,27862,11281
## 8/4/2015,3784,2522,10672,4808,13406,3951,27862,11281
## 8/3/2015,3786,2522,10672,4808,13402,3951,27860,11281
## 7/31/2015,3781,2521,10672,4808,13387,3951,27840,11280
## 7/30/2015,3785,2521,10672,4808,13379,3951,27836,11280
## 7/29/2015,3786,2520,10672,4808,13290,3951,27748,11279
## 7/28/2015,3786,2520,10673,4808,13291,3951,27750,11279
## 7/27/2015,3792,2519,10673,4808,13284,3951,27749,11278
## 7/24/2015,3787,2517,10673,4808,13264,3949,27724,11274
## 7/23/2015,3788,2516,10673,4808,13262,3949,27723,11273
## 7/22/2015,3783,2512,10672,4808,13250,3949,27705,11269
## 7/21/2015,3783,2512,10673,4808,13250,3949,27706,11269
## 7/20/2015,3784,2511,10673,4808,13241,3949,27698,11268
## 7/17/2015,3770,2509,10673,4808,13209,3947,27652,11264
## 7/16/2015,3762,2506,10672,4808,13209,3947,27643,11261
## 7/15/2015,3760,2506,10673,4808,13209,3947,27642,11261
## 7/14/2015,3760,2506,10673,4808,13209,3947,27642,11261
## 7/13/2015,3754,2506,10672,4807,13201,3946,27627,11259
## 7/10/2015,3751,2505,10672,4807,13177,3941,27600,11253
## 7/9/2015,3744,2505,10672,4807,13169,3941,27585,11253
## 7/8/2015,3748,2499,10670,4807,13155,3940,27573,11246
## 7/5/2015,3748,2499,10670,4807,13155,3940,27573,11246
## 7/4/2015,3744,2492,10670,4807,13150,3940,27564,11245
## 7/3/2015,3750,2492,10706,4811,13135,3935,27591,11238
## 6/30/2015,3745,2490,10666,4806,13129,3933,27540,11229
## 6/30/2015,3729,2482,10666,4806,13119,3932,27514,11220
## 6/29/2015,3724,2482,10666,4806,13115,3932,27505,11220
## 6/24/2015,3720,2480,10666,4806,13093,3931,24479,11217
## 6/23/2015,3718,2480,10666,4806,13083,3931,27467,11217
## 6/21/2015,3718,2473,10666,4806,13059,3928,27443,11207
## 6/17/2015,3674,2446,10666,4806,13012,3926,27352,11178
## 6/16/2015,3675,2445,10666,4806,12990,3922,27331,11173
## 6/14/2015,3678,2444,10666,4806,12965,3919,27309,11169
## 6/13/2015,3677,2443,10666,4806,12962,3919,27305,11168
## 6/10/2015,3677,2440,10666,4806,12932,3917,27275,11163
## 6/9/2015,3674,2440,10666,4806,12911,3917,27251,11163
## 6/7/2015,3670,2437,10666,4806,12901,3915,27237,11158
## 6/6/2015,3669,2435,10666,4806,12884,3913,27219,11154
## 6/5/2015,3664,2431,10666,4806,12859,3912,27189,11149
## 6/4/2015,3657,2431,10666,4806,12850,3912,27173,11149
## 5/31/2015,3652,2429,10666,4806,12827,3912,27145,11147
## 5/30/2015,3653,2428,10666,4806,12816,3911,27135,11145
## 5/27/2015,3644,2425,10666,4806,12745,3911,27055,11142
## 5/26/2015,3639,2423,10666,4806,12735,3911,27040,11140
## 5/24/2015,3641,2420,10666,4806,12706,3908,27013,11134
## 5/20/2015,3647,2412,10666,4806,12666,3907,26979,11125
## 5/19/2015,3649,2409,10666,4806,12656,3907,26971,11122
## 5/16/2015,3626,2405,10666,4806,12593,3906,26885,11117
## 5/13/2015,3619,2401,10604,4769,12540,3904,26763,11074
## 5/12/2015,3615,2399,10604,4769,12536,3904,26755,11072
## 5/10/2015,3597,2392,10604,4769,12523,3904,26724,11065
## 5/9/2015,3599,2391,10604,4769,12519,3904,26722,11064
## 5/6/2015,3592,2387,10564,4716,12492,3904,26648,11007
## 5/5/2015,3592,2387,10564,4716,12470,3904,26626,11007
## 5/3/2015,3589,2386,10564,4716,12440,3903,26593,11005
## 5/3/2015,3589,2386,10507,4691,12440,3903,26536,10980
## 5/2/2015,3591,2385,10322,4608,12426,3902,26339,10895
## 4/29/2015,3578,2383,10322,4608,12398,3901,26298,10892
## 4/28/2015,3581,2381,10322,4608,12387,3901,26290,10890
## 4/26/2015,3584,2377,10322,4608,12371,3899,26277,10884
## 4/24/2015,3585,2374,10212,4573,12362,3895,26159,10842
## 4/22/2015,3576,2365,10212,4573,12313,3886,26101,10824
## 4/21/2015,3568,2362,10212,4573,12294,3885,26074,10820
## 4/19/2015,3565,2358,10212,4573,12267,3877,26044,10808
## 4/19/2015,3565,2358,10042,4486,12256,3877,25872,10721
## 4/18/2015,3565,2357,10042,4486,12256,3872,25863,10715
## 4/17/2015,3569,2351,10042,4486,12244,3865,25855,10702
## 4/16/2015,3566,2348,10042,4486,12223,3865,25831,10699
## 4/15/2015,3548,2346,10042,4486,12201,3857,25791,10689
## 4/14/2015,3548,2346,9862,4408,12201,3857,25611,10611
## 4/13/2015,3541,2342,9862,4408,12188,3854,25591,10604
## 4/10/2015,3524,2337,9862,4408,12170,3842,25556,10587
## 4/9/2015,3515,2335,9862,4408,12155,3841,25532,10584
## 4/8/2015,3515,2333,9862,4408,12138,3831,25515,10572
## 4/7/2015,3515,2333,9862,4408,12139,3832,25516,10573
## 4/2/2015,3494,2320,9712,4332,12022,3810,25228,10462
## 4/1/2015,3492,2314,9712,4332,11974,3799,25178,10445
## 3/31/2015,3492,2314,9712,4332,11974,3799,25178,10445
## 3/30/2015,3485,2305,9602,4301,11943,3792,25030,10398
## 3/27/2015,3466,2276,9602,4301,11889,3773,24957,10350
## 3/26/2015,3459,2273,9602,4301,11866,3764,24927,10338
## 3/25/2015,3429,2263,9602,4301,11841,3747,24872,10311
## 3/18/2015,3389,2224,9526,4264,11751,3691,24666,10179
## 3/11/2015,3285,2170,9343,4162,11619,3629,24247,9961
## 3/4/2015,3219,2129,9249,4117,11466,3546,23934,9792
## 2/25/2015,3155,2091,9238,4037,11301,3461,23694,9589
## 2/18/2015,3108,2057,9007,3900,11103,3408,23218,9365
## 2/11/2015,3044,1995,8881,3826,10934,3341,22859,9162
## 2/4/2015,2975,1944,8745,3746,10740,3276,22460,8966
## 1/28/2015,2917,1910,8622,3686,10518,3199,22057,8795
## 1/21/2015,2871,1876,8478,3605,10340,3145,21689,8626
## 1/14/2015,2806,1814,8331,3538,10124,3062,21261,8414
## 1/7/2015,2775,1781,8157,3496,9780,2943,20712,8220
## 12/31/2014,2707,1708,8018,3423,9446,2758,20171,7889
## 12/24/2014,2597,1607,7862,3384,9004,2582,19463,7573
## 12/17/2014,2416,1525,7797,3290,8356,2085,18569,6900
## 12/10/2014,2292,1428,7719,3177,7897,1768,17908,6373
## 12/3/2014,2164,1327,7635,3145,7312,1583,17111,6055
## 11/28/2014,2155,1312,7635,3145,7109,1530,16899,5987
## 11/26/2014,2134,1260,7168,3016,6599,1398,15901,5674
## 11/21/2014,2047,1214,7082,2963,6190,1267,15319,5444
## 11/19/2014,1971,1192,7069,2964,6073,1250,15113,5406
## 11/14/2014,1919,1166,6878,2812,5586,1187,14383,5165
## 11/12/2014,1878,1142,6822,2836,5368,1169,14068,5147
## 11/7/2014,1760,1054,6619,2766,4862,1130,13241,4950
## 11/5/2014,1731,1041,6525,2697,4759,1070,13015,4808
## 10/31/2014,1667,1018,6535,2413,5338,1510,13540,4941
## 10/29/2014,1906,997,6535,2413,5235,1500,13676,4910
## 10/25/2014,1553,926,4665,2705,3896,1281,10114,4912
## 10/22/2014,1540,904,4665,2705,3706,1359,9911,4968
## 10/17/2014,1519,862,4262,2484,3410,1200,9191,4546
## 10/15/2014,1472,843,4249,2458,3252,1183,8973,4484
## 10/10/2014,1350,778,4076,2316,2950,930,8376,4024
## 10/8/2014,1298,768,3924,2210,2789,879,8011,3857
## 10/3/2014,1199,739,3834,2069,2437,623,7470,3431
## 10/1/2014,1157,710,3696,1998,2304,622,7157,3330
## 9/26/2014,1074,648,3458,1830,2021,605,6553,3083
## 9/24/2014,1022,635,3280,1677,1940,597,6242,2909
## 9/22/2014,1008,632,3022,1578,1813,593,5843,2803
## 9/18/2014,942,601,2710,1459,1673,562,5325,2622
## 9/16/2014,936,595,2407,1296,1620,562,4963,2453
## 9/12/2014,861,557,2081,1137,1424,524,4366,2218
## 9/8/2014,862,555,2046,1224,1361,509,4269,2288
## 9/6/2014,812,517,1871,1089,1261,491,3944,2097
## 8/28/2014,648,430,1378,694,1026,422,3052,1546
## 8/22/2014,607,406,1082,624,910,392,2599,1422
## 8/21/2014,579,396,972,576,907,374,2458,1346
## 8/19/2014,543,394,834,466,848,365,2225,1225
## 8/15/2014,519,380,786,348,810,348,2115,1076
## 8/13/2014,510,377,670,355,783,334,1963,1066
## 8/12/2014,506,373,599,323,730,315,1835,1011
## 8/8/2014,495,367,554,294,717,298,1766,959
## 8/4/2014,485,358,486,255,646,273,1617,886
## 8/3/2014,472,346,391,227,574,252,1437,825
## 7/31/2014,460,339,329,156,533,233,1322,728
## 7/28/2014,427,319,249,129,525,224,1201,672
## 7/24/2014,415,314,224,127,454,219,1093,660
## 7/21/2014,410,310,196,116,442,206,1048,632
## 7/16/2014,406,304,172,105,386,192,964,601
## 7/14/2014,409,309,142,88,337,142,888,539
## 7/8/2014,408,307,131,84,305,127,844,518
## 7/7/2014,412,305,115,75,252,101,779,481
## 7/2/2014,413,303,107,65,239,99,759,467
## 6/24/2014,390,270,51,34,158,34,599,338
## 6/18/2014,398,264,33,24,97,49,528,337
## 6/11/2014,376,241,15,10,117,19,508,270
## 6/10/2014,372,236,15,10,89,7,476,253
## 6/5/2014,344,215,13,9,81,7,438,231
## 6/2/2014,291,193,13,9,50,6,354,208
## 5/28/2014,281,186,12,9,16,5,309,200
## 5/27/2014,258,174,12,9,1,4,271,187
## 5/23/2014,258,174,12,9,0,0,270,183
## 5/14/2014,233,157,12,11,0,0,245,168
## 5/5/2014,231,155,13,11,0,0,244,166
## 4/30/2014,221,146,13,11,0,0,234,157
## 4/23/2014,208,136,34,11,0,0,242,147
## 4/21/2014,203,129,27,13,0,0,230,142
## 4/17/2014,197,122,27,13,0,0,224,135
## 4/10/2014,157,101,22,14,0,0,179,115
## 4/7/2014,151,95,18,7,0,0,169,102
## 4/2/2014,127,83,8,5,0,0,135,88
## 4/1/2014,122,80,8,2,0,0,130,82
## 3/31/2014,112,70,8,6,0,0,120,76
## 3/27/2014,103,66,8,6,6,5,117,77
## 3/26/2014,86,60,0,0,0,0,86,60
## 3/25/2014,86,59,0,0,0,0,86,59
## " -> dat_text

## library(tidyverse)
## library(lubridate)

## read_csv(dat_text,comment="#") |>
##   mutate(date=as.Date(date,format="%m/%d/%Y")) |>
##   gather(variable,value,-date) |>
##   separate(variable,into=c("vbl","country")) |>
##   filter(country != "total") |>
##   group_by(vbl,country) |>
##   arrange(date) |>
##   mutate(value=value-lag(value)) |>
##   filter(!is.na(value)) |>
##   ungroup() |>
##   group_by(date,country,vbl) |>
##   summarize(value=sum(value)) |>
##   ungroup() |>
##   mutate(date=ceiling_date(date,"month")-ddays(1)) |>
##   group_by(date,vbl,country) |>
##   summarize(value=sum(value)) |>
##   group_by(date,country) |>
##   spread(vbl,value) |>
##   ungroup() |>
##   filter(date>"2014-04-01") |>
##   mutate(
##     cases=pmax(cases,0),
##     deaths=pmax(deaths,0)
##   ) |>
##   as.data.frame() -> ebolaWA2014

## save(ebolaWA2014,file="ebolaWA2014.rda",compress="xz")
## END DATA-CLEANING CODE
