#
#                _         _    _      _                _    
#               (_)       | |  | |    | |              | |   
#  _ __    ___   _  _ __  | |_ | |__  | |  __ _  _ __  | | __
# | '_ \  / _ \ | || '_ \ | __|| '_ \ | | / _` || '_ \ | |/ /
# | |_) || (_) || || | | || |_ | |_) || || (_| || | | ||   < 
# | .__/  \___/ |_||_| |_| \__||_.__/ |_| \__,_||_| |_||_|\_\
# | |                                                        
# |_|                                                        
# 
# This file is part of the 'rich-iannone/pointblank' package.
# 
# (c) Richard Iannone <riannone@me.com>
# 
# For full copyright and license information, please look at
# https://rich-iannone.github.io/pointblank/LICENSE.html
#


#' Create a **pointblank** *multiagent* object
#'
#' Multiple *agents* can be part of a single object called the *multiagent*.
#' This can be useful when gathering multiple agents that have performed
#' interrogations in the past (perhaps saved to disk with [x_write_disk()]).
#' When be part of a *multiagent*, we can get a report that shows how data
#' quality evolved over time. This can be of interest when it's important to
#' monitor data quality and even the evolution of the validation plan itself.
#' The reporting table, generated by printing a `ptblank_multiagent` object or
#' by using the [get_multiagent_report()] function, is, by default, organized by
#' the interrogation time and it automatically recognizes which validation steps
#' are equivalent across interrogations.
#'
#' @param ... One or more **pointblank** agent objects.
#' @param lang The language to use for any reporting that will be generated from
#'   the *multiagent*. (e.g., individual *agent reports*, *multiagent reports*,
#'   etc.). By default, `NULL` will create English (`"en"`) text. Other options
#'   include French (`"fr"`), German (`"de"`), Italian (`"it"`), Spanish
#'   (`"es"`), Portuguese, (`"pt"`), Chinese (`"zh"`), and Russian (`"ru"`).
#' @param locale An optional locale ID to use for formatting values in the
#'   reporting outputs according the locale's rules. Examples include `"en_US"`
#'   for English (United States) and `"fr_FR"` for French (France); more simply,
#'   this can be a language identifier without a country designation, like "es"
#'   for Spanish (Spain, same as `"es_ES"`).
#'   
#' @return A `ptblank_multiagent` object.
#' 
#' @examples 
#' if (interactive()) {
#' 
#' # Let's walk through several theoretical
#' # data quality analyses of an extremely
#' # small table; that table is called
#' # `small_table` and we can find it as a
#' # dataset in this package
#' small_table
#' 
#' # To set failure limits and signal
#' # conditions, we designate proportional
#' # failure thresholds to the `warn`, `stop`,
#' # and `notify` states using `action_levels()`
#' al <- 
#'   action_levels(
#'     warn_at = 0.05,
#'     stop_at = 0.10,
#'     notify_at = 0.20
#'   )
#' 
#' # We will create four different agents
#' # and have slightly different validation
#' # steps in each of them; in the first,
#' # `agent_1`, eight different validation
#' # steps are created and the agent will
#' # interrogate the `small_table`
#' agent_1 <-
#'   create_agent(
#'     read_fn = ~ small_table,
#'     label = "An example.",
#'     actions = al
#'   ) %>%
#'   col_vals_gt(
#'     vars(date_time),
#'     value = vars(date),
#'     na_pass = TRUE
#'   ) %>%
#'   col_vals_gt(
#'     vars(b), 
#'     value = vars(g),
#'     na_pass = TRUE
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_equal(
#'     vars(d), 
#'     value = vars(d),
#'     na_pass = TRUE
#'   ) %>%
#'   col_vals_between(
#'     vars(c), 
#'     left = vars(a), right = vars(d)
#'   ) %>%
#'   col_vals_not_between(
#'     vars(c),
#'     left = 10, right = 20,
#'     na_pass = TRUE
#'   ) %>%
#'   rows_distinct(vars(d, e, f)) %>%
#'   col_is_integer(vars(a)) %>%
#'   interrogate()
#' 
#' # The second agent, `agent_2`, retains
#' # all of the steps of `agent_1` and adds
#' # two more (the last of which is inactive)
#' agent_2 <- 
#'   agent_1 %>%
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), 
#'     regex = "[0-9]-[a-z]{3}-[0-9]{3}",
#'     active = FALSE
#'   ) %>%
#'   interrogate()
#' 
#' # The third agent, `agent_3`, adds a single
#' # validation step, removes the fifth one,
#' # and deactivates the first
#' agent_3 <- 
#'   agent_2 %>%
#'   col_vals_in_set(
#'     vars(f),
#'     set = c("low", "mid", "high")
#'   ) %>%
#'   remove_steps(i = 5) %>%
#'   deactivate_steps(i = 1) %>%
#'   interrogate()
#' 
#' # The fourth and final agent, `agent_4`,
#' # reactivates steps 1 and 10, and removes
#' # the sixth step
#' agent_4 <-
#'   agent_3 %>%
#'   activate_steps(i = 1) %>%
#'   activate_steps(i = 10) %>%
#'   remove_steps(i = 6) %>%
#'   interrogate()
#' 
#' # While all the agents are slightly
#' # different from each other, we can still
#' # get a combined report of them by
#' # creating a 'multiagent'
#' multiagent <-
#'   create_multiagent(
#'     agent_1, agent_2, agent_3, agent_4
#'   )
#' 
#' # Calling `multiagent` in the console
#' # prints the multiagent report; but we
#' # can get a `gt_tbl` object with the
#' # `get_multiagent_report(agent)` function
#' report <- get_multiagent_report(multiagent)
#' 
#' class(report)
#' 
#' }
#' 
#' @section Figures:
#' \if{html}{\figure{man_create_multiagent_1.png}{options: width=100\%}}
#' 
#' @family The multiagent
#' @section Function ID:
#' 10-1
#'
#' @export
create_multiagent <- function(...,
                              lang = NULL,
                              locale = NULL) {
  
  agent_list <- list(...)
    
  agent_list <- 
    lapply(
      agent_list,
      FUN = function(agent) {
        
        # TODO: Ensure that each `agent` in `agent_list` is
        # actually an agent with `is_ptblank_agent()`
        
        class(agent) <-
          c(setdiff(class(agent), "ptblank_agent"), "ptblank_agent_i")
        
        agent
      }
    )
  
  agent_series <-
    list(
      overview_tbl = list(),
      agents = agent_list
    )
  
  class(agent_series) <- "ptblank_multiagent"
  agent_series
}
