% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stateEstimation.R
\name{particleFilterSVmodel}
\alias{particleFilterSVmodel}
\title{Bootstrap particle filter for state estimate in a simple stochastic 
volatility model}
\usage{
particleFilterSVmodel(y, theta, noParticles)
}
\arguments{
\item{y}{Observations from the model for \eqn{t=1,...,T}.}

\item{theta}{The parameters \eqn{\theta=\{\mu,\phi,\sigma_v\}}. 
The mean of the log-volatility process is denoted \eqn{\mu}. 
The persistence of the log-volatility process is denoted \eqn{\phi}. 
The standard deviation of the log-volatility process is 
denoted \eqn{\sigma_v}.}

\item{noParticles}{The number of particles to use in the filter.}
}
\value{
The function returns a list with the elements:
\itemize{
\item{xHatFiltered: The estimate of the filtered state at time \eqn{t=1,...,T}.}
\item{logLikelihood: The estimate of the log-likelihood.}
}
}
\description{
Estimates the filtered state and the log-likelihood for a stochastic 
volatility model of the form \eqn{x_t = \mu + \phi ( x_{t-1} - \mu ) + 
\sigma_v v_t} and \eqn{y_t = \exp(x_t/2) e_t}, where \eqn{v_t} and \eqn{e_t} 
denote independent standard Gaussian random variables, i.e. \eqn{N(0,1)}.
}
\note{
See Section 5 in the reference for more details.
}
\examples{
\dontshow{
  # Generate fake data
  y <- rnorm(100)

  # Estimate the filtered state using a particle filter
  theta <- c(-0.10, 0.97, 0.15)
  pfOutput <- particleFilterSVmodel(y, theta, noParticles=100)

  # Plot the estimate and the true state
  par(mfrow=c(2, 1))
  plot(y, type="l", xlab="time", ylab="log-returns", bty="n",
    col="#1B9E77")
  plot(pfOutput$xHatFiltered, type="l", xlab="time",
    ylab="estimate of log-volatility", bty="n", col="#D95F02")
}
\dontrun{
  # Get the data from Quandl
  library("Quandl")
  d <- Quandl("NASDAQOMX/OMXS30", start_date="2012-01-02",
              end_date="2014-01-02", type="zoo")
  y <- as.numeric(100 * diff(log(d$"Index Value")))

  # Estimate the filtered state using a particle filter
  theta <- c(-0.10, 0.97, 0.15)
  pfOutput <- particleFilterSVmodel(y, theta, noParticles=100)

  # Plot the estimate and the true state
  par(mfrow=c(2, 1))
  plot(y, type="l", xlab="time", ylab="log-returns", bty="n",
    col="#1B9E77")
  plot(pfOutput$xHatFiltered, type="l", xlab="time",
    ylab="estimate of log-volatility", bty="n", col="#D95F02")
}
}
\references{
Dahlin, J. & Schon, T. B. "Getting started with particle 
Metropolis-Hastings for inference in nonlinear dynamical models." 
pre-print, arXiv:1511.01707, 2017.
}
\author{
Johan Dahlin <uni (at) johandahlin.com.nospam>
}
\keyword{ts}
