new_index <- function(index, contrib, levels, time, chainable) {
  res <- list(index = as.list(index),
              contrib = as.list(contrib),
              levels = as.character(levels),
              time = as.character(time))
  type <- if (chainable) "chainable_index" else "direct_index"
  structure(res, class = c(type, "index"))
}

validate_levels <- function(x) {
  if (length(x) == 0L) {
    stop("cannot make an index with no levels")
  }
  if (anyNA(x) || any(x == "")) {
    stop("cannot make an index with missing levels")
  }
  if (anyDuplicated(x)) {
    stop("cannot make an index with duplicate levels")
  }
  invisible(x)
}

validate_time <- function(x) {
  if (length(x) == 0L) {
    stop("cannot make an index with no time periods")
  }
  if (anyNA(x) || any(x == "")) {
    stop("cannot make an index with missing time periods")
  }
  if (anyDuplicated(x)) {
    stop("cannot make an index with duplicate time periods")
  }
  invisible(x)
}

validate_index_values <- function(x) {
  if (!identical(x$time, names(x$index))) {
    stop("missing index values for each time period")
  }
  if (any(vapply(x$index, \(z) !identical(x$levels, names(z)), logical(1L)))) {
    stop("missing index values for each level")
  }
  invisible(x)
}

validate_contrib <- function(x) {
  if (!identical(x$time, names(x$contrib))) {
    stop("missing contributions for each time period")
  }
  if (any(vapply(x$contrib,
                 \(z) !identical(x$levels, names(z)), logical(1L)))) {
    stop("missing contributions for each level")
  }
  invisible(x)
}

validate_index <- function(x) {
  validate_levels(x$levels)
  validate_time(x$time)
  validate_index_values(x$index)
  validate_contrib(x$contrib)
  x
}

#---- Calculate generalized-mean elemental indexes ----
elemental_index <- function(x, ...) {
  UseMethod("elemental_index")
}

elemental_index.default <- function(x, ...) {
  elemental_index(as.numeric(x), ...)
}

elemental_index.numeric <- function(x,
                                    period = gl(1, length(x)),
                                    ea = gl(1, length(x)),
                                    w = NULL,
                                    contrib = FALSE,
                                    chainable = TRUE,
                                    na.rm = FALSE,
                                    r = 0,
                                    ...) {
  if (different_length(x, period, ea, w)) {
    stop("input vectors must be the same length")
  }
  if (any(x <= 0, na.rm = TRUE) || any(w <= 0, na.rm = TRUE)) {
    warning("some elements of 'x or 'w' are less than or equal to 0")
  }

  period <- as.factor(period)
  ea <- as.factor(ea) # ensures elemental aggregates are balanced
  time <- levels(period)
  levels <- levels(ea)

  if (contrib) {
    if (is.null(names(x))) {
      names(x) <- sequential_names(ea, period)
    } else {
      names(x) <- valid_product_names(names(x), period, ea)
    }
  }
  # splitting 'x' into a nested list by period then ea is the same as
  # using interaction(), but makes it easier to get the results as
  # a list
  ea <- split(ea, period)
  x <- Map(split, split(x, period), ea)
  if (is.null(w)) {
    w <- list(list(NULL))
  } else {
    w <- Map(split, split(as.numeric(w), period), ea)
  }

  index_fun <- Vectorize(generalized_mean(r))
  contrib_fun <- Vectorize(contributions(r), SIMPLIFY = FALSE)

  index <- Map(index_fun, x, w, na.rm = na.rm)
  if (contrib) {
    contributions <- Map(contrib_fun, x, w)
  } else {
    # mimic contributions structure instead of a NULL
    contributions <- rep.int(empty_contrib(levels), length(time))
    names(contributions) <- time
  }

  validate_index(new_index(index, contributions, levels, time, chainable))
}
