% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/group-generics.R
\name{methods-group-generic}
\alias{methods-group-generic}
\alias{Math.pdqr}
\alias{Ops.pdqr}
\alias{Summary.pdqr}
\title{Pdqr methods for S3 group generic functions}
\usage{
\method{Math}{pdqr}(x, ...)

\method{Ops}{pdqr}(e1, e2)

\method{Summary}{pdqr}(..., na.rm = FALSE)
}
\arguments{
\item{x, e1, e2}{Objects.}

\item{...}{Further arguments passed to methods.}

\item{na.rm}{Logical: should missing values be removed?}
}
\value{
All methods return pdqr-function which represents the result of
applying certain function to random variable(s) described with input
pdqr-function(s). \strong{Note} that independence of input random variables is
assumed, i.e. \code{f + f} is not the same as \code{2*f} (see Examples).
}
\description{
There are custom methods implemented for three out of four \link[base:groupGeneric]{S3 group generic functions}: \code{Math}, \code{Ops}, \code{Summary}. \strong{Note} that many
of them have random nature with an idea of generating samples from input
pdqr-functions, performing certain operation on them (results in one
generated sample from desired random variable), and creating new
pdqr-function with appropriate \link[=new-pdqr]{new_*()} function. This is done
with \code{\link[=form_trans]{form_trans()}}, so all rules for determining \link[=meta_class]{class} and
\link[=meta_type]{type} of output is taken from it.
}
\details{
Customization of method behavior may be done using mechanism of
\link[base:options]{options()}. These are the possible options:
\itemize{
\item \strong{\code{pdqr.group_gen.args_new}}. This will be used as \code{args_new} argument for
\code{form_trans()} in methods with random nature. Default is \code{list()}.
\item \strong{\code{pdqr.group_gen.n_sample}}. This will be used as \code{n_sample} argument for
\code{form_trans()} in methods with random nature. Default is 10000.
\item \strong{\code{pdqr.group_gen.repair_supp_method}}. All methods that have random
nature take care of output support by trying to "repair" it, because default
use of \verb{new_*()} functions returns a slightly bigger support than range of
input sample (see Examples). Repairing is done with \code{\link[=form_resupport]{form_resupport()}} where
target support is computed separately and \code{method} argument is controlled by
this option (preferred ones are \code{"reflect"}, default, and \code{"trim"}). In most
cases output support is computed directly based on special features of
generic function. But for some difficult cases, like \code{gamma()}, \code{digamma()},
\code{lgamma()}, \code{psigamma()}, \code{^}, and \code{\%\%} it is a result of simulation (i.e.
slightly random, which slightly increases random nature of those methods).
}
}
\section{Math}{


This family of S3 generics represents mathematical functions. Most of the
methods have \strong{random nature}, except \code{abs()} and \code{sign()} which are
computed directly. Output of \code{sign()} has "discrete" type with 3 "x" values:
-1, 0, 1.

\strong{Note} that \code{cumsum()}, \code{cumprod()}, \code{cummmax()}, and \code{cummin()} functions
don't make much sense in these implementations: their outputs represent
random variable, sample of which is computed by applying \verb{cum*()} function to
a sample, generated from input pdqr-function.
}

\section{Ops}{


This family of S3 generics represents common operators. For all functions
(except \code{&} and \code{|}) input can be a pdqr-function or single number.

A list of methods with \strong{non-random nature}:
\itemize{
\item \code{!}, \code{+}, \code{-} in case of single input, i.e. \code{!f} or \code{-f}.
\item Functions representing linear transformation, i.e. adding, subtracting,
multiplying, and dividing by a single number. For example, all \code{f + 1},
\code{2 - f} (which is actually \code{(-f) + 2}), \code{3*f} and \code{f/2} are linear
transformations, but \code{1 / f}, \code{f + g} are not.
\item Functions for comparing: \code{==}, \code{!=}, \code{<}, \code{<=}, \code{>=}, \code{>}. Their output is
\strong{boolean pdqr-function}: "discrete" type function with elements being
exactly 0 and 1. Probability of 0 represents probability of operator output
being false, and 1 - being true. Probability of being true is computed
directly as \strong{limit of empirical estimation from simulations} (as size of
samples grows to infinity). In other words, output is an exact number which
might be approximated by simulating two big samples of same size from input
\code{e1} and \code{e2} (one of which can be a single number), and estimating
probability as share of those pairs from samples for which comparison is
true. \strong{Note} that if at least one input has "continuous" type, then:
\itemize{
\item \code{==} will always have probability 0 of being true because probability
of generating a certain exact one or two numbers from continuous random
variable is zero.
\item \code{!=} will always have probability 1 of being true for the same reason
as above.
\item Pairs \code{>=} and \code{>}, \code{<=} and \code{<} will return the same input because
probability of being equal is always zero.
}
\item Logical functions \code{&} and \code{|}. Their input can be only pdqr-functions
(because single number input doesn't make much sense). They are most useful
for applying to boolean pdqr-functions (see description of functions for
comparing), and warning is thrown in case any input is not a boolean
pdqr-function. \code{&}'s probability of being true is a product of those
probabilities from input \code{e1} and \code{e2}. \code{|}'s probability of being false is a
product of those probabilities from input \code{e1} and \code{e2}. \strong{Note} that
probability of being false is a probability of being equal to 0; of being
true - complementary to that.
}

All other methods are \strong{random}. For example, \code{f + f}, \code{f^g} are random.
}

\section{Summary}{


Methods for \code{all()} and \code{any()} have \strong{non-random nature}. Their input can
be only pdqr-functions, and if any of them is not boolean, a warning is
thrown (because otherwise output doesn't make much sense). They return a
boolean pdqr-function with the following probability of being true:
\itemize{
\item In \code{all()} - probability of \emph{all} input function being true, i.e. product
of probabilities of being true (implemented as complementary to probability
of being equal to 0).
\item In \code{any()} - probability of \emph{any} input function being true, i.e.
complementary probability to product of all functions being false
(implemented as probability of being equal to 0).
}

Methods for \code{sum()}, \code{prod()}, \code{min()}, \code{max()} have \strong{random nature}. They
are implemented to use vectorized version of certain generic, because
transformation function for \code{form_trans()} should be vectorized: for input
samples which all have size n it should also return sample of size n (where
each element is a transformation output for corresponding elements from input
samples). This way \code{min(f, g)} can be read as "random variable
representing minimum of \code{f} and \code{g}", etc.

\strong{Notes}:
\itemize{
\item \code{range()} function doesn't make sense here because it returns 2 numbers per
input and therefore can't be made vectorized. Error is thrown if it is
applied to pdqr-function.
\item Although all \code{sum()}, \code{prod()}, \code{min()}, \code{max()} accept pdqr-functions or
single numbers, using numbers and "continuous" functions simultaneously is
not a great idea. This is because output will be automatically smoothed (as
\code{form_trans()} will use some \verb{new_*()} function) which will give a misleading
picture. For a more realistic output:
\itemize{
\item Instead of \code{min(f, num)} use
\code{form_resupport(f, c(num, NA), method = "winsor")} (see
\code{\link[=form_resupport]{form_resupport()}}).
\item Instead of \code{max(f, num)} use
\code{form_resupport(f, c(NA, num), method = "winsor")}.
\item Instead of \code{sum(f, num)} use \code{f + num}.
\item Instead of \code{prod(f, num)} use \code{f * num}.
}
}
}

\examples{
d_norm <- as_d(dnorm)
d_unif <- as_d(dunif)
d_dis <- new_d(data.frame(x = 1:4, prob = 1:4 / 10), "discrete")

set.seed(101)

# Math
plot(d_norm, main = "Math methods")
  # `abs()` and `sign()` are not random
lines(abs(d_norm), col = "red")
  # All others are random
lines(cos(d_norm), col = "green")
lines(cos(d_norm), col = "blue")

  # Although here distribution shouldn't change, it changes slightly due to
  # random implementation
meta_x_tbl(d_dis)
meta_x_tbl(floor(d_dis))

# Ops
  # Single input, linear transformations, and logical are not random
d_dis > 1
!(d_dis > 1)
d_norm >= (2*d_norm+1)
  # All others are random
plot(d_norm + d_norm)
  # This is an exact reference curve
lines(as_d(dnorm, sd = sqrt(2)), col = "red")

plot(d_dis + d_norm)

plot(d_unif^d_unif)

# Summary
  # `all()` and `any()` are non-random
all(d_dis > 1, d_dis > 1)
  # Others are random
plot(max(d_norm, d_norm, d_norm))

plot(d_norm + d_norm + d_norm)
lines(sum(d_norm, d_norm, d_norm), col = "red")

  # Using single numbers is allowed, but gives misleading output in case of
  # "continuous" functions. Use other functions instead (see documentation).
plot(min(d_unif, 0.5))
lines(form_resupport(d_unif, c(NA, 0.5), method = "winsor"), col = "blue")

# Use `options()` to control methods
plot(d_unif + d_unif)
op <- options(
  pdqr.group_gen.n_sample = 100,
  pdqr.group_gen.args_new = list(adjust = 0.5)
)
lines(d_unif + d_unif, col = "red")
  # `f + f` is different from `2*f` due to independency assumption. Also the
  # latter implemented non-randomly.
lines(2 * d_unif, col = "blue")

# Methods for generics attempt to repair support, so they are more reasonable
# to use than direct use of `form_trans()`
d_unif + d_unif
form_trans(list(d_unif, d_unif), `+`)

}
\seealso{
\code{\link[=summ_prob_true]{summ_prob_true()}} and \code{\link[=summ_prob_false]{summ_prob_false()}} for extracting
probability from boolean pdqr-functions.

Other pdqr methods for generic functions: 
\code{\link{methods-plot}},
\code{\link{methods-print}}
}
\concept{pdqr methods for generic functions}
