% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/testing_fun.R
\name{pcr_test}
\alias{pcr_test}
\title{Statistical testing of PCR data}
\usage{
pcr_test(df, test = "t.test", ...)
}
\arguments{
\item{df}{A data.frame of \eqn{C_T} values with genes in the columns and
samples in rows rows}

\item{test}{A character string; 't.test' default, 'wilcox.test' or 'lm'}

\item{...}{Other arguments for the testing methods}
}
\value{
A data.frame of 5 columns in addition to term when test == 'lm'
\itemize{
  \item term The linear regression comparison terms
  \item gene The column names of df. reference_gene is dropped
  \item estimate The estimate for each term
  \item p_value The p-value for each term
  \item lower The low 95\% confidence interval
  \item upper The high 95\% confidence interval
}
For details about the test methods themselves and different parameters,
consult \code{\link[stats]{t.test}}, \code{\link[stats]{wilcox.test}}
and \code{\link[stats]{lm}}
}
\description{
A unified interface to different statistical significance tests for qPCR
data
}
\details{
The simple t-test can be used to test the significance of the
difference between two conditions \eqn{\Delta C_T}. t-test assumes in
addition, that the input \eqn{C_T} values are normally distributed and the
variance between conditions are comparable. Wilcoxon test can be used when
sample size is small and those two last assumptions are hard to achieve.

Two use the linear regression here. A null hypothesis is formulated as
following,
\deqn{
  C_{T, target, treatment} - C_{T, control, treatment} =
  C_{T, target, control} - C_{T, control, control}
  \quad \textrm{or} \quad  \Delta\Delta C_T
}
This is exactly the \eqn{\Delta\Delta C_T} as explained earlier. So the
\eqn{\Delta\Delta C_T} is estimated and the null is rejected when
\eqn{\Delta\Delta C_T \ne 0}.
}
\examples{
# locate and read data
fl <- system.file('extdata', 'ct4.csv', package = 'pcr')
ct4 <- read.csv(fl)

# make group variable
group <- rep(c('control', 'treatment'), each = 12)

# test using t-test
pcr_test(ct4,
         group_var = group,
         reference_gene = 'ref',
         reference_group = 'control',
         test = 't.test')

# test using wilcox.test
pcr_test(ct4,
         group_var = group,
         reference_gene = 'ref',
         reference_group = 'control',
         test = 'wilcox.test')

# testing using lm
pcr_test(ct4,
         group_var = group,
         reference_gene = 'ref',
         reference_group = 'control',
         test = 'lm')

# testing advanced designs using a model matrix
# make a model matrix
group <- relevel(factor(group), ref = 'control')
dose <- rep(c(100, 80, 60, 40), each = 3, times = 2)
mm <- model.matrix(~group:dose, data = data.frame(group, dose))

# test using lm
pcr_test(ct4,
         reference_gene = 'ref',
         model_matrix = mm,
         test = 'lm')

# using linear models to check the effect of RNA quality
# make a model matrix
group <- relevel(factor(group), ref = 'control')
set.seed(1234)
quality <- scale(rnorm(n = 24, mean = 1.9, sd = .1))
mm <- model.matrix(~group + group:quality, data = data.frame(group, quality))

# testing using lm
pcr_test(ct4,
         reference_gene = 'ref',
         model_matrix = mm,
         test = 'lm')

# using linear model to check the effects of mixing separate runs
# make a model matrix
group <- relevel(factor(group), ref = 'control')
run <- factor(rep(c(1:3), 8))
mm <- model.matrix(~group + group:run, data = data.frame(group, run))

# test using lm
pcr_test(ct4,
         reference_gene = 'ref',
         model_matrix = mm,
         test = 'lm')

}
\references{
Yuan, Joshua S, Ann Reed, Feng Chen, and Neal Stewart. 2006.
“Statistical Analysis of Real-Time PCR Data.” BMC Bioinformatics 7 (85).
BioMed Central. doi:10.1186/1471-2105-7-85.
}
