% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dbi_to_parquet.R
\name{dbi_to_parquet}
\alias{dbi_to_parquet}
\title{Convert a SQL Query on a DBI connection to parquet format}
\usage{
dbi_to_parquet(
  conn,
  sql_query,
  path_to_parquet,
  max_memory,
  max_rows,
  chunk_memory_sample_lines = 10000,
  partition = "no",
  compression = "snappy",
  compression_level = NULL,
  ...
)
}
\arguments{
\item{conn}{A DBIConnection object, as return by DBI::dbConnect}

\item{sql_query}{a character string containing an SQL query (this argument is passed to DBI::dbSendQuery)}

\item{path_to_parquet}{String that indicates the path to the directory where the parquet files will be stored.}

\item{max_memory}{Memory size (in Mb) in which data of one parquet file should roughly fit.}

\item{max_rows}{Number of lines that defines the size of the chunk.
This argument can not be filled in if max_memory is used.}

\item{chunk_memory_sample_lines}{Number of lines to read to evaluate max_memory. Default to 10 000.}

\item{partition}{String ("yes" or "no" - by default) that indicates whether you want to create a partitioned parquet file.
If "yes", `"partitioning"` argument must be filled in. In this case, a folder will be created for each modality of the variable filled in `"partitioning"`.
Be careful, this argument can not be "yes" if `max_memory` or `max_rows` argument are not NULL.}

\item{compression}{compression algorithm. Default "snappy".}

\item{compression_level}{compression level. Meaning depends on compression algorithm.}

\item{...}{additional format-specific arguments, see \href{https://arrow.apache.org/docs/r/reference/write_parquet.html}{arrow::write_parquet()}
and \href{https://arrow.apache.org/docs/r/reference/write_dataset.html}{arrow::write_dataset()} for more informations.}
}
\value{
A parquet file, invisibly
}
\description{
This function allows to convert a SQL query from a DBI to parquet format.\cr

It handles all DBI supported databases.

Two conversions possibilities are offered :

\itemize{

\item{Convert to a single parquet file. Argument `path_to_parquet` must then be used;}
\item{Convert to a partitioned parquet file. Additionnal arguments `partition` and `partitioning` must then be used;}

}

Examples explain how to convert a query to a chunked dataset
}
\examples{

# Conversion from a sqlite dbi connection to a single parquet file :

dbi_connection <- DBI::dbConnect(RSQLite::SQLite(),
  system.file("extdata","iris.sqlite",package = "parquetize"))

# Reading iris table from local sqlite database
# and conversion to one parquet file :

dbi_to_parquet(
  conn = dbi_connection,
  sql_query = "SELECT * FROM iris",
  path_to_parquet = tempfile(fileext=".parquet"),
)

# Reading iris table from local sqlite database by chunk (using
# `max_memory` argument) and conversion to multiple parquet files

dbi_to_parquet(
  conn = dbi_connection,
  sql_query = "SELECT * FROM iris",
  path_to_parquet = tempdir(),
  max_memory = 2 / 1024
)

# Using chunk and partition together is not possible directly but easy to do :
# Reading iris table from local sqlite database by chunk (using
# `max_memory` argument) and conversion to arrow dataset partitioned by
# species

# get unique values of column "iris from table "iris"
partitions <- get_partitions(dbi_connection, table = "iris", column = "Species")

# loop over those values
for (species in partitions) {
  dbi_to_parquet(
    conn = dbi_connection,
    # use glue_sql to create the query filtering the partition
    sql_query = glue::glue_sql("SELECT * FROM iris where Species = {species}",
                               .con = dbi_connection),
    # add the partition name in the output dir to respect parquet partition schema
    path_to_parquet = file.path(tempdir(), "iris", paste0("Species=", species)),
    max_memory = 2 / 1024,
  )
}

# If you need a more complicated query to get your partitions, you can use
# dbGetQuery directly :
col_to_partition <- DBI::dbGetQuery(dbi_connection, "SELECT distinct(`Species`) FROM `iris`")[,1]

}
