% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/resolveTreeChar.R
\name{resolveTreeChar}
\alias{resolveTreeChar}
\title{Resolve Polytomies Using Parsimony-Based Reconstruction of a Discrete Character}
\usage{
resolveTreeChar(tree, trait, orderedChar = FALSE, stateBias = NULL,
  iterative = TRUE, type = "MPR", cost = NULL)
}
\arguments{
\item{tree}{A cladogram of type 'phylo'. Any branch lengths are ignored.}

\item{trait}{A vector of trait values for a discrete character, preferably named with taxon names
identical to the tip labels on the input tree.}

\item{orderedChar}{Is the character of interest given for \code{trait} ordered or not?
If FALSE (the default), then for each polytomy, all child nodes that appear to have the
same state as the ancestor node will remain in the polytomy, and any additional states held by child nodes will
each be grouped into their own unique polytomy that forms from a descendant node of the original polytomy.
If TRUE, then the character will be reconstructed with a cost (step)
matrix of a linear, ordered character, and polytomies will be resolved so that lineages with different
states will be placed into a nested ladder that reflects the ordered character. As with the unordered option,
child nodes with a state equivalent to the ancestral node will remain in the polytomy, while more primitive
or more derived states will be sorted into their own separate ladders composed of paraphyletic groups, ordered
so to move 'away' state-by-state from the ancestral node's inferred character state.
This option is not applicable if \code{type = "ACCTRAN"}, as cost matrices cannot
be used with ACCTRAN in \code{ancestral.pars}, and an error will be returned if \code{orderedChar=TRUE} but
a cost matrix is given manually.}

\item{stateBias}{This argument controls how \code{resolveTreeChar} handles ancestral node reconstructions that have
multiple states competing for the maximum weight of any state (i.e. if states 0 and 1 both have 0.4 of the weight). The
default, where \code{stateBias = NULL} causes uncertainty at nodes among states to be treated as a single 'group' identical
to any states within it. Essentially, this means that for the example polytomy where the ancestor hax maximum weight for both 0 and 1,
any child nodes with 0, 1 or both of these states will be considered to have an identical state for the purpose of grouping nodes
for the purpose of further resolving polytomies. If and only if \code{orderedChar = TRUE}, then additional options of
\code{stateBias = 'primitive'} and \code{stateBias = 'derived'} become available, which instead force uncertain node
assignments to either be the most primitive (i.e. the minimum) or the most derived (i.e. the maximum) among the
maximum-weight states. In particular, \code{stateBias = 'primitive'} should favor gains and bias any analysis of
character transitions against finding reversals.}

\item{iterative}{A logical argument which, if TRUE (the default), causes the function to repeat the polytomy-resolving
functionality across the entire tree until the number of nodes stabilizes. If FALSE, polytomies are only passed a single
time.}

\item{type}{The parsimony algorithm applied by \code{ancestral.pars}, which can apply one of two:
"MPR" (the default) is a relatively fast algorithm developed by Hamazawa et al. (1995) and Narushima
and Hanazawa (1997), which relies on reconstructing the states at each internal node by re-rooting at
that node.  "ACCTRAN", the 'accelerated transitions' algorithm (Swofford and Maddison, 1987), favors
character reversal over independent gains when there is ambiguity. The "ACCTRAN" option in
ancestral.pars avoids repeated rerooting of the tree to search for a smaller set of maximum-parsimony
solutions that satisfy the ACCTRAN algorithm, but does so by assigning edge weights.
As of phangorn v1.99-12, both of these algorithms apply
the Sankoff parsimony algorithm, which allows multifurcations (polytomies).}

\item{cost}{A matrix of the cost (i.e. number of steps) necessary to change between states of the input
character trait. If NULL (the
default), the character is assumed to be unordered with equal cost to change from any state to another.
Cost matrices only impact the "MPR" algorithm; if a cost matrix is given but \code{type = "ACCTRAN"}, an error
is issued.}
}
\value{
Returns the resulting tree, which may be fully resolved, partly more resolved or not more resolved at all
(i.e. have less polytomies) depending on what was possible, as constrained by ambiguities in character
reconstructions. Applying \code{\link{multi2di}} is suggested as a post-step to obtain a fully-resolved
cladogram, if one is desired.
}
\description{
This function resolves a set of given topology with less than fully-binary phylogenetic resolution so that
lineages are shifted and internal nodes added that minimize the number of independent character transitions needed to explain
an observed distribution of discrete character states for the taxa on such a tree, under various maximum-parsimony algorithms of
ancestral character reconstruction, powered ultimately by function \code{ancestral.pars} in library \code{phangorn}.
This function is mainly designed for use with poorly resolved trees which are being assessed with the function
\code{\link{minCharChange}}.
}
\details{
As shown in the example code below, this function offers a wide variety of options for manipulating the
maximum-parsimony algorithm used (i.e. MPR versus ACCTRAN), the ordering (or not) of character states,
and potential biasing of uncertainty character state reconstructions (when ordered characters are
assessed). This allows for a wide variety of possible resolutions for a given tree with polytomies
and a discrete character. In general, the author expects that use of this function will be optimal
when applied to ordered characters using one of the \code{stateBias} options, perhaps
\code{stateBias = "primitive"} (based on theoretical expectations for slow evolving characters). However,
anecdotal use of this function with various simulation datasets suggests that the results are quite
variable, and so the best option needs to be assessed based on the prior assumptions regarding the
data and the performance of the dataset with the various arguments of this function.
}
\examples{
# let's write a quick&dirty ancestral trait plotting function

 quickAncPlot<-function(tree,trait,cex,orderedChar=FALSE,type="MPR",cost=NULL){
	ancData<-ancPropStateMat(tree=tree, trait=trait, orderedChar=orderedChar)
	ancCol<-(1:ncol(ancData))+1
 	plot(tree,show.tip.label=FALSE,no.margin=TRUE,direction="upwards")
 	tiplabels(pch=16,pie=ancData[(1:Ntip(tree)),],cex=cex,piecol=ancCol,
		col=0)
 	nodelabels(pie=ancData[-(1:Ntip(tree)),],cex=cex,piecol=ancCol)
 	}

##########

# examples with simulated data

set.seed(2)
tree<-rtree(50)
#simulate under a likelihood model
trait<-rTraitDisc(tree,k=3,rate=0.7)
tree<-degradeTree(tree,prop_collapse=0.6)
tree<-ladderize(tree,right=FALSE)

#a bunch of type=MPR (default) examples
treeUnord<-resolveTreeChar(tree,trait,orderedChar=FALSE)
treeOrd<-resolveTreeChar(tree,trait,orderedChar=TRUE,stateBias=NULL)
treeOrdPrim<-resolveTreeChar(tree,trait,orderedChar=TRUE,stateBias="primitive")
treeOrdDer<-resolveTreeChar(tree,trait,orderedChar=TRUE,stateBias="derived")
#and finally an unordered one with ACCTRAN
treeACCTRAN<-resolveTreeChar(tree,trait,orderedChar=FALSE,type="ACCTRAN")

#compare number of nodes
Nnode(tree)			#original
Nnode(treeUnord)		#unordered, biasStates=NULL, MPR
Nnode(treeOrd)		#ordered, biasStates=NULL
Nnode(treeOrdPrim)	#ordered, biasStates='primitive'
Nnode(treeOrdDer)	#ordered, biasStates='derived'
Nnode(treeACCTRAN)	#unordered, biasStates=NULL, ACCTRAN

#let's compare original tree with unordered-resolved tree
layout(1:2)
quickAncPlot(tree,trait,orderedChar=FALSE,cex=0.3)
text(x=43,y=10,"Original",cex=1.5)
quickAncPlot(treeUnord,trait,orderedChar=FALSE,cex=0.3)
text(x=43,y=10,"orderedChar=FALSE",cex=1.5)
#some resolution gained

#now let's compare the original and ordered, both biasStates=NULL
layout(1:2)
quickAncPlot(tree,trait,orderedChar=FALSE,cex=0.3)
text(x=43,y=10,"Original",cex=1.5)
quickAncPlot(treeOrd,trait,orderedChar=TRUE,cex=0.3)
text(x=43,y=10,"orderedChar=TRUE",cex=1.5)

#now let's compare the three ordered trees
layout(1:3)
quickAncPlot(treeOrd,trait,orderedChar=TRUE,cex=0.3)
text(x=41,y=8,"ordered, biasStates=NULL",cex=1.5)
quickAncPlot(treeOrdPrim,trait,orderedChar=TRUE,cex=0.3)
text(x=41.5,y=8,"ordered, biasStates='primitive'",cex=1.5)
quickAncPlot(treeOrdDer,trait,orderedChar=TRUE,cex=0.3)
text(x=42,y=8,"ordered, biasStates='derived'",cex=1.5)

#let's compare unordered with ordered, biasStates='primitive'
layout(1:2)
quickAncPlot(treeUnord,trait,orderedChar=FALSE,cex=0.3)
text(x=41,y=8,"orderedChar=FALSE",cex=1.5)
quickAncPlot(treeOrdPrim,trait,orderedChar=TRUE,cex=0.3)
text(x=40,y=11,"orderedChar=TRUE",cex=1.5)
text(x=40,y=4,"biasStates='primitive'",cex=1.5)

#let's compare unordered with MPR to unordered with ACCTRAN
layout(1:2)
quickAncPlot(treeUnord,trait,orderedChar=FALSE,type="MPR",cex=0.3)
text(x=41,y=8,"unordered: MPR",cex=1.5)
quickAncPlot(treeACCTRAN,trait,orderedChar=FALSE,type="ACCTRAN",cex=0.3)
text(x=41,y=8,"unordered: ACCTRAN",cex=1.5)

#these comparisons will differ greatly between datasets
	# need to try them on your own

layout(1)
}
\author{
David W. Bapst
}
\references{
Hanazawa, M., H. Narushima, and N. Minaka. 1995. Generating most parsimonious reconstructions on
a tree: A generalization of the Farris-Swofford-Maddison method. Discrete Applied Mathematics
56(2-3):245-265.

Narushima, H., and M. Hanazawa. 1997. A more efficient algorithm for MPR problems in phylogeny.
Discrete Applied Mathematics 80(2-3):231-238.

Schliep, K. P. 2011. phangorn: phylogenetic analysis in R. \emph{Bioinformatics} 27(4):592-593.

Swofford, D. L., and W. P. Maddison. 1987. Reconstructing ancestral character states under
Wagner parsimony. Mathematical Biosciences 87(2):199-229.
}
\seealso{
\code{\link{ancPropStateMat}} which is used internally by this function. This function was
intentionally designed for use with \code{\link{minCharChange}}.
}

