#' Retrieve  QTL colocalisation results of a variant
#'
#' The colocalisation analysis in Open Target Genetics is performed using the coloc method (Giambartolomei et al., 2014).
#'   Coloc is a Bayesian method which, for two traits, integrates evidence over all variants
#'   at a locus to evaluate the following hypotheses:
#'   - H0: No association with either trait
#'   - H1: Association with trait 1, not with trait 2
#'   - H2: Association with trait 2, not with trait 1
#'   - H3: Association with trait 1 and trait 2, two independent SNPs
#'   - H4: Association with trait 1 and trait 2, one shared SNP
#'   This analysis tests whether two independent associations at the same locus are consistent
#'   with having a shared causal variant. Colocalisation of two independent associations from two GWAS studies may suggest a shared causal mechanism.
#'
#' @param study_id Character:  Study ID(s) generated by Open Targets Genetics (e.g GCST90002357).
#' @param variant_id Character: generated ID for variants by Open Targets Genetics (e.g. 1_154119580_C_A) or rsId (rs2494663).
#'
#' @return Returns a data frame of the colocalisation information for a lead variant in a specific study.
#' The output is a tidy data frame with the following data structure:
#' \itemize{
#'   \item{\code{qtlStudyName}:} \emph{Character vector}. QTL study name.
#'   \item{\code{phenotypeId}:} \emph{Character vector}. Phenotype ID.
#'   \item{\code{gene.id}:} \emph{Character vector}. Gene ID.
#'   \item{\code{gene.symbol}:} \emph{Character vector}. Gene symbol.
#'   \item{\code{name}:} \emph{Character vector}. Tissue name.
#'   \item{\code{indexVariant.id}:} \emph{Character vector}. Index variant ID.
#'   \item{\code{indexVariant.rsId}:} \emph{Character vector}. Index variant rsID.
#'   \item{\code{beta}:} \emph{Numeric}. Beta value.
#'   \item{\code{h4}:} \emph{Numeric}. h4 value.
#'   \item{\code{h3}:} \emph{Numeric}. h3 value.
#'   \item{\code{log2h4h3}:} \emph{Numeric}. Log2(h4/h3) value.
#' }
#'
#' @examples
#' \dontrun{
#' result <- qtlColocalisationVariantQuery(study_id = "GCST90002357", variant_id = "1_154119580_C_A")
#' result <- qtlColocalisationVariantQuery(study_id = "GCST90002357", variant_id = "rs2494663")
#' }
#' @import dplyr
#' @import cli
#' @import ghql
#' @importFrom magrittr %>%
#' @export
#'
#'

qtlColocalisationVariantQuery <- function(study_id, variant_id) {

  # Connect to the database

tryCatch({
  cli::cli_progress_step("Connecting to the Open Targets Genetics GrpahQL API...", spinner = TRUE)
  otg_cli <- ghql::GraphqlClient$new(url = "https://api.genetics.opentargets.org/graphql")
  otg_qry <- ghql::Query$new()

  # Check variant ID format
  if (grepl(pattern = "rs\\d+", variant_id)) {

    # Convert rs ID to variant ID
    query_searchid <- "query ConvertRSIDtoVID($queryString: String!) {
      search(queryString: $queryString) {
        totalVariants
        variants {
          id
        }
      }
    }"

    variables <- list(queryString = variant_id)
    otg_qry$query(name = "convertid", x = query_searchid)
    id_result <- jsonlite::fromJSON(otg_cli$exec(otg_qry$queries$convertid, variables), flatten = TRUE)$data
    input_variant_id <- id_result$search$variants$id
  }
  else if (grepl(pattern = "\\d+_\\d+_[a-zA-Z]+_[a-zA-Z]+", variant_id)) {
    input_variant_id <- variant_id
  }
  else {
    stop("\nPlease provide a variant ID.")
  }

  ## Query for QTL colocalisation
  query <- "query qtlColocalisationVariantQuery($studyId: String!, $variantId: String!) {
    qtlColocalisation(studyId: $studyId, variantId: $variantId) {
      qtlStudyName
      phenotypeId
      gene {
        id
        symbol
      }
      tissue {
        name
      }
      indexVariant {
        id
        rsId
      }
      beta
      h4
      h3
      log2h4h3
    }
  }"

  # Execute the query
  variables <- list(studyId = study_id, variantId = input_variant_id)
  otg_qry$query(name = "qtl_query", x = query)

  cli::cli_progress_step("Downloading data...", spinner = TRUE)
  qtlcoloc_result <- jsonlite::fromJSON(otg_cli$exec(otg_qry$queries$qtl_query, variables, flatten = TRUE))$data
  df_qtlcoloc <- as.data.frame(qtlcoloc_result$qtlColocalisation)

  return(df_qtlcoloc)

}, error = function(e) {
  # Handling connection timeout
  if(grepl("Timeout was reached", e$message)) {
    stop("Connection timeout reached while connecting to the Open Targets Genetics GraphQL API.")
  } else {
    stop(e) # Handle other types of errors
  }
})
}
