% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/f_posteriorCalcs.R
\name{quantBisect}
\alias{quantBisect}
\title{Find quantiles of the posterior distribution}
\usage{
quantBisect(percent, theta_hat, N, E, qn, digits = 2, limits = c(-1e+05,
  1e+05), max_iter = 2000)
}
\arguments{
\item{percent}{A numeric scalar between 1 and 99 for the desired
percentile (e.g., 5 for 5th percentile).}

\item{theta_hat}{A numeric vector of hyperparameter estimates (likely from
\code{\link{autoHyper}} or from directly minimizing
\code{\link{negLLsquash}}) ordered as:
\eqn{\alpha_1, \beta_1, \alpha_2, \beta_2, P}.}

\item{N}{A whole number vector of actual counts from
\code{\link{processRaw}}.}

\item{E}{A numeric vector of expected counts from \code{\link{processRaw}}.}

\item{qn}{A numeric vector of posterior probabilities that \eqn{\lambda} came
from the first component of the mixture, given \emph{N = n} (i.e., the
mixture fraction). See function \code{\link{Qn}}.}

\item{digits}{A scalar whole number that determines the number of decimal
places used when rounding the results.}

\item{limits}{A whole number vector of length 2 for the upper and lower
bounds of the search space.}

\item{max_iter}{A whole number scalar for the maximum number of iterations.
Used to prevent infinite loops.}
}
\value{
A numeric vector of quantile estimates.
}
\description{
\code{quantBisect} finds the desired quantile of the posterior distribution
using the bisection method. Used to create credibility limits.
}
\details{
The hyperparameter estimates (\code{theta_hat}) are:
  \itemize{
    \item{\eqn{\alpha_1, \beta_1}: }{Parameter estimates of the first
      component of the prior distribution}
    \item{\eqn{\alpha_2, \beta_2}: }{Parameter estimates of the second
      component}
    \item{\eqn{P}: }{Mixture fraction estimate of the prior distribution}
}

Although this function can find any quantile of the posterior
  distribution, it will often be used to calculate the 5th and 95th
  percentiles to create a 90\% credibility interval.

The quantile is calculated by solving for \eqn{x} in the general
  equation \eqn{F(x) = cutoff}, or equivalently, \eqn{F(x) - cutoff = 0},
  where \eqn{F(x)} is the cumulative distribution function of the posterior
  distribution and \eqn{cutoff} is the appropriate cutoff level (e.g., 0.05
  for the 5th percentile).
}
\section{Warning}{

  The \code{digits} argument determines the tolerance for the bisection
  algorithm. The more decimal places you want returned, the longer the run
  time.
}

\examples{
theta_init <- data.frame(
  alpha1 = c(0.2, 0.1),
  beta1  = c(0.1, 0.1),
  alpha2 = c(2,   10),
  beta2  = c(4,   10),
  p      = c(1/3, 0.2)
)
data(caers)
proc <- processRaw(caers)
squashed <- squashData(proc, bin_size = 100, keep_pts = 100)
squashed <- squashData(squashed, count = 2, bin_size = 10, keep_pts = 20)
suppressWarnings(
  theta_hat <- autoHyper(data = squashed, theta_init = theta_init)$estimates
)
qn <- Qn(theta_hat, N = proc$N, E = proc$E)
proc$QUANT_05 <- quantBisect(percent = 5, theta = theta_hat, N = proc$N,
                             E = proc$E, qn = qn)
proc$QUANT_95 <- quantBisect(percent = 95, theta = theta_hat, N = proc$N,
                             E = proc$E, qn = qn)
head(proc)

}
\seealso{
\url{https://en.wikipedia.org/wiki/Bisection_method}

\code{\link{autoHyper}},  \code{\link{exploreHypers}},
  \code{\link{negLLsquash}}, \code{\link{negLL}},
  \code{\link{negLLzero}}, and \code{\link{negLLzeroSquash}} for
  hyperparameter estimation.

\code{\link{processRaw}} for finding counts.

\code{\link{Qn}} for finding mixture fractions.

Other posterior distribution functions: \code{\link{Qn}},
  \code{\link{ebgm}}
}
\keyword{openEBGM}
