% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clean_eval.R
\name{clean_eval}
\alias{clean_eval}
\title{Evaluate the cleaning of occurrences records}
\usage{
clean_eval(
  occ.cl,
  geo.space,
  env.space = NULL,
  level.filter = c("1_det_by_spec"),
  r,
  species = "species",
  decimal.longitude = "decimalLongitude",
  decimal.latitude = "decimalLatitude",
  scientific.name,
  longitude,
  latitude
)
}
\arguments{
\item{occ.cl}{data frame with occurrence records information already
classified by \code{\link{classify_occ}} function.}

\item{geo.space}{a SpatialPolygons* or sf object defining the geographical
space}

\item{env.space}{a SpatialPolygons* or sf object defining the environmental
space. Use the \code{\link{define_env_space}} for create this object.
By default \code{env.space = NULL}, hence do not evaluate the cleaning
in the environmental space.}

\item{level.filter}{a character vector including the levels in
'naturaList_levels' column which filter the occurrence data set.}

\item{r}{a raster with 2 layers representing the environmental variables. If
\code{env.space = NULL}, it could be a single layer raster, from which
the cell size and extent are extracted to produce the composition matrix.}

\item{species}{column name of \code{occ.cl} with the species names.}

\item{decimal.longitude}{column name of \code{occ.cl} longitude in decimal
degrees.}

\item{decimal.latitude}{column name of \code{occ.cl} latitude in decimal
degrees.}

\item{scientific.name}{deprecated, use \code{species} instead.}

\item{longitude}{deprecated, use \code{decimal.longitude} instead}

\item{latitude}{deprecated, use \code{decimal.latitude} instead}
}
\value{
a list in which:

\code{area} data frame remaining area after cleaning proportional to the area
  before cleaning. The values vary from 0 to 1. Column named \code{r.geo.area}
  is the remaining area for all species in the geographic space and the
  \code{r.env.area} in the environmental space.

\code{comp} data frame with composition of species in sites (cells from raster
  layers) before cleaning (\code{comp$comp$BC}) and after cleaning
  (\code{comp$comp$AC}). The number of rows is equal the number of cells in
  \code{r}, and number of columns is equal to the number of species in the
  \code{occ.cl}.

\code{rich} data frame with a single column with the richness of each site

\code{site.coords} data frame with site's coordinates. It facilitates to built
  raster layers from results using \code{\link[raster]{rasterFromXYZ}}
}
\description{
This function compare the area occupied by
a species before and after pass through the cleaning procedure according to
the chosen level of filter.
The comparison can be made by measuring area in the geographical and in the
environmental space
}
\examples{
\donttest{

  library(raster)
  library(sp)

  data("speciaLists")
  data("cyathea.br")

  # classify
  occ.cl <- classify_occ(cyathea.br, speciaLists)
  nrow(occ.cl)

  # delimit the geographic space
  # land area
  data("BR")
  un.land <- BR

  # Transform occurrence data in SpatialPointsDataFrame
  spdf.occ.cl <- sp::SpatialPoints(occ.cl[, c("decimalLongitude", "decimalLatitude")])

  # Crop by geographic space
  spdf.occ.cl <- raster::crop(spdf.occ.cl, un.land)

  # download climate data
  # this function will download the bioclim raster layer
  # to your work directory
  bioclim <- raster::getData('worldclim', var='bio', res=10)

  # redefine the extent of bioclim layers based on buffer around the occurrences
  c.bioclim <- raster::crop(
    bioclim,
    raster::buffer(spdf.occ.cl, 100000)
  ) # 100km buffer

  # select two layers
  raster.temp.prec <- c.bioclim[[c("bio1", "bio12")]]
  df.temp.prec <- raster::as.data.frame(raster.temp.prec)

  ### Define the environmental space for analysis
  # this function will create a boundary of available environmental space,
  # analogous to the continent boundary in the geographical space
  env.space <- define_env_space(df.temp.prec, buffer.size = 0.05)

  # geo space based on crop
  c.geo.space <- raster::crop(un.land, c.bioclim)

  # filter by year to be consistent with the environmental data
  occ.class.1970 <-  occ.cl \%>\%
    dplyr::filter(year >= 1970)

  ### run the evaluation
  cl.eval <- clean_eval(occ.class.1970,
                        env.space = env.space,
                        geo.space = c.geo.space,
                        r = raster.temp.prec)

  #area results
  cl.eval$area


  ### richness maps
  ## it makes sense if there are more than one species
  rich.before.clean <- raster::rasterFromXYZ(cbind(cl.eval$site.coords,
                                           cl.eval$rich$rich.BC))
  rich.after.clean <- raster::rasterFromXYZ(cbind(cl.eval$site.coords,
                                          cl.eval$rich$rich.AC))

  raster::plot(rich.before.clean)
  raster::plot(rich.after.clean)

  ### species area map
  comp.bc <- as.data.frame(cl.eval$comp$comp.BC)
  comp.ac <- as.data.frame(cl.eval$comp$comp.AC)

  c.villosa.bc <- raster::rasterFromXYZ(cbind(cl.eval$site.coords,
                                              comp.bc$`Cyathea villosa`))
  c.villosa.ac <- raster::rasterFromXYZ(cbind(cl.eval$site.coords,
                                              comp.ac$`Cyathea villosa`))

  raster::plot(c.villosa.bc)
  raster::plot(c.villosa.ac)
}

}
\seealso{
[define_env_space()]
}
