% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glm.mp.R
\name{glm.mp}
\alias{glm.mp}
\title{Build a multinomial-Poisson GLM for nominal response data}
\usage{
glm.mp(formula, data, ...)
}
\arguments{
\item{formula}{A formula object in the style of, e.g., \code{Y ~ X1*X2}, where \code{X1} and \code{X2}
are factors or predictors. The response \code{Y} must be of type \code{factor}. See the \code{formula}
entry for \code{\link[stats]{glm}}.}

\item{data}{A data frame in long-format. See the \code{data} entry for \code{\link[stats]{glm}}.}

\item{...}{Additional arguments to be passed to \code{\link[stats]{glm}}. Generally, these are
unnecessary but are provided for advanced users. They should not specify \code{formula}, \code{data},
or \code{family} arguments. See \code{\link[stats]{glm}} for valid arguments.}
}
\value{
A Poisson regression model of type \code{\link[stats]{glm}}. See the return value for
\code{\link[stats]{glm}}.
}
\description{
This function uses the multinomial-Poisson trick to analyze \strong{nominal response} data using a Poisson
generalized linear model (GLM). The nominal response should be a factor with two or more unordered
categories. The independent variables should be between-subjects factors and/or numeric predictors.
}
\details{
This function should be used for nominal response data with only between-subjects factors or predictors.
In essence, it provides for the equivalent of \code{\link[stats]{glm}} with \code{family=multinomial},
were that option to exist. (That option does not exist, but \code{\link[nnet]{multinom}} serves the same
purpose.)

For data with repeated measures, use \code{\link{glmer.mp}}, which can take random factors and thus handle
correlated responses.

Users wishing to verify the correctness of \code{glm.mp} should compare \code{\link{Anova.mp}} results
to \code{\link[car]{Anova}} results for models built with \code{\link[stats]{glm}} using
\code{family=binomial} (for dichotomous responses) or \code{\link[nnet]{multinom}} (for polytomous
responses). The results should be similar.

\emph{Post hoc} pairwise comparisons for factors can be conducted with \code{\link{glm.mp.con}}.
}
\examples{
library(car)
library(nnet)

## between-subjects factors (X1,X2) with dichotomous response (Y)
data(bs2, package="multpois")

bs2$PId = factor(bs2$PId)
bs2$Y = factor(bs2$Y)
bs2$X1 = factor(bs2$X1)
bs2$X2 = factor(bs2$X2)
contrasts(bs2$X1) <- "contr.sum"
contrasts(bs2$X2) <- "contr.sum"

m1 = glm(Y ~ X1*X2, data=bs2, family=binomial)
Anova(m1, type=3)

m2 = glm.mp(Y ~ X1*X2, data=bs2) # compare
Anova.mp(m2, type=3)

## between-subjects factors (X1,X2) with polytomous response (Y)
data(bs3, package="multpois")

bs3$PId = factor(bs3$PId)
bs3$Y = factor(bs3$Y)
bs3$X1 = factor(bs3$X1)
bs3$X2 = factor(bs3$X2)
contrasts(bs3$X1) <- "contr.sum"
contrasts(bs3$X2) <- "contr.sum"

m3 = multinom(Y ~ X1*X2, data=bs3, trace=FALSE)
Anova(m3, type=3)

m4 = glm.mp(Y ~ X1*X2, data=bs3) # compare
Anova.mp(m4, type=3)

}
\references{
Baker, S.G. (1994). The multinomial-Poisson transformation.
\emph{The Statistician 43} (4), pp. 495-504. \doi{10.2307/2348134}

Guimaraes, P. (2004). Understanding the multinomial-Poisson
transformation. \emph{The Stata Journal 4} (3), pp. 265-273.
\url{https://www.stata-journal.com/article.html?article=st0069}

Lee, J.Y.L., Green, P.J.,and Ryan, L.M. (2017). On the “Poisson
trick” and its extensions for fitting multinomial regression models. \emph{arXiv
preprint} available at \doi{10.48550/arXiv.1707.08538}
}
\seealso{
\code{\link[=Anova.mp]{Anova.mp()}}, \code{\link[=glm.mp.con]{glm.mp.con()}}, \code{\link[=glmer.mp]{glmer.mp()}}, \code{\link[=glmer.mp.con]{glmer.mp.con()}}, \code{\link[stats:glm]{stats::glm()}}, \code{\link[nnet:multinom]{nnet::multinom()}}
}
\author{
Jacob O. Wobbrock
}
