% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optim_mego.R
\name{MEGO}
\alias{MEGO}
\title{MEGO: Multi-Objective Efficient Global Optimization Algorithm based on
scalarization of the objectives}
\usage{
MEGO(
  model,
  fun,
  nsteps,
  lower = rep(0, model@d),
  upper = rep(1, model@d),
  quiet = TRUE,
  control = NULL,
  optimcontrol = NULL
)
}
\arguments{
\item{model}{An object of class \code{\link{mkm}}. This \code{model} must have a single
objective (\code{model@m == 1}).}

\item{fun}{The multi-objective and constraint cost function to be optimized.
This function must return a vector with the size of \code{model@m +
model@j} where \code{model@m} are the number of objectives and
\code{model@j} the number of the constraints,}

\item{nsteps}{An integer representing the desired number of iterations,}

\item{lower}{Vector of lower bounds for the variables to be optimized over
(default: 0 with length = \code{model@d}),}

\item{upper}{Vector of upper bounds for the variables to be optimized over
(default: 1 with length = \code{model@d}),}

\item{quiet}{Logical indicating the verbosity of the routine,}

\item{control}{An optional list of control parameters, some of them passed to
the EI function. One can control:
  \describe{
   \item{\code{minimization}}{logical specifying if EI is used in minimization or in maximization
   (default: \code{TRUE})}
   \item{\code{plugin}}{optional scalar, if not provided, the minimum (or maximum) of the current
    feasible observations. If there isn't any feasible design plugin is set to \code{NA} and the
    algorithm returns the value of the probability of constraints be met.}
   \item{\code{envir}}{optional environment specifying where to assign intermediate values.
    Default: \code{NULL}.}
  }}

\item{optimcontrol}{Optional list of control parameters passed to the
\code{\link[GenSA]{GenSA}} function. Please, note that the values are
passed as the \code{control} parameter inside the \code{GenSA} function (\code{genSA(control = optimcontrol)}).}
}
\value{
updated \code{\link{mkm}} model
}
\description{
Executes \code{nsteps} iterations of the MEGO method to an object of class
\code{\link{mkm}}. At each step, a weighted kriging model is re-estimated
(including covariance parameters re-estimation) based on the initial design
points plus the points visited during all previous iterations; then a new
point is obtained by maximizing the Constrained Expected Improvement
criterion (EI).
}
\details{
Note that since MEGO is works by scalarizing a cost function, this technique
is well suited for single objective problems with multiple constraints.
}
\examples{
# ----------------
# The Nowacki Beam
# ----------------
n <- 20
d <- 2
nsteps <- 1 # value has been set to 1 to save compliation time, change this value to 40.
fun <- nowacki_beam
doe <- replicate(d,sample(0:n,n))/n
res <- t(apply(doe, 1, fun))
model <- mkm(doe, res, modelcontrol = list(objective = 1:2, lower = rep(0.1,d)))
model <- MEGO(model, fun, nsteps, quiet = FALSE, control = list(rho = 0.1))
plot(nowacki_beam_tps$set)
points(ps(model@response[which(model@feasible),model@objective])$set, col = 'green', pch = 19)

############################################
#### some single objective optimization ####
############################################
\dontrun{
## Those examples are flagged as "don't run" only to save compilation time. ##
n.grid <- 20
x.grid <- y.grid <- seq(0,1,length=n.grid)
design.grid <- expand.grid(x.grid, y.grid)
response.grid <- apply(design.grid, 1, DiceKriging::branin)
z.grid <- matrix(response.grid, n.grid, n.grid)

# -----------------------------------
# Branin-Hoo function (unconstrained)
# -----------------------------------
n <- 10
d <- 2
doe <- replicate(d,sample(0:n,n))/n
fun <- DiceKriging::branin
res <- apply(doe, 1, fun)
model <- mkm(doe, res, modelcontrol = list(lower=rep(0.1,d)))
model <- MEGO(model, fun, 10, quiet = FALSE)
contour(x.grid,y.grid,z.grid,40)
points(model@design, col=ifelse(model@feasible,'blue','red'))
# ---------------------------------------
# Branin-Hoo function (simple constraint)
# ---------------------------------------
n <- 10
d <- 2
doe <- replicate(d,sample(0:n,n))/n
fun_cost <- DiceKriging::branin
fun_cntr <- function(x) 0.2 - prod(x)
fun <- function(x) return(c(fun_cost(x),fun_cntr(x)))
res <- t(apply(doe, 1, fun))
model <- mkm(doe, res, modelcontrol = list(objective = 1, lower=rep(0.1,d)))
model <- MEGO(model, fun, 10, quiet = FALSE)
contour(x.grid,y.grid,z.grid,40)
points(model@design, col=ifelse(model@feasible,'blue','red'))
# ---------------------------------------
# Branin-Hoo function (narrow constraint)
# ---------------------------------------
n <- 10
d <- 2
doe <- replicate(d,sample(0:n,n))/n
fun_cost <- DiceKriging::branin
fun_cntr <- function(x){
 g1 <- 0.9 - sum(x)
 g2 <- sum(x) - 1.1
 g3 <- - x[1] + 0.75
 g4 <- x[2] - 0.25
 return(c(g1,g2,g3,g4))
}
fun <- function(x) return(c(fun_cost(x),fun_cntr(x)))
res <- t(apply(doe, 1, fun))
model <- mkm(doe, res, modelcontrol = list(objective = 1, lower=rep(0.1,d)))
model <- MEGO(model, fun, 10, quiet = FALSE)
contour(x.grid,y.grid,z.grid,40)
points(model@design, col=ifelse(model@feasible,'blue','red'))
# ---------------------------------------------
# Branin-Hoo function (disconnected constraint)
# ---------------------------------------------
n <- 10
d <- 2
doe <- replicate(d,sample(0:n,n))/n
Griewank <-  function(x) {
 ii <- c(1:length(x))
  sum <- sum(x^2/4000)
  prod <- prod(cos(x/sqrt(ii)))
  y <- sum - prod + 1
  return(y)
}
fun_cost <- DiceKriging::branin
fun_cntr <- function(x) 1.6 - Griewank(x*10-5)
fun <- function(x) return(c(fun_cost(x),fun_cntr(x)))
res <- t(apply(doe, 1, fun))
model <- mkm(doe, res, modelcontrol = list(objective = 1, lower=c(0.1,0.1)))
model <- MEGO(model, fun, 10, quiet = FALSE)
contour(x.grid,y.grid,z.grid,40)
points(model@design, col=ifelse(model@feasible,'blue','red'))
}
}
\references{
Knowles, J. (2006). ParEGO: a hybrid algorithm with on-line
  landscape approximation for expensive multiobjective optimization problems.
  \emph{IEEE Transactions on Evolutionary Computation}, 10(1), 50-66.
}
