% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Tuner.R
\name{Tuner}
\alias{Tuner}
\title{Tuner}
\format{\link[R6:R6Class]{R6::R6Class} object.}
\description{
Abstract \code{Tuner} class that implements the base functionality each tuner must provide.
A tuner is an object that describes the tuning strategy, i.e. how to optimize the black-box
function and its feasible set defined by the \link{TuningInstance} object.

A list of measures can be passed to the instance, and they will always be all evaluated.
However, single-criteria tuners optimize only the first measure.

A tuner must write its result to the \code{assign_result} method of the \link[=TuningInstance]{Tuninginstance} at the end of its tuning in
order to store the best selected hyperparameter configuration and its estimated performance vector.
}
\section{Construction}{
\preformatted{tuner = Tuner$new(param_set, param_classes, properties, packages = character())
}
\itemize{
\item \code{param_set} :: \link[paradox:ParamSet]{paradox::ParamSet}\cr
Set of control parameters for tuner.
\item \code{param_classes} :: \code{character()}\cr
Supported parameter classes for learner hyperparameters that the tuner can optimize, subclasses of \link[paradox:Param]{paradox::Param}.
\item \code{properties} :: \code{character()}\cr
Set of properties of the tuner. Must be a subset of \code{\link[=mlr_reflections]{mlr_reflections$tuner_properties}}.
\item \code{packages} :: \code{character()}\cr
Set of required packages.
Note that these packages will be loaded via \code{\link[=requireNamespace]{requireNamespace()}}, and are not attached.
}
}

\section{Fields}{

\itemize{
\item \code{param_set} :: \link[paradox:ParamSet]{paradox::ParamSet}; from construction.
\item \code{param_classes} :: \code{character()}\cr
\item \code{properties} :: `character(); from construction.
\item \code{packages} :: \code{character()}; from construction.
}
}

\section{Methods}{

\itemize{
\item \code{tune(instance)}\cr
\link{TuningInstance} -> \code{self}\cr
Performs the tuning on a \link{TuningInstance} until termination.
}
}

\section{Private Methods}{

\itemize{
\item \code{tune_internal(instance)} -> \code{NULL}\cr
Abstract base method. Implement to specify tuning of your subclass.
See technical details sections.
\item \code{assign_result(instance)} -> \code{NULL}\cr
Abstract base method. Implement to specify how the final configuration is selected.
See technical details sections.
}
}

\section{Technical Details and Subclasses}{

A subclass is implemented in the following way:
\itemize{
\item Inherit from Tuner
\item Specify the private abstract method \verb{$tune_internal()} and use it to call into your optimizer.
\item You need to call \code{instance$eval_batch()} to evaluate design points.
\item The batch evaluation is requested at the \link{TuningInstance} object \code{instance},
so each batch is possibly executed in parallel via \code{\link[mlr3:benchmark]{mlr3::benchmark()}},
and all evaluations are stored inside of \code{instance$bmr}.
\item Before and after the batch evaluation, the \link{Terminator} is checked, and if it is positive,
an exception of class \code{"terminated_error"} is generated.
In the later case the current batch of evaluations is still stored in \code{instance},
but the numeric scores are not sent back to the handling optimizer as it has lost execution control.
\item After such an exception was caught we select the best configuration from \code{instance$bmr} and
return it.
\item Note that therefore more points than specified by the \link{Terminator} may be evaluated,
as the Terminator is only checked before and after a batch evaluation,
and not in-between evaluation in a batch.
How many more depends on the setting of the batch size.
\item Overwrite the private super-method \code{assign_result} if you want to decide yourself how to estimate
the final configuration in the instance and its estimated performance.
The default behavior is: We pick the best resample-experiment, regarding the first measure,
then assign its configuration and aggregated performance to the instance.
}
}

\examples{
library(mlr3)
library(paradox)
param_set = ParamSet$new(list(
  ParamDbl$new("cp", lower = 0.001, upper = 0.1)
))
terminator = term("evals", n_evals = 3)
instance = TuningInstance$new(
  task = tsk("iris"),
  learner = lrn("classif.rpart"),
  resampling = rsmp("holdout"),
  measures = msr("classif.ce"),
  param_set = param_set,
  terminator = terminator
)
tt = tnr("random_search") # swap this line to use a different Tuner
tt$tune(instance) # modifies the instance by reference
instance$result # returns best configuration and best performance
instance$archive() # allows access of data.table / benchmark result of full path of all evaluations
}
\seealso{
Other Tuner: 
\code{\link{TunerDesignPoints}},
\code{\link{TunerGenSA}},
\code{\link{TunerGridSearch}},
\code{\link{TunerRandomSearch}},
\code{\link{mlr_tuners}}
}
\concept{Tuner}
