##
##  PURPOSE:  Function to calculate fitted profiles from GLMM_MCMC object
##            based on posterior means of regression coefficients
##
##  AUTHOR:    Arnost Komarek (LaTeX: Arno\v{s}t Kom\'arek)
##             arnost.komarek[AT]mff.cuni.cz
##
##  CREATED:   24/02/2010 (as a stand alone function)
##             26/11/2010:  added to the mixAK package
##             27/12/2010:  argument statistic added
##
##  FUNCTIONS: fitted.GLMM_MCMC
##
## ==========================================================================

## *************************************************************
## fitted.GLMM_MCMC
## *************************************************************
##
fitted.GLMM_MCMC <- function(object, x, z, statistic=c("median", "mean", "Q1", "Q3", "2.5%", "97.5%"), overall=FALSE, ...)
{
  statistic <- match.arg(statistic)
  
  if (statistic == "median") Stat <- "Median"
  else if (statistic == "mean") Stat <- "Mean"
       else if (statistic == "Q1") Stat <- "1st Qu."
            else if (statistic == "Q3") Stat <- "3rd Qu."
                 else if (statistic == "2.5%") Stat <- "2.5%"
                      else if (statistic == "97.5%") Stat <- "97.5%"
  
  RR <- sum(object$R)
  
  if (object$prior.b$priorK != "fixed") stop("implemented only for models with a fixed number of mixture components")
  K_b <- object$K_b[1]
  
  if (sum(object$p)){
    if (missing(x)) stop("x must be given")
    if (RR == 1 & !is.list(x)) x <- list(x)
    if (!is.list(x)) stop("x must be a list")
    if (length(x) != RR) stop("x must be of length", RR)
  }    
  if (sum(object$q)){
    if (missing(z)) stop("z must be given")
    if (RR == 1 & !is.list(z)) z <- list(z)
    if (!is.list(z)) stop("z must be a list")
    if (length(z) != RR) stop("z must be of length", RR)
  }    
  
  #### Posterior summary statistic of means of random effects in each component
  if (object$dimb){
    if (overall){
      K_b <- 1
      if (object$dimb == 1) mu_b <- matrix(object$summ.b.Mean[Stat], nrow=1)
      else                  mu_b <- matrix(object$summ.b.Mean[Stat,], nrow=1)
    }else{
      mu_b <- object$poster.mean.mu_b * matrix(object$scale.b$scale, nrow=K_b, ncol=object$dimb, byrow=TRUE) + matrix(object$scale.b$shift, nrow=K_b, ncol=object$dimb, byrow=TRUE)
    }  
  }

  #### Posterior summary statistic of fixed effects
  if (object$lalpha){
    if (object$lalpha == 1) alpha <- object$summ.alpha[Stat]
    else{
      alpha <- as.numeric(object$summ.alpha[Stat,])
      names(alpha) <- colnames(object$summ.alpha)
    }  
  }  
  
  #### Loop over response types
  qri <- object$q + object$random.intercept
  pfi <- object$p + object$fixed.intercept

  fit <- list()
  for (r in 1:RR){        ### loop (rr)

    ### there are random effects
    if (qri[r]){
      if (r == 1) Eb <- if (K_b > 1) mu_b[, 1:qri[r]] else matrix(mu_b[, 1:qri[r]], nrow=1)
      else        Eb <- if (K_b > 1) mu_b[, (sum(qri[1:(r-1)])+1):sum(qri[1:r])] else matrix(mu_b[, (sum(qri[1:(r-1)])+1):sum(qri[1:r])], nrow=1)

      if (object$q[r]){
        if (object$q[r] == 1){
          if (is.matrix(z[[r]])) if (ncol(z[[r]]) != 1) stop("z[[", r, "]] must have 1 column", sep="") 
          z[[r]] <- matrix(z[[r]], ncol=1)
        }  
        if (!is.matrix(z[[r]])) stop("z[[", r, "]] must be a matrix", sep="")
        if (ncol(z[[r]]) != object$q[r]) stop("z[[", r, "]] must have ", object$q[r], " columns", sep="")
      }  

      if (object$random.intercept[r]){
        if (object$q[r]){
          fit[[r]] <- Eb[1, 1] + z[[r]] %*% Eb[1, 2:qri[r]]
          if (K_b > 1) for (k in 2:K_b) fit[[r]] <- cbind(fit[[r]], Eb[k, 1] + z[[r]] %*% Eb[k, 2:qri[r]])
        }else{
          fit[[r]] <- matrix(Eb, nrow=1, ncol=K_b)
        }          
      }else{
        fit[[r]] <- z[[r]] %*% Eb[1, 1:qri[r]]
        if (K_b > 1) for (k in 2:K_b) fit[[r]] <- cbind(fit[[r]], z[[r]] %*% Eb[k, 1:qri[r]])        
      }  

      if (pfi[r]){                           ### there are also fixed effects

        if (r == 1) alpha_r <- alpha[1:pfi[r]]
        else        alpha_r <- alpha[(sum(pfi[1:(r-1)])+1):sum(pfi[1:r])]
        
        if (object$fixed.intercept[r]){
          fit[[r]] <- fit[[r]] + alpha_r[1]
          alpha_r <- alpha_r[-1]
        }

        if (object$p[r]){
          if (object$p[r] == 1){
            if (is.matrix(x[[r]])) if (ncol(x[[r]]) != 1) stop("x[[", r, "]] must have 1 column", sep="")             
            x[[r]] <- matrix(x[[r]], ncol=1)
          }  
          if (!is.matrix(x[[r]])) stop("x[[", r, "]] must be a matrix", sep="")
          if (ncol(x[[r]]) != object$p[r]) stop("x[[", r, "]] must have ", object$p[r], " columns", sep="")

          if (object$q[r]){      ## fit[[r]] is n x K matrix
            fit[[r]] <- fit[[r]] + matrix(rep(x[[r]] %*% alpha_r, K_b), ncol=K_b)
          }else{                 ## only random intercept among random effects -> fit[[r]] is a 1 x K matrix
            fit[[r]] <- matrix(rep(fit[[r]], nrow(x[[r]])), byrow=TRUE, ncol=K_b) + matrix(rep(x[[r]] %*% alpha_r, K_b), ncol=K_b)
          }  
        }  
      }

    ### only fixed effects in the model  
    }else{

      if (object$p[r]){
        if (object$p[r] == 1) x[[r]] <- matrix(x[[r]], ncol=1)
        if (!is.matrix(x[[r]])){
          if (is.matrix(x[[r]])) if (ncol(x[[r]]) != 1) stop("x[[", r, "]] must have 1 column", sep="")
          stop("x[[", r, "]] must be a matrix", sep="")
        }  
        if (ncol(x[[r]]) != object$p[r]) stop("x[[", r, "]] must have ", object$p[r], " columns", sep="")
      }  
      
      stop("This part not yet implemented. Please, contact the author.")
    }

    ### inverse link function
    if (object$dist[r] == "binomial(logit)"){
      efit <- exp(fit[[r]])
      fit[[r]] <- efit / (1 + efit)
    }else{
      if (object$dist[r] == "poisson(log)"){
        fit[[r]] <- exp(fit[[r]])
      }else{
        if (object$dist[r] != "gaussian") stop("Not (yet) implemented for dist: ", object$dist[r], sep="")
      }  
    }  
  }     ### end of loop (rr)
 
  return(fit)  
}  

