% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_aggregate_generations}
\alias{mies_aggregate_generations}
\title{Get Aggregated Performance Values by Generation}
\usage{
mies_aggregate_generations(
  inst,
  objectives = inst$archive$codomain$ids(),
  aggregations = list(min = min, mean = mean, max = max, median = stats::median, size =
    length),
  as_fitnesses = TRUE,
  survivors_only = TRUE,
  condition_on_budget_id = NULL
)
}
\arguments{
\item{inst}{(\code{OptimInstance})\cr
Optimization instance to evaluate.}

\item{objectives}{(\code{character})\cr
Objectives for which to calculate aggregates. Must be a subset of the codomain elements of \code{inst}, but when \code{as_fitnesses} is \code{TRUE}, elements
that are neither being minimized nor maximized are ignored.}

\item{aggregations}{(named \code{list} of \code{function})\cr
List containing aggregation functions to be evaluated on a vector of objective falues for each generation. These functions should take
a single argument and return a scalar value.}

\item{as_fitnesses}{(\code{logical(1)})\cr
Whether to transform performance values into "fitness" values that are always to be maximized.
This means that values that objectives that should originally be minimized are multiplied with -1,
and that parts of the objective codomain that are neither being minimized nor maximized are dropped.
Default \code{TRUE}.}

\item{survivors_only}{(\code{logical(1)})\cr
Whether to ignore configurations that have \code{"eol"} set to the given generation, i.e. individuals that were killed during that generation.
When this is \code{TRUE} (default), then only individuals that are alive at the \emph{end} of a generation are considered; otherwise all individuals
alive at any point of a generation are considered. If it is \code{TRUE}, this leads to individuals that have \code{"dob"} == \code{"eol"} being ignored.}

\item{condition_on_budget_id}{(\code{character(1)} | \code{NULL})\cr
Budget component when doing multi-fidelity optimization. When this is given, then for each generation, only individuals with the highest value for this
component are considered. If \code{survivors_only} is \code{TRUE}, this means the highest value of all survivors of a given generation, if it is \code{FALSE}, then it
is the highest value of all individuals alive at any point of a generation. To ignore possible budget-parameters, set this to \code{NULL} (default).
This is inparticular necessary when fidelity is not monotonically increasing (e.g. if it is categorical).}
}
\value{
a \code{\link[data.table:data.table]{data.table}} with the column \code{"dob"}, indicating the generation, as well as further
columns named by the items in \code{aggregations}. There is more on element in \code{objectives}
(or more than one element not being minimized/maximized when \code{as_fitnesses} is \code{TRUE}), then columns are named \verb{<aggregations element name>.<objective name>}.
Otherwise, they are named by \verb{<aggregations element name>} only. To get a guarantee that elements are only named after elements in \code{aggregations}, set \code{objectives}
to a length 1 \code{character}.
}
\description{
Get evaluated performance values from an \code{OptimInstance} aggregated for each generation.
This may either concern all individuals that were alive at the end of a given generation (\code{survivors_only} \code{TRUE})
or at any point during a generation (\code{survivors_only} \code{FALSE}).

The result is a single \code{\link[data.table:data.table]{data.table}} object with a \code{dob} column indicating the
generation, as well as one column for each \code{aggregations} entry crossed with each objective of \code{inst}.

See \code{\link[=mies_generation_apply]{mies_generation_apply()}} on how to apply functions to entire fitness-matrices, not only individual objectives.
}
\examples{
library("bbotk")
lgr::threshold("warn")

# Define the objective to optimize
objective <- ObjectiveRFun$new(
  fun = function(xs) {
    z <- 10 - exp(-xs$x^2 - xs$y^2) + 2 * exp(-(2 - xs$x)^2 - (2 - xs$y)^2)
    list(Obj = z)
  },
  domain = ps(x = p_dbl(-2, 4), y = p_dbl(-2, 4)),
  codomain = ps(Obj = p_dbl(tags = "minimize"))
)

oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 6)
)

op <- opt("mies",
  lambda = 2, mu = 2,
  mutator = mut("gauss", sdev = 0.1),
  recombinator = rec("xounif"),
  parent_selector = sel("best")
)
set.seed(1)
op$optimize(oi)

# negates objectives that are minimized:
mies_aggregate_generations(oi)

# silly aggregation: first element
mies_aggregate_generations(oi, aggregations = list(first = function(x) x[1]))

# real objective values:
mies_aggregate_generations(oi, as_fitnesses = FALSE)

# Individuals that died are included:
mies_aggregate_generations(oi, survivors_only = FALSE)
}
\seealso{
Other aggregation methods: 
\code{\link{mies_aggregate_single_generation}()},
\code{\link{mies_get_generation_results}()}
}
\concept{aggregation methods}
