#' Row and column notation
#' 
#' @description
#' It is often convenient to represent row and column names 
#' with notation that includes a prefix and a suffix,
#' with corresponding separators or start-end string sequences.
#' There are several functions that call `notation_vec()` to generate specialized versions
#' or otherwise manipulate row and column names on their own or as row or column names.
#' 
#' * `notation_vec()` Builds a vector of notation symbols in a standard format 
#'                    that is used by `matsbyname` in several places.
#'                    By default, it builds a list of notation symbols that provides an arrow 
#'                    separator (" -> ") between prefix and suffix.
#' * `arrow_notation()` Builds a list of notation symbols that provides an arrow separator (" -> ")
#'                      between prefix and suffix.
#' * `paren_notation()` Builds a list of notation symbols that provides parentheses around the suffix ("prefix (suffix)").
#' * `bracket_notation()` builds a list of notation symbols that provides square brackets around the suffix ("prefix \[suffix\]").
#' * `split_pref_suff()` Splits prefixes from suffixes, returning each in a list with names `pref` and `suff`. 
#'                       If no prefix or suffix delimiters are found, `x` is returned in the `pref` item, unmodified, 
#'                       and the `suff` item is returned as `""` (an empty string).
#'                       If there is no prefix, and empty string is returned for the `pref` item.
#'                       If there is no suffix, and empty string is returned for the `suff` item.
#' * `paste_pref_suff()` `paste0`'s prefixes and suffixes, the inverse of `split_pref_suff()`.
#' * `flip_pref_suff()` Switches the location of prefix and suffix, such that the prefix becomes the suffix, and
#'                      the suffix becomes the prefix.
#'                      E.g., "a -> b" becomes "b -> a" or "a \[b\]" becomes "b \[a\]".
#' * `keep_pref_suff()` Selects only prefix or suffix, discarding notational elements 
#'                      and the rejected part.
#' * `switch_notation()` Switches from one type of notation to another based on the `from` and `to` arguments.
#'                       Optionally, prefix and suffix can be `flip`ped.
#' * `switch_notation_byname()` Switches matrix row and/or column names from one type of notation to another 
#'                              based on the `from` and `to` arguments.
#'                              Optionally, prefix and suffix can be `flip`ped.
#' 
#' If `sep` only is specified (default is " -> "), 
#' `pref_start`, `pref_end`, `suff_start`, and `suff_end` are 
#' set appropriately.
#' 
#' None of the strings in a notation vector are considered part of the prefix or suffix.
#' E.g., "a -> b" in arrow notation means that "a" is the prefix and "b" is the suffix.
#'
#' @param sep A string separator between prefix and suffix. Default is " -> ".
#' @param pref_start A string indicating the start of a prefix. Default is `NULL`.
#' @param pref_end A string indicating the end of a prefix. Default is the value of `sep`.
#' @param suff_start A string indicating the start of a suffix. Default is the value of `sep`.
#' @param suff_end A string indicating the end of a suffix. Default is `NULL`.
#' @param x A string or list of strings to be operated upon.
#' @param pref A string or list of strings that are prefixes. Default is `NULL`.
#' @param suff A string of list of strings that are suffixes. Default is `NULL`.
#' @param keep Tells which 
#' @param ps A list of prefixes and suffixes in which each item of the list is itself a list with two items named `pref` and `suff`.
#' @param notation A notation vector generated by one of the `*_notation()` functions, such as
#'                 `notation_vec()`, `arrow_notation()`, or `bracket_notation()`. 
#'                 Default is `arrow_notation()`.
#' @param from The `notation` to switch _away from_.
#' @param to The `notation` to switch _to_.
#' @param flip A boolean that tells whether to also flip the notation. Default is `FALSE`.
#' @param a A matrix or list of matrices whose row and/or column notation is to be changed.
#' @param margin `1` for rows, `2` for columns, or `c(1, 2)` for both rows and columns. Default is `c(1, 2)`.
#'
#' @return For `notation_vec()`, `arrow_notation()`, and `bracket_notation()`, 
#'           a string vector with named items `pref_start`, `pref_end`, `suff_start`, and `suff_end`;
#'         For `split_pref_suff()`, a string list with named items `pref` and `suff`.
#'         For `paste_pref_suff()`, `split_pref_suff()`, and `switch_notation()`, 
#'           a string list in notation format specified by various `notation` arguments, including
#'           `from`, and `to`.
#'         For `keep_pref_suff`, one of the prefix or suffix or a list of prefixes or suffixes.
#'         For `switch_row_col_notation_byname()`, matrices with row and column names with switched notation, 
#'           per arguments.
#'
#' @examples
#' notation_vec()
#' arrow_notation()
#' bracket_notation()
#' split_pref_suff("a -> b", notation = arrow_notation())
#' flip_pref_suff("a [b]", notation = bracket_notation())
#' keep_pref_suff("a -> b", keep = "suff", notation = arrow_notation())
#' switch_notation("a -> b", from = arrow_notation(), to = bracket_notation())
#' switch_notation("a -> b", from = arrow_notation(), to = bracket_notation(), 
#'                 flip = TRUE)
#' m <- matrix(c(1, 2, 
#'               3, 4), nrow = 2, ncol = 2, byrow = TRUE, 
#'             dimnames = list(c("b [a]", "d [c]"), c("f [e]", "h [g]"))) %>% 
#'   setrowtype("Products [Industries]") %>% setcoltype("Industries [Products]")
#' m
#' switch_notation_byname(m, from = bracket_notation(), to = arrow_notation(), 
#'                        flip = TRUE)
#' # Also works for lists.
#' # Note that margin must be specified as a list here.
#' switch_notation_byname(list(m, m), margin = list(c(1, 2)), 
#'                        from = bracket_notation(), 
#'                        to = arrow_notation(), flip = TRUE)
#' @name row-col-notation
NULL


#' @export
#' @rdname row-col-notation
notation_vec <- function(sep = " -> ",
                         pref_start = "", pref_end = "", 
                         suff_start = "", suff_end = "") {
  if (all(nchar(c(pref_start, pref_end, suff_start, suff_end)) == 0) & nchar(sep) != 0) {
    return(c(pref_start = "", pref_end = sep, 
             suff_start = sep, suff_end = ""))
  }
  c(pref_start = pref_start, 
    pref_end = pref_end,
    suff_start = suff_start, 
    suff_end = suff_end)
}


#' @export
#' @rdname row-col-notation
arrow_notation <- function() {
  notation_vec(sep = " -> ")
}


#' @export
#' @rdname row-col-notation
paren_notation <- function(suff_start = " (", suff_end = ")") {
  notation_vec(sep = "",
               pref_start = "", 
               pref_end = suff_start,
               suff_start = suff_start, 
               suff_end = suff_end)
}


#' @export
#' @rdname row-col-notation
bracket_notation <- function(suff_start = " [", suff_end = "]") {
  notation_vec(sep = "",
               pref_start = "", 
               pref_end = suff_start,
               suff_start = suff_start, 
               suff_end = suff_end)
}


#' @export
#' @rdname row-col-notation
split_pref_suff <- function(x, notation = arrow_notation()) {
  # Strip off first pref_start
  no_pref_start <- gsub(pattern = paste0("^", Hmisc::escapeRegex(notation[["pref_start"]])), replacement = "", x = x)
  # Strip off everything from first pref_end to end of string
  pref <- gsub(pattern = paste0(Hmisc::escapeRegex(notation[["pref_end"]]), ".*$"), replacement = "", x = no_pref_start)
  
  # Strip off last suff_end
  no_suff_end <- gsub(pattern = paste0(Hmisc::escapeRegex(notation[["suff_end"]]), "$"), replacement = "", x = x)
  # Strip off everything from start of the string to first suff_start
  ss <- notation[["suff_start"]]
  if (!is.na(notation[["pref_start"]]) & !is.na(notation[["suff_start"]])) {
    if (notation[["pref_start"]] == notation[["suff_start"]]) {
      ss <- paste0(notation[["pref_end"]], notation[["suff_start"]])
    }
  }
  # Split at the first instance of suff_start to get two pieces
  suff <- stringi::stri_split_fixed(str = no_suff_end, pattern = ss, n = 2)
  suff <- lapply(suff, function(s) {
    if (length(s) == 2) {
      # If we got two pieces, choose the second piece.
      return(s[[2]])
    }
    # We got only 1 piece. Return an empty string ("") to indicate a missing suffix
    return("")
  })
  if (length(x) == 1) {
    suff <- unlist(suff)
  }

  # Decide what the outgoing structure is
  if (length(x) > 1) {
    return(purrr::transpose(list(pref = pref, suff = suff)))
  } 
  list(pref = pref, suff = suff)
}


#' @export
#' @rdname row-col-notation
paste_pref_suff <- function(ps = list(pref = pref, suff = suff), pref = NULL, suff = NULL, notation = arrow_notation()) {
  join_func <- function(ps) {
    out <- paste0(notation[["pref_start"]], ps[["pref"]], notation[["pref_end"]])
    if (notation[["pref_end"]] != notation[["suff_start"]]) {
      out <- paste0(out, notation[["suff_start"]])
    }
    paste0(out, ps[["suff"]], notation[["suff_end"]])
  }
  if (!is.null(names(ps))){
    if (length(ps) == 2 & all(names(ps) == c("pref", "suff"))) {
      # We have a single list of the form list(pref = xxxx, suff = yyyy)
      return(join_func(ps))
    }
  }
  lapply(ps, FUN = join_func)
}


#' @export
#' @rdname row-col-notation
flip_pref_suff <- function(x, notation = arrow_notation()) {
  # Split prefixes and suffixes
  pref_suff <- split_pref_suff(x, notation = notation)
  
  flip_ps_func <- function(ps) {
    # pf is a list with only 2 items, pref and suff.
    out <- paste0(notation[["pref_start"]], ps$suff, notation[["pref_end"]])
    if (notation[["pref_end"]] != notation[["suff_start"]]) {
      out <- paste0(out, notation[["suff_start"]])
    }
    paste0(out, ps$pref, notation[["suff_end"]])
  }
  
  if (length(x) > 1) {
    # pref_suff is a list. So lapply to build a flipped list
    return(lapply(pref_suff, FUN = flip_ps_func))
  } 
  flip_ps_func(pref_suff)
}


#' @export
#' @rdname row-col-notation
keep_pref_suff <- function(x, keep = c("pref", "suff"), notation) {
  keep <- match.arg(keep)
  
  choose_pref_or_suff <- function(ps, which_to_keep) {
    # ps should be an item with "pref" and "suff" named elements.
    if (which_to_keep == "pref" | ps[["suff"]] == "") {
      return(ps[["pref"]])
    }
    return(ps[["suff"]])
  }

  pref_suff <- split_pref_suff(x, notation)
  if (length(x) > 1) {
    out <- lapply(pref_suff, choose_pref_or_suff, keep) %>% 
      unlist()
  } else {
    out <- choose_pref_or_suff(pref_suff, keep)
  }
  return(out)
}


#' @export
#' @rdname row-col-notation
switch_notation <- function(x, from, to, flip = FALSE) {
  switch_func <- function(x) {
    ps <- split_pref_suff(x, notation = from)
    if (ps$suff == "") {
      # No split occurred, meaning the notation for prefix and suffix wasn't found.
      # In this case, return the string unmodified.
      return(x)
    }
    if (flip) {
      ps <- list(pref = ps$suff, suff = ps$pref)
    }
    paste_pref_suff(ps, notation = to)
  }
  
  if (length(x) > 1) {
    return(lapply(x, FUN = switch_func))
  }
  switch_func(x)
}


#' @export
#' @rdname row-col-notation
switch_notation_byname <- function(a, margin = c(1, 2), from, to, flip = FALSE) {
  margin <- prep_vector_arg(a, margin)
  from <- prep_vector_arg(a, from)
  to <- prep_vector_arg(a, to)
  flip <- prep_vector_arg(a, flip)
  switch_func <- function(a_mat, margin, from, to, flip) {
    # When we get here, we should have a single matrix a_mat.
    assertthat::assert_that(all(margin %in% c(1, 2)), msg = paste0("In switch_notation_byname, margin must be 1, 2, or both. ", 
                                                                   "Found margin = ", paste(margin, collapse = ", ")))
    
    out <- a_mat
    if (2 %in% margin) {
      # Transpose the matrices
      transposed <- matsbyname::transpose_byname(out)
      # re-call with margin = 1 to change from arrow to paren notation on the rows (which are really columns)
      switched <- switch_notation_byname(transposed, margin = 1, from = from, to = to, flip = flip)
      # Transpose
      out <- transpose_byname(switched)
    }
    if (1 %in% margin) {
      # Get the row names
      old_rownames <- getrownames_byname(out)
      # call func on old row names to create new row names
      new_rownames <- switch_notation(old_rownames, from = from, to = to, flip = flip)
      # Set row names to the new row names
      out <- setrownames_byname(out, new_rownames)
      # Perform the same transformation on the row type, but only if we had a rowtype in a_mat
      if (!is.null(rowtype(out))) {
        old_rowtype <- rowtype(out)
        new_rowtype <- switch_notation(old_rowtype, 
                                       from = from, to = to, flip = flip)
        out <- out %>% setrowtype(new_rowtype)
      }
    }
    # Return the result
    return(out)
  }

  unaryapply_byname(switch_func, a, 
                    .FUNdots = list(margin = margin, from = from, to = to, flip = flip), 
                    # We control row and column types in this function, so 
                    # prevent unaryapply_byname from setting them.
                    rowcoltypes = "none")
}
