% Generated by roxygen2 (4.0.2): do not edit by hand
\name{inner.GSCA}
\alias{inner.GSCA}
\title{GSCA inner estimation}
\usage{
inner.GSCA(S, W, inner.mod, ...)
}
\arguments{
\item{S}{Covariance matrix of the data.}

\item{W}{Weight matrix, where the indicators are on colums and composites are on the rows.}

\item{inner.mod}{A square matrix specifying the relationships of the composites in the model.}

\item{...}{Other parameters are ignored}
}
\value{
A matrix of unscaled inner weights \code{E} with the same dimesions as \code{inner.mod}.
}
\description{
First step of the GSCA algorithm
}
\details{
The two step GSCA algorithm is designed to minimize.

\code{SS(ZV-ZWA)}

The parameter matrix \code{A} contains all model parameters including
\code{inner}, reflective \code{inner}, and \code{formative}. The weight
matrices \code{V} and \code{W}, which can contain duplicate elements,
contain the indicator weights.

The first step of GSCA estimation method is calculation of regressions
coefficients \code{A} given weights \code{W} and \code{V}. The function
\code{\link{inner.GSCA}} update the part of \code{A} corresponding to
regressions between the composites, corresponding to \code{E} matrix in
matrixpls. The regressions between composites and indicators are estimated
in \code{\link{outer.GSCA}}.

This algorithm for estimating the relationships between the composites
is therefore identical to the PLS path weighting scheme with
the exception that correlations are not used for inverse relationships and
there is no falling back to identity scheme for composites that are not
connected to other composites
The second step of GSCA is calculating a new set of weights conditional on
the regression coeffcients \code{A} to minimize the sum of error terms in
the regressions. This step is implemented in \code{\link{outer.GSCA}} after
updating the regresions between indicators and composites.
The implementation of GSCA in matrixpls differs from the Hwang & Takane (2004)
version in that during the first step, only regressions between composites are
estimated. The regressions between composites and indicators are estimated by
the second stage
the indicators and compositess. Since these covariances need to be calculated in the second step, it is more
efficient to not calculate them during the first step.

The matrixpls implementation of the GSCA algorithm extends the criterion
presented by Hwang and Takane (2004) by including also the minimization of the
residual variances of the formative part of the model. The formative
regressions in a model are typically specified to be identical to the
weight pattern \code{W.mod} resulting zero residual variances by definition.
However, it is possible to specify a formative model that does not completely
overlap with the weight pattern leading to non-zero residuals that can be
optimized.
}
\examples{
# Run the example from ASGSCA package using GSCA estimation

 if(require(ASGSCA)) {
  
  # Run the GSCA example from the ASGSCA package
  
  #Scenario (g) in Romdhani et al. (2014): 4 SNPs mapped to 2 genes and 4 
  #traits involved in 2 clinical pathways 
  #In total: 8 observed variables and 4 latent variables.
  #One of the traits is involved in both clinical pathways.
  #One gene is connected to one of the clinical pathways and
  #the other to both of them.
  
  data(GenPhen)
  W0 <- matrix(c(rep(1,2),rep(0,8),rep(1,2),rep(0,8),rep(1,3),rep(0,7),rep(1,2)),nrow=8,ncol=4)
  B0 <- matrix(c(rep(0,8),rep(1,2),rep(0,3),1,rep(0,2)),nrow=4,ncol=4)
  
  #Estimation only
  GSCA.res <- GSCA(GenPhen,W0, B0,estim=TRUE,path.test=FALSE, 
                   latent.names=c("Gene1","Gene2","Clinical pathway 1","Clinical pathway 2"))
  
  # Setup matrixpls to estimate the same model. Note GSCA places dependent variables
  # on columns but matrixpls uses rows for dependent variables
  
  inner <- t(B0)
  reflective <- matrix(0,8,4)
  formative <- t(W0)
  
  colnames(formative) <- rownames(reflective) <- names(GenPhen)
  colnames(inner) <- rownames(inner) <- rownames(formative) <- colnames(reflective) <-
    c("Gene1","Gene2","Clinical pathway 1","Clinical pathway 2")
  
  model <- list(inner = inner, 
                reflective = reflective,
                formative = formative)
    
  # Estimate using alternating least squares
  
  matrixpls.res1 <- matrixpls(cov(GenPhen),  model,
                              outerEstimators = outer.GSCA,
                              innerEstimator = inner.GSCA,
                              tol = 0.000000000001)

  print("Comparing alternating least squares estimates to original estimats")
  # Compare thw weights
  print(GSCA.res$Weight)
  print(t(attr(matrixpls.res1,"W")))

  print(GSCA.res$Weight - t(attr(matrixpls.res1,"W")))
  
  
  # Compare the paths
  print(GSCA.res$Path)
  print(t(attr(matrixpls.res1,"beta")))
  
  print(GSCA.res$Path-t(attr(matrixpls.res1,"beta")))
  
  # Estimate using direct minimization of the estimation criterion
  
  matrixpls.res2 <- matrixpls(cov(GenPhen),  model,
                              weightFunction = weight.optim,
                              optimCriterion = optim.GSCA)

  print("Comparing numeric minimization estimates to original estimats")
  
  # Compare thw weights
  print(GSCA.res$Weight)
  print(t(attr(matrixpls.res2,"W")))
  
  print(GSCA.res$Weight - t(attr(matrixpls.res2,"W")))
  
  
  # Compare the paths
  print(GSCA.res$Path)
  print(t(attr(matrixpls.res2,"beta")))
  
  print(GSCA.res$Path-t(attr(matrixpls.res2,"beta")))
  
  
  # Check the criterion function values
  print(optim.GSCA(matrixpls.res1))
  print(optim.GSCA(matrixpls.res2))
  
} else{
  print("This example requires the ASGSCA package")
}

}
\references{
Hwang, H., & Takane, Y. (2004). Generalized structured component analysis. Psychometrika, 69(1), 81–99. doi:10.1007/BF02295841
}
\seealso{
Other GSCA functions: \code{\link{optim.GSCA}};
  \code{\link{outer.GSCA}}

Other inner estimators: \code{\link{inner.Horst}};
  \code{\link{inner.centroid}}; \code{\link{inner.factor}};
  \code{\link{inner.identity}}; \code{\link{inner.path}}
}

