#' Quick parameter update
#'
#' @param x model object
#' @param p a named list/vector of parameters to update
#'
#' @return model object
#'
#' @details
#' Faster update of parameters inside the model object. Useful in the context of parameter optimization, otherwise consider the regular `param()` because speed comes at the cost of safety.
#'
#' @examples
#' library(mrgsolve)
#' ho <- house()
#' \dontrun{
#' system.time(replicate(1000, param(ho, c(CL = .1, VC = 1))))
#' system.time(replicate(1000, qparam(ho, c(CL = .1, VC = 1))))
#' }
qparam <- function(x, p){
  x@param@data[names(p)] <- as.double(p)
  return(x)
}

#' Compute the H matrix
#'
#' @description Partial derivative of predictions with respect to epsilon
#'
#' @param pred predictions (typically obtained from `f()`)
#' @param cmt compartments predictions belong to (typically the `cmt` column of the dataset)
#' @param all_cmt all possible compartments with observations as defined in the model in `$SIGMA`
#' @param ... for compatibility (not used)
#'
#' @return a matrix of dimensions `[length(pred), 2 * length(all_cmt)]`
#'
#' @examples
#' mapbayr:::h(
#' pred = c(400, 40, 200, 20),
#' cmt = c(2, 3, 2, 3),
#' all_cmt = c(2, 3)
#' )
h <- function(pred, cmt, all_cmt, ...){
  pred[pred==0] <- 1 #Avoid division by zero for proportional error
  m <- matrix(nrow = length(pred), ncol = 0)
  for(i in all_cmt){
    add <- as.double(cmt == i)
    m <- cbind(m, pred*add, add)
  }
  return(unname(m))
}

f <- function(qmod, data){
  output <- mrgsim_q(x = qmod, data = data, output = "df")
  output$DV[data[,"mdv"]==0]
}

ofv_kang <- function(obs, pred, eta, var, omega_inv){
  eta <- unlist(eta)
  sum(log(var) + (obs - pred)^2/var) + diag(matrix(eta, nrow = 1) %*% omega_inv %*% matrix(eta, ncol = 1))
}

#' Compute the objective function value
#' @param eta a named vector/list of parameters
#' @param qmod,sigma,log_transformation,omega_inv,all_cmt generated by \code{\link{preprocess.ofv.fix}}
#' @param idvaliddata,idDV,idcmt generated by \code{\link{preprocess.ofv.id}}
#' @param argofv above mentioned arguments as a list
#' @param ... for compatibility (not used)
#'
#' @details
#' This function is called iteratively by the optimization function. Arguments should not be passed directly, but generated by the pre-processing functions (see \code{\link{preprocess.ofv}}).
#'
#' @return a single numeric value (the objective function value)
#' @export
compute_ofv <- function(eta, qmod, sigma, omega_inv, all_cmt, log_transformation, idvaliddata, idDV, idcmt, ...){
  #Update ETA values
  qmod <- qparam(x = qmod, p = eta)

  #Predict concentrations
  pred <- f(qmod = qmod, data = idvaliddata)
  if(log_transformation) pred <- log(pred)

  #Compute variance associated to predictions
  H <- h(pred = pred, cmt = idcmt, all_cmt = all_cmt)
  g2 <- diag(H %*% sigma %*% t(H))

  #Compute the objective function value per se
  ofv_kang(obs = idDV, pred = pred, eta = eta, var = g2, omega_inv = omega_inv)
}

#' @rdname compute_ofv
#' @export
do_compute_ofv <- function(eta, argofv, ...){
  dots <- list(...)
  args <- c(list(eta = eta), argofv, dots)
  do.call(compute_ofv, args)
}
