% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prepare_network_data.R
\name{prepare_network_data}
\alias{prepare_network_data}
\title{Prepare Network Data for LPANDA}
\usage{
prepare_network_data(df, input_variable_map = list(), verbose = TRUE, ...)
}
\arguments{
\item{df}{A \link[base]{data.frame} containing data from elections, with one
row per candidate. The function also accepts a single election,
though diachronic outputs will then be empty or trivial. See the
\emph{Expected structure of input data} section for the expected data
format and required variables.}

\item{input_variable_map}{A \link[base]{list} mapping variable names in \code{df}
that differ from the expected ones:\cr \cr
\code{elections}    = \strong{unique} election identifiers
(\link[base]{numeric}),\cr
\code{candidate}    = candidate's name used as a \strong{unique}
identifier (\link[base]{character}),\cr
\code{list_name}    = name of the candidate list
(\link[base]{character}),\cr
\code{list_pos}     = candidate's position on the list
(\link[base]{numeric}),\cr
\code{pref_votes}   = preferential votes received by the
candidate (\link[base]{numeric}),\cr
\code{list_votes}   = * total votes received by the candidate
list (\link[base]{numeric}),\cr
\code{elected}      = whether the candidate was elected
(\link[base]{logical}),\cr
\code{nom_party}    = party that nominated the candidate
(\link[base]{character}),\cr
\code{pol_affil}    = declared political affiliation of the candidate
(\link[base]{character}),\cr
\code{mayor}        = whether the councillor became mayor
(\link[base]{logical}),\cr
\code{dep_mayor}    = whether the councillor became deputy mayor
(\link[base]{logical}),\cr
\code{board}        = whether the councillor became a member
of the executive board (\link[base]{logical}),\cr
\code{gov_support}  = whether the councillor supported
the executive body (\link[base]{logical}),\cr
\code{elig_voters}  = * number of eligible voters
(\link[base]{numeric}),\cr
\code{ballots_cast} = * number of ballots cast
(\link[base]{numeric}),\cr
\code{const_size}   = * size of the constituency (number of seats)
(\link[base]{numeric})\cr \cr
* Variables marked with an asterisk should appear
only once per election and constituency — in the
row of any \strong{one} candidate running in that specific
elections and constituency.\cr \cr
See the \emph{Expected input data structure} section
to find out how to use it.}

\item{verbose}{Logical, default TRUE. If FALSE, suppresses informative messages.}

\item{...}{Optional arguments reserved for internal development, experimental
features and future extensions, such as \code{include_cores} (logical,
default FALSE). Not intended for standard use yet (behavior may
change without notice). Unknown keys in ... are ignored.}
}
\value{
A \link[base]{list} of network data objects for diachronic analysis
using LPANDA or other social network analysis tools. Each component contains
\code{edgelist} (data.frame of edges) and \code{node_attr} (data.frame of node
attributes). The exact set of columns depends on the input and may evolve. See
\emph{Output data structure} for a description of the returned object.
}
\description{
Transforms time series data of local election results into a set of network
data for use in Local Political Actor Network Diachronic Analysis (LPANDA).
The function constructs a bipartite network (candidate – candidate list), its
projected one-mode networks (candidate – candidate and list – list), a continuity
graph (linking candidate lists between adjacent elections), and an elections
network (its node attributes can serve as electoral statistics). It also
detects parties (as clusters of candidate lists based on community detection
applied to the bipartite network) and constructs their network.
}
\note{
Cores are currently experimental, as the conversion of their definition into
code is still being sought, and may be subject to change in future versions.
It is not yet intended for standard use in analyses and academic studies, since
their calculation is not yet comprehensive, so the cores' network structure
will not appear in the standard output network data unless explicitly called
with the \code{include_cores = TRUE} parameter. Use with caution, their interpretation
is highly questionable.
}
\section{Expected structure of input data}{

The input data frame (\code{df}) \strong{must} include at least the election identifiers
(year[.month]), candidates' names (uniquely identifying individuals), and
list names. Other variables are optional. If variable names in the dataset
differ from the expected ones, they should be specified in the \code{input_variable_map}
as a named \link[base]{list} (only differing names need to be listed).

\emph{Just in case - a named list is a list where each element has a name (the
expected variable name) and a value (the actual name used in your data frame),
for example: \code{list(list_name = "party", elected = "seat", list_votes = "votes_total")}.}

Examples of expected and acceptable values in \code{df}:
\itemize{
\item \code{elections} (required): Election identifier in the format YY[YY][.MM]:
e.g., 94 | 02 | 1998 | "2024" | 2022.11
\item \code{candidate} (required): e.g., "John Doe" | "John Smith (5)" | "Jane Doe, jr."
\item \code{list_name} (required): \emph{for independent candidates, you can use:}
e.g., "John Smith, Independent Candidate" | "J.S., IND."
\item \code{list_pos}, \code{pref_votes}, \code{list_votes}: must be \link[base]{numeric}
\item \code{elected}, \code{mayor}, \code{dep_mayor}, \code{board}, \code{gov_support}: 1 | "0" | T | "F" | "TRUE" | FALSE
(non‑logical inputs will be coerced to logical).
\item \code{nom_party}: \emph{for independent candidates, you can use:} "IND" | "Independent Candidate"
\item \code{pol_affil}: \emph{for independent candidates, you can use:} "non-partisan"
\item \code{elig_voters}, \code{ballots_cast}, \code{const_size}: A \link[base]{numeric} that
should appear only once in any candidate row within a given election and constituency
}

If \code{pref_votes} are present but \code{list_votes} are not, the function assumes
a voting system where list votes are calculated by summing the preferential
votes of candidates on the list.

If \code{const_size} is missing, it will be estimated based on the number of
elected candidates (if available).

For the purposes of analysis, a new variable \code{list_id} (class \link[base]{character})
is added to the internally processed copy of \code{df} and carried to the output.
It uniquely identifies each candidate list in a given election (combining
\code{list_name} and \code{elections}), e.g., \emph{Besti Flokkurinn (2010)}, \emph{SNP (2019)},
or \emph{"John Smith (5), IND. (2022.11)"}. This variable serves as a \strong{key identifier}
in LPANDA for tracking candidate lists across elections and constructing
network relations.
}

\section{Output data structure}{

The returned object is a named \link[base]{list} with up to seven network
objects:
\itemize{
\item \code{bipartite}: bipartite network (candidates-lists).
\item \code{candidates}: projected candidate–candidate network.
\item \code{lists}: projected list–list network (directed by election order).
\item \code{continuity}: filtered version of \code{lists} network (edges of adjacent elections only).
\item \code{parties}: network of detected party clusters (via community detection applied
on \code{bipartite} network).
\item (\code{cores}): higher-level clusters of \code{parties}. Cores are currently experimental
and will not appear in the standard output network data. See \strong{Note}
below.
\item \code{elections}: inter-election candidate flow and election-level stats
}

Each object is a list with two components:
\itemize{
\item \code{edgelist}: a \link[base]{data.frame} representing network edges
\item \code{node_attr}: a \link[base]{data.frame} with attributes for each node
}

For example, \code{...$candidates$edgelist} contains edges between individuals
who appeared on the same candidate list, and \code{...$elections$node_attr}
includes several election statistics (e.g., number of candidates, distributed
seats, plurality index, voter turnout for each election, etc.).
}

\examples{
data(sample_diff_varnames, package = "lpanda")
df <- sample_diff_varnames
str(df) # different variable names: "party" and "seat"
input_variable_map <- list(list_name = "party", elected = "seat")
\donttest{
netdata <- prepare_network_data(df, input_variable_map, verbose = FALSE)
str(netdata, vec.len = 1)
}

}
