% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bin.R
\name{search.bin}
\alias{search.bin}
\title{Create a Model Set for Binary Choice Models}
\usage{
search.bin(
  data,
  combinations,
  metrics = get.search.metrics(),
  modelChecks = get.search.modelchecks(),
  items = get.search.items(),
  options = get.search.options(),
  costMatrices = NULL,
  searchLogit = TRUE,
  searchProbit = FALSE,
  optimOptions = get.options.newton(),
  aucOptions = get.options.roc()
)
}
\arguments{
\item{data}{A list that determines data and other required information for the search process.
Use \code{\link[=get.data]{get.data()}} function to generate it from a \code{matrix} or a \code{data.frame}.}

\item{combinations}{A list that determines the combinations of the exogenous variables in the search process.
Use \code{\link[=get.combinations]{get.combinations()}} function to define it.}

\item{metrics}{A list of options for measuring performance. Use \link{get.search.metrics} function to get them.}

\item{modelChecks}{A list of options for excluding a subset of the model set. Use \link{get.search.modelchecks} function to get them.}

\item{items}{A list of options for specifying the purpose of the search. Use \link{get.search.items} function to get them.}

\item{options}{A list of extra options for performing the search. Use \link{get.search.options} function to get them.}

\item{costMatrices}{A list of numeric matrices where each one determines how to score the calculated probabilities.
Given the number of choices \code{n}, a frequency cost matrix is an \code{m x n+1} matrix.
The first column determines the thresholds.
Elements in the \code{j}-th column determine the costs corresponding to the \code{j-1}-th choice in \code{y}.
It can be \code{NULL} if it is not selected in \code{metrics}.}

\item{searchLogit}{If \code{TRUE}, logit regressions are added to the model set.}

\item{searchProbit}{If \code{TRUE}, probit regressions are added to the model set.}

\item{optimOptions}{A list for Newton optimization options.
Use \link{get.options.newton} function to get the options.}

\item{aucOptions}{A list for AUC calculation options.
Use \link{get.options.roc} function to get the options.}
}
\value{
A nested list with the following members:
\item{counts}{Information about the expected number of models, number of estimated models, failed estimations, and some details about the failures.}
\item{results}{A data frame with requested information in \code{items} list.}
\item{info}{The arguments and some general information about the search process such as the elapsed time.}

Note that the output does not contain any estimation results, but minimum required data to estimate the models (Use \code{summary()} function to get the estimation).
}
\description{
Use this function to create a binary choice model set and search for the best models (and other information) based on in-sample and out-of-sample evaluation metrics.
}
\examples{
# We simulate some data for this example:
# sample data:
n = 50 # number of observations
num_x_r <- 3L # number of relevant explanatory variables
num_x_ir <- 20 # (relatively large) number of irrelevant explanatory variables
set.seed(340)
sample <- sim.bin(num_x_r, n)
x_ir <- lapply(1:num_x_ir, function(x) rnorm(n))

# prepare data:
data <- data.frame(sample$y, sample$x, x_ir)
colnames(data) <- c("Y", colnames(sample$x), paste0("z", 1:num_x_ir))

# Use glm function to estimate and analyse:
fit <- glm(Y ~ . - Y, data = data, family = binomial())
summary(fit)

# You can also use this package estimation function:
data0 <- get.data(data,
                equations = list(Y ~ . - Y),
                addIntercept = FALSE)
fit <- estim.bin(data = data0)
# format and print coefficients:
print(fit)

# Alternatively, You can define a binary choice model set:
x_sizes = c(1:3) # assuming we know the number of relevant explanatory variables is less than 3
metric_options <- get.search.metrics(typesIn = c("sic")) # We use SIC for searching
search_res <- search.bin(data = data0,
                         combinations = get.combinations(sizes = x_sizes),
                         metrics = metric_options)
print(search_res)

# Use summary function to estimate the best model:
search_sum <- summary(search_res, y = sample$y, x = data[,3:ncol(data)])

# format and print coefficients:
s_fit <- summary(search_res)
print(s_fit$results[[1]]$value)

# Try a step-wise search for creating a larger model set:
search_res <- search.bin(data = data0,
                         combinations = get.combinations(
                           sizes = list(c(1, 2, 3), c(4)),
                           stepsNumVariables = c(NA, 7)),
                         metrics = metric_options)
# combinations argument is different

print(search_res)
# Use summary like before.

}
\seealso{
\link{estim.bin}
}
