% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{fit_model}
\alias{fit_model}
\title{Fitting kDGLM models}
\usage{
fit_model(
  ...,
  smooth = TRUE,
  p.monit = NA,
  condition = "TRUE",
  metric = "log.like",
  lag = 1,
  pred.cred = 0.95,
  metric.cutoff = NA,
  save.models = FALSE,
  silent = FALSE
)
}
\arguments{
\item{...}{dlm_block or dlm_distr objects or named values: The structural blocks of the model (dlm_block objects), alongside the model outcomes (dlm_distr object). If at least one block is undefined, the user must also provide its value in this argument (see last example).}

\item{smooth}{boolean: A flag indicating if the smoothed distribution for the latent states should be calculated.}

\item{p.monit}{numeric (optional): The prior probability of changes in the latent space variables that are not part of its dynamic. Only used when performing sensitivity analysis.}

\item{condition}{character: A character defining which combinations of undefined hyper parameter should be tested. See example for details. Only used when performing sensitivity analysis.}

\item{metric}{character: The name of the metric to use for model selection. One of log-likelihood for the one-step-ahead prediction ("log.like"), Mean Absolute Scaled Error ("mase") \insertCite{mase}{kDGLM} or Interval Score ("interval.score") \insertCite{interval_score}{kDGLM}.  Only used when performing sensitivity analysis.}

\item{lag}{integer: The number of steps ahead used for the prediction when calculating the metrics. If lag<0, predictions are made using the smoothed distribution of the latent states.  Only used when performing sensitivity analysis.}

\item{pred.cred}{numeric: A number between 0 and 1 (not included) indicating the credibility interval for predictions. If not within the valid range of values, 0.95 will be used. Only used when performing sensitivity analysis.}

\item{metric.cutoff}{integer: The number of observations to ignore when calculating the metrics. Default is 1/10 of the number of observations (rounded down). Only used when performing sensitivity analysis.}

\item{save.models}{boolean: A flag indicating if all evaluated models should be saved. If FALSE, only the best model (according to the chosen metric) will be saved. Only used when performing sensitivity analysis.}

\item{silent}{boolean: A flag indicating if a progress bar should be printed. Only used when performing sensitivity analysis.}
}
\value{
A fitted_dlm object.
}
\description{
Fit a model given its structure and the observed data. This function can be used for any supported family (see vignette).
}
\details{
This is the main function of the kDGLM package, as it is used to fit all models.

For the details about the implementation see  \insertCite{ArtigoPacote;textual}{kDGLM}.

For the details about the methodology see  \insertCite{ArtigokParametrico;textual}{kDGLM}.

For the details about the Dynamic Linear Models see  \insertCite{WestHarr-DLM;textual}{kDGLM} and \insertCite{Petris-DLM;textual}{kDGLM}.
}
\examples{

# Poisson case
data <- c(AirPassengers)

level <- polynomial_block(rate = 1, order = 2, D = 0.95)
season <- harmonic_block(rate = 1, order = 2, period = 12, D = 0.975)

outcome <- Poisson(lambda = "rate", data = data)

fitted.data <- fit_model(level, season,
  AirPassengers = outcome
)
summary(fitted.data)

plot(fitted.data, plot.pkg = "base")

##################################################################

# Multinomial case
structure <- (
  polynomial_block(p = 1, order = 2, D = 0.95) +
    harmonic_block(p = 1, period = 12, D = 0.975) +
    noise_block(p = 1, R1 = 0.1) +
    regression_block(p = chickenPox$date >= as.Date("2013-09-01"))
  # Vaccine was introduced in September of 2013
) * 4

outcome <- Multinom(p = structure$pred.names, data = chickenPox[, c(2, 3, 4, 6, 5)])
fitted.data <- fit_model(structure, chickenPox = outcome)
summary(fitted.data)
plot(fitted.data, plot.pkg = "base")

##################################################################

# Univariate Normal case
structure <- polynomial_block(mu = 1, D = 0.95) +
  polynomial_block(V = 1, D = 0.95)

outcome <- Normal(mu = "mu", V = "V", data = cornWheat$corn.log.return[1:500])
fitted.data <- fit_model(structure, corn = outcome)
summary(fitted.data)
plot(fitted.data, plot.pkg = "base")

##################################################################

# Bivariate Normal case
structure <- (polynomial_block(mu = 1, D = 0.95) +
  polynomial_block(V = 1, D = 0.95)) * 2 +
  polynomial_block(rho = 1, D = 0.95)

outcome <- Normal(
  mu = c("mu.1", "mu.2"),
  V = matrix(c("V.1", "rho", "rho", "V.2"), 2, 2),
  data = cornWheat[1:500, c(4, 5)]
)
fitted.data <- fit_model(structure, cornWheat = outcome)
summary(fitted.data)
plot(fitted.data, plot.pkg = "base")

##################################################################

# Gamma case
structure <- polynomial_block(mu = 1, D = 0.95)

Y <- (cornWheat$corn.log.return[1:500] - mean(cornWheat$corn.log.return[1:500]))**2
outcome <- Gamma(phi = 0.5, mu = "mu", data = Y)
fitted.data <- fit_model(structure, corn = outcome)
summary(fitted.data)
plot(fitted.data, plot.pkg = "base")

##################################################################
\donttest{
# Sensitivity analysis
data <- c(AirPassengers)

level <- polynomial_block(rate = 1, order = 2, D = "D.level")
season <- harmonic_block(rate = "sazo.effect", order = 2, period = 12, D = "D.sazo")

outcome <- Poisson(lambda = "rate", data = data)

fit_model(level, season, outcome,
  sazo.effect = c(0, 1),
  D.level = c(seq.int(0.8, 1, l = 11)),
  D.sazo = c(seq.int(0.95, 1, l = 11)),
  condition = "sazo.effect==1 | D.sazo==1"
)
}

}
\seealso{
auxiliary functions for creating outcomes \code{\link{Poisson}}, \code{\link{Multinom}}, \code{\link{Normal}}, \code{\link{Gamma}}

auxiliary functions for creating structural blocks \code{\link{polynomial_block}}, \code{\link{regression_block}}, \code{\link{harmonic_block}}, \code{\link{TF_block}}

auxiliary functions for defining priors \code{\link{zero_sum_prior}}, \code{\link{CAR_prior}}

Other auxiliary functions for fitted_dlm objects: 
\code{\link{coef.fitted_dlm}()},
\code{\link{eval_dlm_norm_const}()},
\code{\link{forecast.fitted_dlm}()},
\code{\link{kdglm}()},
\code{\link{simulate.fitted_dlm}()},
\code{\link{smoothing}()},
\code{\link{update.fitted_dlm}()}
}
\concept{auxiliary functions for fitted_dlm objects}
