% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel_smoothing.R
\name{ks}
\alias{ks}
\title{Item Characteristic Curve Estimation using Kernel Smoothing}
\usage{
ks(
  resp,
  h = NULL,
  kernel_func = "gauss",
  criterion = NULL,
  points = seq(-3, 3, 0.05)
)
}
\arguments{
\item{resp}{A response matrix where each row is the responses of an examinee
  and each column represents an item.

  \code{resp} does not necessarily be a matrix. It can be \code{data.frame}
  or any other object that can be convertible to matrix using
  \code{as.matrix} function.

  \code{resp} can contain missing responses.}

\item{h}{The bandwidth parameter that controls the amount of smoothing.
  A small value will decrease the bias whereas increase the sampling
  variability. For a standard normally distributed \code{criterion} and
  Gaussian kernel smoothing function,
  \eqn{h = 0.2} is recommended for large sample sizes (like 3000),
  \eqn{h = 0.3} is recommended for medium sample sizes (like 500), and
  \eqn{h = 0.4} is recommended for small sample sizes (like 100), and

  The default value is \eqn{1.06  \sigma(criterion) n_{examinees}^{-1/5}}.}

\item{kernel_func}{Choice of kernel function. Possible choices are:

  \itemize{
    \item{\code{"gauss"}}{Gaussian kernel. \eqn{f(x) = e^{-u^2/2}}.}
    \item{\code{"unif"}}{Uniform kernel. \eqn{f(x) = 0.5, |u| < 0.5}, else 0.}
    \item{\code{"quadratic"}}{Quadratic kernel.
      \eqn{f(x) = 0.75(1-u^2), |u| < 1}, else 0.}
    \item{Custom Function}{You can provide a custom kernel function object.
      The function should be maximum at \eqn{u = 0} and gets closer to 0
      on either side.}
  }

  The default value is \code{"gauss"}, i.e. Gaussian kernel function.}

\item{criterion}{The ability estimates for each examinee. The default is
\code{NULL} where the abilities will be estimated from the sum scores.
First sum scores will be calculated, then the rank of each examinee's
sum score will be calculated. These ranks will be divided by the number
of examinees plus 1 in order to get values between 0 and 1. Finally, these
values will be put on standard normal scale (by inverse CDF).}

\item{points}{The points at which the item characteristic curve will be
calculated. The default value is \code{points = seq(-3, 3, 0.05)}.}
}
\value{
A \code{list} with following elements will be returned:
  \itemize{
    \item{\code{points}}{The quadrature points at which ICC is calculated.}
    \item{\code{icc}}{A matrix where each cell represents probability of
      selecting a response (for dichotomous models, probability of correct
      response). Items are on columns and quadrature points are on rows.}
    \item{\code{se}}{A matrix of standard errors of each point of
      \code{icc}. This matrix has the same dimension as \code{icc}.}
    \item{\code{criterion}}{The criterion values used for examinees. If
      \code{criterion = NULL} these numbers will be based on sum scores.}
    \item{\code{h}}{The bandwidth parameter.}
  }
}
\description{
Item Characteristic Curve Estimation using Kernel Smoothing
}
\examples{
ip <- generate_ip(model = "3PL", n = 50)
true_theta <- rnorm(10000)
resp <- sim_resp(ip = ip, theta = true_theta, prop_missing = 0.3)

kern_output <- ks(resp)

# Plot ICC
i <- 12 # select an item to plot
x <- kern_output$icc[, i]
se <- kern_output$se[, i]
p <- prob(ip = ip[i], theta = kern_output$points)
p <- sapply(p, `[`, 2) # get the probability of correct responses

graph_data <- data.frame(
  theta = kern_output$points,
  icc = x,
  ci_low = sapply(x - qnorm(.975) * se, function(x) max(x, 0)),
  ci_high = sapply(x + qnorm(.975) * se, function(x) min(x, 1)),
  p = p)

\dontrun{
p <- ggplot(data = graph_data) +
  geom_line(aes(x = theta, y = icc), color = "blue", alpha = .7, size = 1) +
  geom_line(aes(x = theta, y = p), color = "red", size = 1, alpha = .7) +
  geom_ribbon(data = graph_data,
              aes(x = theta, ymin = ci_low, ymax = ci_high),
              alpha = .25) +
  ylim(0, 1) +
  labs(x = "Theta", y = "Probability",
       title = "Item Characteristic Curve") +
  theme_bw()

p
}

}
\author{
Emre Gonulates
}
