#' Process a TAF repo with the taf roclet
#'
#' This function builds documentation for a TAF repository
#' using roxygen syntax headers.  It depnends on the roxygen2
#' package adding some extra functionality to produce citation
#' entries for data sources
#'
#' @param path location of taf repository top level directory. Default
#'   is working directory.
#' @param files a vector of file names to parse for documentation.
#'
#' @return NULL
#'
#' @examples
#' \dontrun{
#'
#' # Create bootstrap folder
#' mkdir("bootstrap")
#'
#' # Create bootstrap script, bootstrap/mydata.R
#' draft.data.script(name="mydata", title="Title", description="Description",
#'                   format="txt", originator="Me", year="2022",
#'                   period=c(2000,2020), access="Public",
#'                   content='write(pi, file="pi.txt")')
#'
#' # Create metadata, bootstrap/DATA.bib
#' taf.roxygenise(files="mydata.R")
#'
#' # Run bootstrap script, creating bootstrap/data/mydata/pi.txt
#' taf.bootstrap()
#' }
#'
#' @export
#' @importFrom purrr flatten

taf.roxygenise <- function(path = ".", files) {

  base_path <- normalizePath(path, mustWork = TRUE)
  bootstrap <- normalizePath(file.path(base_path, "bootstrap"))

  if (missing(files)) {
    files <- dir(bootstrap, pattern = "\\.R$")
  }
  files <- normalizePath(file.path(bootstrap, files))

  list_of_blocks <- lapply(files, roxygen2::parse_file, env = NULL)

  blocks <- flatten(list_of_blocks)

  ## order blocks ?...

  results <- roxygen2::roclet_process(taf_roclet(), blocks)
  roxygen2::roclet_output(taf_roclet(), results, base_path)
}


# internal functions required for taf.roxygenise
# taf tags
#' @rdname icesTAF-internal
#' @export
#' @importFrom roxygen2 roxy_tag_parse
roxy_tag_parse.roxy_tag_tafKey <- function(x) {
  roxygen2::tag_markdown(x)
}
#' @rdname icesTAF-internal
#' @export
roxy_tag_parse.roxy_tag_tafOriginator <- function(x) {
  roxygen2::tag_markdown(x)
}
#' @rdname icesTAF-internal
#' @export
roxy_tag_parse.roxy_tag_tafYear <- function(x) {
  roxygen2::tag_markdown(x)
}
#' @rdname icesTAF-internal
#' @export
roxy_tag_parse.roxy_tag_tafPeriod <- function(x) {
  roxygen2::tag_markdown(x)
}
#' @rdname icesTAF-internal
#' @export
roxy_tag_parse.roxy_tag_tafAccess <- function(x) {
  # could contain logic for testing values
  roxygen2::tag_markdown(x)
}
#' @rdname icesTAF-internal
#' @export
roxy_tag_parse.roxy_tag_tafSource <- function(x) {
  roxygen2::tag_markdown(x)
}

## taf_roclet
taf_roclet <- function() {
  roxygen2::roclet("taf")
}

roclet_process.roclet_taf <- function(x, blocks, env, base_path) {
  tags <-
    lapply(
      blocks,
      function(block) {
        # read in all tags
        tag.names <-
          c(
            "title", "name",
            "tafOriginator", "tafYear", "tafPeriod",
            "tafAccess",
            "tafSource"
          )
        tags <-
          sapply(
            tag.names,
            function(x) {
              tag <- roxygen2::block_get_tag(block, x)
              if (is.null(tag$val)) NA else tag$val
            },
            simplify = FALSE
          )
        names(tags) <- gsub("taf", "", tolower(tag.names))
        tags <- data.frame(tags)
        tags$file <- basename(block$file)

        tags
      }
    )

  list(metadata = do.call(rbind, tags))
}


roclet_output.roclet_taf <- function(x, results, base_path, ...) {
  dots <- list(...)
  if ("bibfile" %in% names(dots)) {
    path <- dots$bibfile
  } else {
    bootstrap <- normalizePath(file.path(base_path, "bootstrap"))
    bibfile <- "DATA.bib"
    path <- file.path(bootstrap, bibfile)
  }

  # add header?
  header <- "% Generated by icesTAF: do not edit by hand\n\n"
  cat(header, file = path)

  out <-
    paste(
      by(
        results$metadata,
        seq(nrow(results$metadata)),
        list2bibtex,
        simplify = FALSE
      ),
      collapse = "\n"
    )

  cat(
    out,
    file = path,
    append = TRUE
  )

  invisible(NULL)
}

# convert a list to bibtex
list2bibtex <- function(entry) {
  out <-
    paste0(
      "@Misc{", entry$name, ",\n",
      "  originator = {", entry$originator, "},\n",
      "  year       = {", entry$year, "},\n",
      "  title      = {", entry$title, "},\n",
      "  period     = {", entry$period, "},\n",
      "  access     = {", entry$access, "},\n",
      "  source     = {", entry$source, "},\n",
      "}\n"
    )

  class(out) <- "Bibtex"
  out
}

