#' @title Plot calibration curves water levels vs discharges
#'
#' @author P. Chevallier - Sep 2017 - Nov 2020
#'
#' @description
#' Experimental function, which is for instance limited to only two calibration curves on the same plot.
#'
#' The function plot the discharges measurements and the corresponding
#' calibration curves starting from a 'caldis' list generated by the \code{\link{d_exp_discalib}}
#' function.
#'
#' Only the "active" discharge measurements are plotted. The parameter plotdism displays
#' them or not.
#'
#' One can zoom on a subpart of the plot using the limit values on the x and y axis.
#'
#' The savefig (default = FALSE by default) parameter allows to save the result i a  png, jpg or pdf
#' file, according to the extention of fout.
#'
#' @param caldis List of calibration  and discharge measurments to proceed
#' @param plotcalib Plot calibrations TRUE (defautl) / FALSE
#' @param plotdism Plot discharge measurements TRUE (defautl) / FALSE
#' @param title Plot title (default: Title)
#' @param savefig Save plot in a png file TRUE (default) / FALSE
#' @param width Plot width (x 100 pixels) (default = 8)
#' @param height Plot height (x 100 pixels) (default = 6)
#' @param fout Plot file name (default = "plot.png")
#' @param limx Limit x axis TRUE / FALSE (default)
#' @param limy Limit y axis TRUE / FALSE (default)
#' @param xinf Low value for x (default = NA)
#' @param xsup High value for x (default = NA)
#' @param yinf Low value for y (default = NA)
#' @param ysup High value for y (default = NA)
#'
#'
#'



p_discalib <- function (caldis, plotcalib= TRUE, plotdism=TRUE, title="Title", savefig=FALSE, width= 8,
                       height= 6, fout="plot.png", limx =FALSE, limy = FALSE, xinf=NA,
                       xsup=NA, yinf=NA, ysup=NA) {

  calibtab <- dismtab <- H1 <- Q1 <- Calibration <- H2 <- Q2 <- NULL

  #controle
  calibtab <- caldis[[1]]
  dismtab <- caldis[[2]]
  sen <- calibtab$Sensor[1]
  if(is.na(calibtab) && is.na(dismtab))
    return(warning("\nWrong file.\n"))
  if(plotdism == TRUE && is.na(dismtab)){
    warning("\nNo discharge measurements in the calib file.")
    plotdism <- FALSE
  }
  if(limx==TRUE & (is.na(xinf)==TRUE | is.na(xsup)==TRUE))
    return(warning("\nVerify x axis limits.\n"))
  if(limy==TRUE & (is.na(yinf)==TRUE | is.na(ysup)==TRUE))
    return(warning("\nVerify y axis limits.\n"))

  # preparation etalonnages
  d <- NA
  ld <- NA
  if(plotcalib == TRUE){
    d1 <- tibble(calibtab)
    d1 <- dplyr::filter(calibtab, calibtab$Sensor == sen)
    d1 <- data.frame(calibtab$Date,as.numeric(calibtab$H),
                     as.numeric(calibtab$Q),as.numeric(NA),as.numeric(NA),
                     as.numeric(NA),NA,as.numeric(NA))
    colnames(d1) <- c("Date1","H1","Q1","Date2","H2","Q2","Calibration","active")
    limd <- as.character(levels(as.factor(d1$Date1)))
    ld <- length(limd)
    d1$Date1 <- as.numeric(as.POSIXct(d1$Date1, origin ="1970-01-01"))
    limd <- as.numeric(as.POSIXct(limd, origin ="1970-01-01"))
    for (i in 1:nrow(d1)){
      for (j in 1:ld){
        if(d1$Date1[i]==limd[j]) d1$Calibration[i] <- paste0("calib_",j)
      }
    }
    d <- d1
  }

# preparation jaugeages
  if(plotdism==TRUE) {
    d2 <- tibble(dismtab)
    d2 <- filter(dismtab, dismtab$Sensor == sen)
    d2 <- data.frame(NA,as.numeric(NA),as.numeric(NA),dismtab$Date,
                     as.numeric(dismtab$H), as.numeric(dismtab$Q),NA,as.numeric(dismtab$Active))
    colnames(d2) <- c("Date1","H1","Q1","Date2","H2","Q2","Calibration","active")
    d2 <- d2[d2$active==1,]
    d2$Date2 <- as.numeric(as.POSIXct(d2$Date2, origin ="1970-01-01"))
    for (i in 1:nrow(d2)){
      if (is.na(ld)) d2$Calibration[i] <- "no_calib"
      else if (ld==1) d2$Calibration[i] <- "calib_1"
        else {
        for (j in 1:(ld)){
          if (j < ld){
            if(d2$Date2[i] >= limd[j] & d2$Date2[i] < limd[j+1])
             d2$Calibration[i] <- paste0("calib_",j)
          } else {
            if(d2$Date2[i] >= limd[j])
              d2$Calibration[i] <- paste0("calib_",j)
          }
        }
      }
    d <- rbind(d,d2)
    }
  }

# trace etalonnages
  p <- ggplot(d) + geom_line(aes(x=H1, y=Q1, colour=Calibration))
  p <- p + theme(panel.background=element_rect(fill="white", colour="grey30"),
                 panel.grid.major=element_line(colour="grey30"),
                 panel.grid.minor=element_line(colour=NA))
  if(limx==TRUE) p <- p + xlim(xinf,xsup)
  if(limy==TRUE) p <- p + ylim(yinf,ysup)

  p <- p +
    ylab("Q (m3/s)") + xlab("H (cm)") +
    theme (axis.text.x = element_text(size=20),
           axis.text.y = element_text(size=20),
           axis.title.x = element_text(size=20),
           axis.title.y = element_text(size=20),
           plot.title=element_text(face="bold", size=20)) +
    ggtitle(title)
  p <- p+ theme(legend.position="bottom") +
    theme(legend.text=element_text(size =16))


# trace jaugeages
  if(plotdism==TRUE) {
    p <- p + geom_point (aes(x=H2 , y=Q2, colour=Calibration ))
  }

# affichage et sauvegarde
  if (savefig==TRUE) {
    ggsave(fout, width=width, height=height, dpi=100)
    message("Graphique sauve dans le fichier : ", fout, "\n")
  }
  show(p)
  return(p)
}
# FIN
