/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 
#define H5F_FRIEND     

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5FLprivate.h" 
#include "H5Opkg.h"      
#include "H5MFprivate.h" 

static void *H5O__mdci_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags, size_t p_size,
                              const uint8_t *p);
static herr_t H5O__mdci_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                               const void *_mesg);
static void  *H5O__mdci_copy(const void *_mesg, void *_dest);
static size_t H5O__mdci_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__mdci_free(void *mesg);
static herr_t H5O__mdci_delete(H5F_t *f, H5O_t *open_oh, void *_mesg);
static herr_t H5O__mdci_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_MDCI[1] = {{
    H5O_MDCI_MSG_ID,    
    "mdci",             
    sizeof(H5O_mdci_t), 
    0,                  
    H5O__mdci_decode,   
    H5O__mdci_encode,   
    H5O__mdci_copy,     
    H5O__mdci_size,     
    NULL,               
    H5O__mdci_free,     
    H5O__mdci_delete,   
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    H5O__mdci_debug     
}};

#define H5O_MDCI_VERSION_0 0

H5FL_DEFINE_STATIC(H5O_mdci_t);

static void *
H5O__mdci_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                 unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_mdci_t    *mesg      = NULL;           
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_MDCI_VERSION_0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (mesg = (H5O_mdci_t *)H5FL_MALLOC(H5O_mdci_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for metadata cache image message");

    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_addr_decode(f, &p, &(mesg->addr));

    if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_size(f), p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_DECODE_LENGTH(f, p, mesg->size);

    if (mesg->addr >= (HADDR_UNDEF - mesg->size))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "address plus size overflows");
    if (mesg->addr == HADDR_UNDEF)
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "address is undefined");
    if ((mesg->addr + mesg->size) > H5F_get_eoa(f, H5FD_MEM_SUPER))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "address plus size exceeds file eoa");

    
    ret_value = (void *)mesg;

done:
    if (!ret_value && mesg)
        H5FL_FREE(H5O_mdci_t, mesg);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__mdci_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                 const void *_mesg)
{
    const H5O_mdci_t *mesg = (const H5O_mdci_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(mesg);

    
    *p++ = H5O_MDCI_VERSION_0;
    H5F_addr_encode(f, &p, mesg->addr);
    H5F_ENCODE_LENGTH(f, p, mesg->size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__mdci_copy(const void *_mesg, void *_dest)
{
    const H5O_mdci_t *mesg      = (const H5O_mdci_t *)_mesg;
    H5O_mdci_t       *dest      = (H5O_mdci_t *)_dest;
    void             *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_mdci_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *mesg;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__mdci_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void H5_ATTR_UNUSED *_mesg)
{
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = (size_t)(1 +                  
                         H5F_SIZEOF_ADDR(f) + 
                                              
                         H5F_SIZEOF_SIZE(f)); 
                                              

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__mdci_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_mdci_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__mdci_delete(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, void *_mesg)
{
    H5O_mdci_t *mesg = (H5O_mdci_t *)_mesg;
    haddr_t     final_eoa;           
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(mesg);

    
    if (H5_addr_defined(mesg->addr)) {
        
        if (f->shared->closing) {
            
            if (HADDR_UNDEF == (final_eoa = H5FD_get_eoa(f->shared->lf, H5FD_MEM_DEFAULT)))
                HGOTO_ERROR(H5E_CACHE, H5E_CANTGET, FAIL, "unable to get file size");

            assert(H5_addr_eq(final_eoa, mesg->addr + mesg->size));

            if (H5FD_free(f->shared->lf, H5FD_MEM_SUPER, f, mesg->addr, mesg->size) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_CANTFREE, FAIL, "can't free MDC image");
        }
        else if (H5MF_xfree(f, H5FD_MEM_SUPER, mesg->addr, mesg->size) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free file space for cache image block");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__mdci_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_mdci_t *mdci = (const H5O_mdci_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mdci);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Metadata Cache Image Block address:", mdci->addr);

    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth,
            "Metadata Cache Image Block size in bytes:", mdci->size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
