/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5HLprivate.h" 

const unsigned H5O_layout_ver_bounds[] = {
    H5O_LAYOUT_VERSION_1, 
    H5O_LAYOUT_VERSION_3,
          
    H5O_LAYOUT_VERSION_4,     
    H5O_LAYOUT_VERSION_4,     
    H5O_LAYOUT_VERSION_4,     
    H5O_LAYOUT_VERSION_5,     
    H5O_LAYOUT_VERSION_LATEST 
};

herr_t
H5D__layout_set_io_ops(const H5D_t *dataset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dataset);

    
    switch (dataset->shared->layout.type) {
        case H5D_CONTIGUOUS:
            if (dataset->shared->dcpl_cache.efl.nused > 0)
                dataset->shared->layout.ops = H5D_LOPS_EFL;
            else
                dataset->shared->layout.ops = H5D_LOPS_CONTIG;
            break;

        case H5D_CHUNKED:
            dataset->shared->layout.ops = H5D_LOPS_CHUNK;

            
            switch (dataset->shared->layout.u.chunk.idx_type) {
                case H5D_CHUNK_IDX_BTREE:
                    dataset->shared->layout.storage.u.chunk.ops = H5D_COPS_BTREE;
                    break;

                case H5D_CHUNK_IDX_NONE:
                    dataset->shared->layout.storage.u.chunk.ops = H5D_COPS_NONE;
                    break;

                case H5D_CHUNK_IDX_SINGLE:
                    dataset->shared->layout.storage.u.chunk.ops = H5D_COPS_SINGLE;
                    break;

                case H5D_CHUNK_IDX_FARRAY:
                    dataset->shared->layout.storage.u.chunk.ops = H5D_COPS_FARRAY;
                    break;

                case H5D_CHUNK_IDX_EARRAY:
                    dataset->shared->layout.storage.u.chunk.ops = H5D_COPS_EARRAY;
                    break;

                case H5D_CHUNK_IDX_BT2:
                    dataset->shared->layout.storage.u.chunk.ops = H5D_COPS_BT2;
                    break;

                case H5D_CHUNK_IDX_NTYPES:
                default:
                    assert(0 && "Unknown chunk index method!");
                    HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "unknown chunk index method");
            } 
            break;

        case H5D_COMPACT:
            dataset->shared->layout.ops = H5D_LOPS_COMPACT;
            break;

        case H5D_VIRTUAL:
            dataset->shared->layout.ops = H5D_LOPS_VIRTUAL;
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "unknown storage method");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5D__layout_meta_size(const H5F_t *f, const H5O_layout_t *layout, bool include_compact_data)
{
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(layout);

    ret_value = 1 + 
                1;  

    switch (layout->type) {
        case H5D_COMPACT:
            
            
            ret_value += 2;
            if (include_compact_data)
                ret_value += layout->storage.u.compact.size; 
            break;

        case H5D_CONTIGUOUS:
            
            ret_value += H5F_SIZEOF_ADDR(f); 
            ret_value += H5F_SIZEOF_SIZE(f); 
            break;

        case H5D_CHUNKED:
            if (layout->version < H5O_LAYOUT_VERSION_4) {
                
                assert(layout->u.chunk.ndims > 0 && layout->u.chunk.ndims <= H5O_LAYOUT_NDIMS);
                ret_value++;

                
                ret_value += H5F_SIZEOF_ADDR(f); 

                
                ret_value += layout->u.chunk.ndims * 4;
            } 
            else {
                
                ret_value++;

                
                assert(layout->u.chunk.ndims > 0 && layout->u.chunk.ndims <= H5O_LAYOUT_NDIMS);
                ret_value++;

                
                assert(layout->u.chunk.enc_bytes_per_dim > 0 && layout->u.chunk.enc_bytes_per_dim <= 8);
                ret_value++;

                
                ret_value += layout->u.chunk.ndims * (size_t)layout->u.chunk.enc_bytes_per_dim;

                
                ret_value++;

                switch (layout->u.chunk.idx_type) {
                    case H5D_CHUNK_IDX_BTREE:
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, 0,
                                    "v1 B-tree index type found for layout message >v3");

                    case H5D_CHUNK_IDX_NONE:
                        
                        break;

                    case H5D_CHUNK_IDX_SINGLE:
                        
                        if (layout->u.chunk.flags & H5O_LAYOUT_CHUNK_SINGLE_INDEX_WITH_FILTER) {
                            ret_value += H5F_SIZEOF_SIZE(f); 
                            ret_value += 4;                  
                        }                                    
                        break;

                    case H5D_CHUNK_IDX_FARRAY:
                        
                        ret_value += H5D_FARRAY_CREATE_PARAM_SIZE;
                        break;

                    case H5D_CHUNK_IDX_EARRAY:
                        
                        ret_value += H5D_EARRAY_CREATE_PARAM_SIZE;
                        break;

                    case H5D_CHUNK_IDX_BT2:
                        
                        ret_value += H5D_BT2_CREATE_PARAM_SIZE;
                        break;

                    case H5D_CHUNK_IDX_NTYPES:
                    default:
                        HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, 0, "Invalid chunk index type");
                } 

                
                ret_value += H5F_SIZEOF_ADDR(f);
            } 
            break;

        case H5D_VIRTUAL:
            ret_value += H5F_SIZEOF_ADDR(f); 
            ret_value += 4;                  
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_OHDR, H5E_CANTENCODE, 0, "Invalid layout class");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__layout_oh_create(H5F_t *file, H5O_t *oh, H5D_t *dset, hid_t dapl_id)
{
    H5O_layout_t     *layout;                
    const H5O_fill_t *fill_prop;             
    unsigned          layout_mesg_flags;     
    bool              layout_init = false;   
    herr_t            ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(file);
    assert(oh);
    assert(dset);

    
    layout    = &dset->shared->layout;
    fill_prop = &dset->shared->dcpl_cache.fill;

    
    if (layout->type == H5D_CHUNKED) {
        H5O_pline_t *pline; 

        pline = &dset->shared->dcpl_cache.pline;
        if (pline->nused > 0 &&
            H5O_msg_append_oh(file, oh, H5O_PLINE_ID, H5O_MSG_FLAG_CONSTANT, 0, pline) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update filter header message");
    } 

    
    if (dset->shared->layout.ops->init && (dset->shared->layout.ops->init)(file, dset, dapl_id, false) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize layout information");

    
    layout_init = true;

    
    if (fill_prop->alloc_time == H5D_ALLOC_TIME_EARLY)
        if (H5D__alloc_storage(dset, H5D_ALLOC_CREATE, false, NULL) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize storage");

    
    if (dset->shared->dcpl_cache.efl.nused > 0) {
        H5O_efl_t *efl = &dset->shared->dcpl_cache.efl; 
        H5HL_t    *heap;                                
        size_t     heap_size = H5HL_ALIGN(1);
        size_t     u;
        size_t     name_offset;

        
        for (u = 0; u < efl->nused; ++u)
            heap_size += H5HL_ALIGN(strlen(efl->slot[u].name) + 1);

        
        if (H5HL_create(file, heap_size, &efl->heap_addr ) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create EFL file name heap");

        
        if (NULL == (heap = H5HL_protect(file, efl->heap_addr, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTPROTECT, FAIL, "unable to protect EFL file name heap");

        
        if (H5HL_insert(file, heap, (size_t)1, "", &name_offset) < 0) {
            H5HL_unprotect(heap);
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL, "unable to insert file name into heap");
        }

        for (u = 0; u < efl->nused; ++u) {
            
            if (H5HL_insert(file, heap, strlen(efl->slot[u].name) + 1, efl->slot[u].name, &name_offset) < 0) {
                H5HL_unprotect(heap);
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, FAIL, "unable to insert file name into heap");
            }

            
            efl->slot[u].name_offset = name_offset;
        }

        
        if (H5HL_unprotect(heap) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTUNPROTECT, FAIL, "unable to unprotect EFL file name heap");
        heap = NULL;

        
        if (H5O_msg_append_oh(file, oh, H5O_EFL_ID, H5O_MSG_FLAG_CONSTANT, 0, efl) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update external file list message");
    } 

    
    
    
    if (fill_prop->alloc_time == H5D_ALLOC_TIME_EARLY && H5D_COMPACT != layout->type &&
        !dset->shared->dcpl_cache.pline.nused && (0 != H5S_GET_EXTENT_NPOINTS(dset->shared->space)))
        layout_mesg_flags = H5O_MSG_FLAG_CONSTANT;
    else
        layout_mesg_flags = 0;

    
    if (H5D_VIRTUAL == layout->type)
        if (H5D__virtual_store_layout(file, layout) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to store VDS info");

    
    if (H5O_msg_append_oh(file, oh, H5O_LAYOUT_ID, layout_mesg_flags, 0, layout) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update layout");

done:
    
    if (ret_value < 0)
        if (layout_init)
            
            if (dset->shared->layout.ops->dest && (dset->shared->layout.ops->dest)(dset) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to destroy layout info");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D__layout_oh_read(H5D_t *dataset, hid_t dapl_id, H5P_genplist_t *plist)
{
    htri_t msg_exists;                      
    bool   pline_copied          = false;   
    bool   layout_copied_to_dset = false;   
    bool   efl_copied            = false;   
    herr_t ret_value             = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dataset);
    assert(plist);

    
    if ((msg_exists = H5O_msg_exists(&(dataset->oloc), H5O_PLINE_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if message exists");
    if (msg_exists) {
        
        if (NULL == H5O_msg_read(&(dataset->oloc), H5O_PLINE_ID, &dataset->shared->dcpl_cache.pline))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve message");
        pline_copied = true;
        
        if (H5P_set(plist, H5O_CRT_PIPELINE_NAME, &dataset->shared->dcpl_cache.pline) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set pipeline");
    } 

    
    if (NULL == H5O_msg_read(&(dataset->oloc), H5O_LAYOUT_ID, &(dataset->shared->layout)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to read data layout message");
    layout_copied_to_dset = true;

    
    if ((msg_exists = H5O_msg_exists(&(dataset->oloc), H5O_EFL_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if message exists");
    if (msg_exists) {
        
        if (NULL == H5O_msg_read(&(dataset->oloc), H5O_EFL_ID, &dataset->shared->dcpl_cache.efl))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve message");
        efl_copied = true;

        
        if (H5P_set(plist, H5D_CRT_EXT_FILE_LIST_NAME, &dataset->shared->dcpl_cache.efl) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set external file list");

        
        dataset->shared->layout.ops = H5D_LOPS_EFL;
    } 

    
    assert(dataset->shared->layout.ops);

    
    if (dataset->shared->layout.ops->init &&
        (dataset->shared->layout.ops->init)(dataset->oloc.file, dataset, dapl_id, true) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize layout information");

#ifndef NDEBUG
    
    H5O_layout_t error_layout;
    error_layout.type         = H5D_LAYOUT_ERROR;
    error_layout.version      = 0;
    error_layout.ops          = NULL;
    error_layout.storage.type = H5D_LAYOUT_ERROR;

    if (H5P_poke(plist, H5D_CRT_LAYOUT_NAME, &error_layout) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to setup placeholder layout");
#endif

done:
    if (ret_value < 0) {
        if (pline_copied)
            if (H5O_msg_reset(H5O_PLINE_ID, &dataset->shared->dcpl_cache.pline) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset pipeline info");
        if (layout_copied_to_dset)
            if (H5O_msg_reset(H5O_LAYOUT_ID, &dataset->shared->layout) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset layout info");
        if (efl_copied)
            if (H5O_msg_reset(H5O_EFL_ID, &dataset->shared->dcpl_cache.efl) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset efl message");
    }
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__layout_oh_write(const H5D_t *dataset, H5O_t *oh, unsigned update_flags)
{
    htri_t msg_exists;          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dataset);
    assert(oh);

    
    if ((msg_exists = H5O_msg_exists_oh(oh, H5O_LAYOUT_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to check if layout message exists");
    if (msg_exists) {
        
        if (H5O_msg_write_oh(dataset->oloc.file, oh, H5O_LAYOUT_ID, 0, update_flags,
                             &dataset->shared->layout) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to update layout message");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
