/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 
#define H5HF_DEBUGGING  

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5HFpkg.h"     
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

typedef struct {
    FILE    *stream;      
    int      indent;      
    int      fwidth;      
    haddr_t  dblock_addr; 
    hsize_t  dblock_size; 
    uint8_t *marker;      
    size_t   sect_count;  
    size_t   amount_free; 
} H5HF_debug_iter_ud1_t;

typedef struct {
    H5FS_t *fspace; 
    FILE   *stream; 
    int     indent; 
    int     fwidth; 
} H5HF_debug_iter_ud2_t;

static herr_t H5HF__dtable_debug(const H5HF_dtable_t *dtable, FILE *stream, int indent, int fwidth);

herr_t
H5HF_id_print(H5HF_t *fh, const void *_id, FILE *stream, int indent, int fwidth)
{
    const uint8_t *id = (const uint8_t *)_id; 
    uint8_t        id_flags;                  
    hsize_t        obj_off;                   
    size_t         obj_len;                   
    char           id_type;                   
    herr_t         ret_value = SUCCEED;       

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(fh);
    assert(id);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    id_flags = *id;

    
    if ((id_flags & H5HF_ID_VERS_MASK) != H5HF_ID_VERS_CURR)
        HGOTO_ERROR(H5E_HEAP, H5E_VERSION, FAIL, "incorrect heap ID version");

    
    if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_MAN) {
        id_type = 'M';
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_HUGE) {
        id_type = 'H';
    } 
    else if ((id_flags & H5HF_ID_TYPE_MASK) == H5HF_ID_TYPE_TINY) {
        id_type = 'T';
    } 
    else {
        Rfprintf(Rstderr, "%s: Heap ID type not supported yet!\n", __func__);
        HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "heap ID type not supported yet");
    } 

    
    if (H5HF_get_obj_len(fh, id, &obj_len) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve heap ID length");

    
    if (H5HF_get_obj_off(fh, id, &obj_off) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve heap ID length");

    
    Rfprintf(stream, "%*s%-*s (%c, %" PRIuHSIZE " , %llu)\n", indent, "", fwidth,
            "Heap ID info: (type, offset, length)", id_type, obj_off, (unsigned long long)obj_len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__dtable_debug(const H5HF_dtable_t *dtable, FILE *stream, int indent, int fwidth)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dtable);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Doubling table width:", dtable->cparam.width);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "Starting block size:", (unsigned long long)dtable->cparam.start_block_size);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
            "Max. direct block size:", (unsigned long long)dtable->cparam.max_direct_size);
    Rfprintf(stream, "%*s%-*s %u (bits)\n", indent, "", fwidth, "Max. index size:", dtable->cparam.max_index);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Starting # of rows in root indirect block:", dtable->cparam.start_root_rows);

    
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Table's root address:", dtable->table_addr);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Current # of rows in root indirect block:", dtable->curr_root_rows);

    
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Max. # of rows in root indirect block:", dtable->max_root_rows);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Max. # of direct rows in any indirect block:", dtable->max_direct_rows);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "# of bits for IDs in first row:", dtable->first_row_bits);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "# of IDs in first row:", dtable->num_id_first_row);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

void
H5HF_hdr_print(const H5HF_hdr_t *hdr, bool dump_internal, FILE *stream, int indent, int fwidth)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(hdr);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    Rfprintf(stream, "%*sFractal Heap Header...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Heap is:", hdr->man_dtable.curr_root_rows > 0 ? "Indirect" : "Direct");
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Objects stored in 'debugging' format:", hdr->debug_objs ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "'Write once' flag:", hdr->write_once ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "'Huge' object IDs have wrapped:", hdr->huge_ids_wrapped ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Free space in managed blocks:", hdr->total_man_free);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Managed space data block size:", hdr->man_size);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Total managed space allocated:", hdr->man_alloc_size);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Offset of managed space iterator:", hdr->man_iter_off);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Number of managed objects in heap:", hdr->man_nobjs);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Address of free space manager for managed blocks:", hdr->fs_addr);
    Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
            "Max. size of managed object:", (unsigned long)hdr->max_man_size);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "'Huge' object space used:", hdr->huge_size);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Number of 'huge' objects in heap:", hdr->huge_nobjs);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "ID of next 'huge' object:", hdr->huge_next_id);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Address of v2 B-tree for 'huge' objects:", hdr->huge_bt2_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "'Tiny' object space used:", hdr->tiny_size);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Number of 'tiny' objects in heap:", hdr->tiny_nobjs);

    Rfprintf(stream, "%*sManaged Objects Doubling-Table Info...\n", indent, "");
    H5HF__dtable_debug(&hdr->man_dtable, stream, indent + 3, MAX(0, fwidth - 3));

    
    if (hdr->filter_len > 0) {
        Rfprintf(stream, "%*sI/O filter Info...\n", indent, "");
        if (hdr->man_dtable.curr_root_rows == 0) {
            Rfprintf(stream, "%*s%-*s %llu\n", indent + 3, "", MAX(0, fwidth - 3),
                    "Compressed size of root direct block:", (unsigned long long)hdr->pline_root_direct_size);
            Rfprintf(stream, "%*s%-*s %x\n", indent + 3, "", MAX(0, fwidth - 3),
                    "Filter mask for root direct block:", hdr->pline_root_direct_filter_mask);
        } 
        H5O_debug_id(H5O_PLINE_ID, hdr->f, &(hdr->pline), stream, indent + 3, MAX(0, fwidth - 3));
    } 

    
    if (dump_internal) {
        Rfprintf(stream, "%*sFractal Heap Header Internal Information:\n", indent, "");

        
        Rfprintf(stream, "%*s%-*s %x\n", indent + 3, "", MAX(0, fwidth - 3),
                "Root indirect block flags:", hdr->root_iblock_flags);
        Rfprintf(stream, "%*s%-*s %p\n", indent + 3, "", MAX(0, fwidth - 3),
                "Root indirect block pointer:", (void *)hdr->root_iblock);
        if (hdr->root_iblock)
            H5HF_iblock_print(hdr->root_iblock, dump_internal, stream, indent + 3, fwidth);
    } 

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5HF_hdr_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth)
{
    H5HF_hdr_t *hdr       = NULL;    
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    if (NULL == (hdr = H5HF__hdr_protect(f, addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap header");

    
    H5HF_hdr_print(hdr, false, stream, indent, fwidth);

done:
    if (hdr && H5AC_unprotect(f, H5AC_FHEAP_HDR, addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_PROTECT, FAIL, "unable to release fractal heap header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF_dblock_debug_cb(H5FS_section_info_t *_sect, void *_udata)
{
    H5HF_free_section_t   *sect  = (H5HF_free_section_t *)_sect;    
    H5HF_debug_iter_ud1_t *udata = (H5HF_debug_iter_ud1_t *)_udata; 
    haddr_t                sect_start, sect_end;     
    haddr_t                dblock_start, dblock_end; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(sect);
    assert(udata);

    
    sect_start = sect->sect_info.addr;
    sect_end   = (sect->sect_info.addr + sect->sect_info.size) - 1;
    assert(sect_end >= sect_start);
    dblock_start = udata->dblock_addr;
    dblock_end   = (udata->dblock_addr + udata->dblock_size) - 1;
    assert(dblock_end >= dblock_start);

    
    if ((sect_start <= dblock_end &&
         sect_end >= dblock_start) || 
        (sect_start <= dblock_end && sect_end >= dblock_end)) { 
        char   temp_str[32];                                    
        size_t start, end;                                      
        size_t len;                                             
        size_t overlap;                                         
        size_t u;                                               

        
        if (sect_start < dblock_start)
            start = 0;
        else
            H5_CHECKED_ASSIGN(start, size_t, (sect_start - dblock_start), hsize_t);
        if (sect_end > dblock_end)
            H5_CHECKED_ASSIGN(end, size_t, udata->dblock_size, hsize_t);
        else
            H5_CHECKED_ASSIGN(end, size_t, ((sect_end - dblock_start) + 1), hsize_t);

        
        len = end - start;

        snprintf(temp_str, sizeof(temp_str), "Section #%u:", (unsigned)udata->sect_count);
        Rfprintf(udata->stream, "%*s%-*s %8zu, %8zu\n", udata->indent + 3, "", MAX(0, udata->fwidth - 9),
                temp_str, start, len);
        udata->sect_count++;

        
        overlap = 0;
        for (u = start; u < end; u++) {
            if (udata->marker[u])
                overlap++;
            udata->marker[u] = 1;
        } 

        
        if (overlap)
            Rfprintf(udata->stream, "***THAT FREE BLOCK OVERLAPPED A PREVIOUS ONE!\n");
        else
            udata->amount_free += len;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF_dblock_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, haddr_t hdr_addr,
                  size_t block_size)
{
    H5HF_hdr_t    *hdr    = NULL;       
    H5HF_direct_t *dblock = NULL;       
    size_t         blk_prefix_size;     
    size_t         amount_free;         
    uint8_t       *marker    = NULL;    
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(H5_addr_defined(hdr_addr));
    assert(block_size > 0);

    
    if (NULL == (hdr = H5HF__hdr_protect(f, hdr_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap header");

    
    if (NULL == (dblock = H5HF__man_dblock_protect(hdr, addr, block_size, NULL, 0, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTLOAD, FAIL, "unable to load fractal heap direct block");

    
    Rfprintf(stream, "%*sFractal Heap Direct Block...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Address of fractal heap that owns this block:", hdr->heap_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Offset of direct block in heap:", dblock->block_off);
    blk_prefix_size = H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Size of block header:", (unsigned long long)blk_prefix_size);

    
    if (NULL == (marker = (uint8_t *)H5MM_calloc(dblock->size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    if (H5HF__space_start(hdr, false) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize heap free space");

    
    if (hdr->fspace) {
        H5HF_debug_iter_ud1_t udata; 

        
        udata.stream      = stream;
        udata.indent      = indent;
        udata.fwidth      = fwidth;
        udata.dblock_addr = dblock->block_off;
        udata.dblock_size = block_size;
        udata.marker      = marker;
        udata.sect_count  = 0;
        udata.amount_free = 0;

        
        Rfprintf(stream, "%*sFree Blocks (offset, size):\n", indent, "");

        
        if (H5FS_sect_iterate(f, hdr->fspace, H5HF_dblock_debug_cb, &udata) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_BADITER, FAIL, "can't iterate over heap's free space");

        
        if (H5HF__space_close(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release free space info");

        
        amount_free = udata.amount_free;

        
        if (amount_free == 0)
            Rfprintf(stream, "%*s<none>\n", indent + 3, "");
    } 
    else
        amount_free = 0;

    Rfprintf(stream, "%*s%-*s %.2f%%\n", indent, "", fwidth, "Percent of available space for data used:",
            (100.0 * (double)((dblock->size - blk_prefix_size) - amount_free) /
             (double)(dblock->size - blk_prefix_size)));

    
    H5_buffer_dump(stream, indent, dblock->blk, marker, (size_t)0, dblock->size);

done:
    if (dblock && H5AC_unprotect(f, H5AC_FHEAP_DBLOCK, addr, dblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_PROTECT, FAIL, "unable to release fractal heap direct block");
    if (hdr && H5AC_unprotect(f, H5AC_FHEAP_HDR, hdr_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_PROTECT, FAIL, "unable to release fractal heap header");
    H5MM_xfree(marker);

    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5HF_iblock_print(const H5HF_indirect_t *iblock, bool dump_internal, FILE *stream, int indent, int fwidth)
{
    const H5HF_hdr_t *hdr;          
    char              temp_str[64]; 
    size_t            u, v;         

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(iblock);
    assert(iblock->hdr);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    hdr = iblock->hdr;

    
    Rfprintf(stream, "%*sFractal Heap Indirect Block...\n", indent, "");

    
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "Address of fractal heap that owns this block:", hdr->heap_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE " \n", indent, "", fwidth,
            "Offset of indirect block in heap:", iblock->block_off);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, "Size of indirect block:", (unsigned long long)iblock->size);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Current # of rows:", iblock->nrows);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Max. # of rows:", iblock->max_rows);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Max direct block rows:", hdr->man_dtable.max_direct_rows);

    
    if (hdr->filter_len > 0)
        Rfprintf(stream, "%*sDirect Block Entries: (address/compressed size/filter mask)\n", indent, "");
    else
        Rfprintf(stream, "%*sDirect Block Entries: (address)\n", indent, "");
    for (u = 0; u < hdr->man_dtable.max_direct_rows && u < iblock->nrows; u++) {
        snprintf(temp_str, sizeof(temp_str), "Row #%u: (block size: %lu)", (unsigned)u,
                 (unsigned long)hdr->man_dtable.row_block_size[u]);
        Rfprintf(stream, "%*s%-*s\n", indent + 3, "", MAX(0, fwidth - 3), temp_str);
        for (v = 0; v < hdr->man_dtable.cparam.width; v++) {
            size_t off = (u * hdr->man_dtable.cparam.width) + v;

            snprintf(temp_str, sizeof(temp_str), "Col #%u:", (unsigned)v);
            if (hdr->filter_len > 0)
                Rfprintf(stream, "%*s%-*s %9" PRIuHADDR "/%6zu/%x\n", indent + 6, "", MAX(0, fwidth - 6),
                        temp_str, iblock->ents[off].addr, iblock->filt_ents[off].size,
                        iblock->filt_ents[off].filter_mask);
            else
                Rfprintf(stream, "%*s%-*s %9" PRIuHADDR "\n", indent + 6, "", MAX(0, fwidth - 6), temp_str,
                        iblock->ents[off].addr);
        } 
    }     
    Rfprintf(stream, "%*sIndirect Block Entries:\n", indent, "");
    if (iblock->nrows > hdr->man_dtable.max_direct_rows) {
        unsigned first_row_bits;    
        unsigned num_indirect_rows; 

        first_row_bits = H5VM_log2_of2((uint32_t)hdr->man_dtable.cparam.start_block_size) +
                         H5VM_log2_of2(hdr->man_dtable.cparam.width);
        for (u = hdr->man_dtable.max_direct_rows; u < iblock->nrows; u++) {
            num_indirect_rows = (H5VM_log2_gen(hdr->man_dtable.row_block_size[u]) - first_row_bits) + 1;
            snprintf(temp_str, sizeof(temp_str), "Row #%u: (# of rows: %u)", (unsigned)u, num_indirect_rows);
            Rfprintf(stream, "%*s%-*s\n", indent + 3, "", MAX(0, fwidth - 3), temp_str);
            for (v = 0; v < hdr->man_dtable.cparam.width; v++) {
                size_t off = (u * hdr->man_dtable.cparam.width) + v;

                snprintf(temp_str, sizeof(temp_str), "Col #%u:", (unsigned)v);
                Rfprintf(stream, "%*s%-*s %9" PRIuHADDR "\n", indent + 6, "", MAX(0, fwidth - 6), temp_str,
                        iblock->ents[off].addr);
            } 
        }     
    }         
    else
        Rfprintf(stream, "%*s%-*s\n", indent + 3, "", MAX(0, fwidth - 3), "<none>");

    
    if (dump_internal) {
        Rfprintf(stream, "%*sFractal Indirect Block Internal Information:\n", indent, "");

        
        Rfprintf(stream, "%*s%-*s %llu\n", indent + 3, "", MAX(0, fwidth - 3), "Reference count:", (unsigned long long)iblock->rc);

        
        Rfprintf(stream, "%*s%-*s %p\n", indent + 3, "", MAX(0, fwidth - 3),
                "Parent indirect block address:", (void *)iblock->parent);
        if (iblock->parent)
            H5HF_iblock_print(iblock->parent, true, stream, indent + 6, fwidth);
    } 

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5HF_iblock_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, haddr_t hdr_addr,
                  unsigned nrows)
{
    H5HF_hdr_t      *hdr         = NULL;    
    H5HF_indirect_t *iblock      = NULL;    
    bool             did_protect = false;   
    herr_t           ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);
    assert(H5_addr_defined(hdr_addr));
    assert(nrows > 0);

    
    if (NULL == (hdr = H5HF__hdr_protect(f, hdr_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap header");

    
    if (NULL == (iblock = H5HF__man_iblock_protect(hdr, addr, nrows, NULL, 0, false, H5AC__READ_ONLY_FLAG,
                                                   &did_protect)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTLOAD, FAIL, "unable to load fractal heap indirect block");

    
    H5HF_iblock_print(iblock, false, stream, indent, fwidth);

done:
    if (iblock && H5HF__man_iblock_unprotect(iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_PROTECT, FAIL, "unable to release fractal heap direct block");
    if (hdr && H5AC_unprotect(f, H5AC_FHEAP_HDR, hdr_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_PROTECT, FAIL, "unable to release fractal heap header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF_sects_debug_cb(H5FS_section_info_t *_sect, void *_udata)
{
    H5HF_free_section_t   *sect      = (H5HF_free_section_t *)_sect;    
    H5HF_debug_iter_ud2_t *udata     = (H5HF_debug_iter_ud2_t *)_udata; 
    herr_t                 ret_value = SUCCEED;                         

    FUNC_ENTER_NOAPI_NOINIT

    
    assert(sect);
    assert(udata);

    
    Rfprintf(udata->stream, "%*s%-*s %s\n", udata->indent, "", udata->fwidth, "Section type:",
            (sect->sect_info.type == H5HF_FSPACE_SECT_SINGLE
                 ? "single"
                 : (sect->sect_info.type == H5HF_FSPACE_SECT_FIRST_ROW
                        ? "first row"
                        : (sect->sect_info.type == H5HF_FSPACE_SECT_NORMAL_ROW ? "normal row" : "unknown"))));
    Rfprintf(udata->stream, "%*s%-*s %" PRIuHADDR "\n", udata->indent, "", udata->fwidth,
            "Section address:", sect->sect_info.addr);
    Rfprintf(udata->stream, "%*s%-*s %" PRIuHSIZE "\n", udata->indent, "", udata->fwidth,
            "Section size:", sect->sect_info.size);

    
    if (H5FS_sect_debug(udata->fspace, _sect, udata->stream, udata->indent + 3, MAX(0, udata->fwidth - 3)) <
        0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADITER, FAIL, "can't dump section's debugging info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_sects_debug(H5F_t *f, haddr_t fh_addr, FILE *stream, int indent, int fwidth)
{
    H5HF_hdr_t *hdr       = NULL;    
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(fh_addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    if (NULL == (hdr = H5HF__hdr_protect(f, fh_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap header");

    
    if (H5HF__space_start(hdr, false) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize heap free space");

    
    if (hdr->fspace) {
        H5HF_debug_iter_ud2_t udata; 

        
        udata.fspace = hdr->fspace;
        udata.stream = stream;
        udata.indent = indent;
        udata.fwidth = fwidth;

        
        if (H5FS_sect_iterate(f, hdr->fspace, H5HF_sects_debug_cb, &udata) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_BADITER, FAIL, "can't iterate over heap's free space");

        
        if (H5HF__space_close(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release free space info");
    } 

done:
    if (hdr && H5AC_unprotect(f, H5AC_FHEAP_HDR, fh_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_PROTECT, FAIL, "unable to release fractal heap header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
