/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the COPYING file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#if !(defined H5HF_FRIEND || defined H5HF_MODULE)
#error "Do not include this file outside the H5HF package!"
#endif
#ifndef H5HFpkg_H
#define H5HFpkg_H
#include "H5HFprivate.h"
#include "H5ACprivate.h"
#include "H5B2private.h"
#include "H5FLprivate.h"
#include "H5FSprivate.h"
#define H5HF_SIZEOF_CHKSUM 4
#define H5HF_METADATA_PREFIX_SIZE(c)                                                                         \
    (H5_SIZEOF_MAGIC                                    \
     + (unsigned)1                                      \
     + ((c) ? (unsigned)H5HF_SIZEOF_CHKSUM : (unsigned)0)                            \
    )
#define H5HF_DTABLE_INFO_SIZE(h)                                                                             \
    ((unsigned)2                                   \
     + (unsigned)(h)->sizeof_size                                                  \
     + (unsigned)(h)->sizeof_size                                            \
     + (unsigned)2        \
     + (unsigned)2                            \
     + (unsigned)(h)->sizeof_addr                                        \
     + (unsigned)2                             \
    )
#define H5HF_HDR_FLAGS_HUGE_ID_WRAPPED  0x01
#define H5HF_HDR_FLAGS_CHECKSUM_DBLOCKS 0x02
#define H5HF_HEADER_SIZE(h)                                                                                  \
    (                                                                           \
     H5HF_METADATA_PREFIX_SIZE(true)                                                                         \
                                                                                                             \
                                                               \
                                                                                                             \
                                                                          \
     + (unsigned)2                                                                         \
     + (unsigned)2                                                            \
     + (unsigned)1                                                                        \
                                                                                                             \
                                                                              \
     + (unsigned)4                                        \
     + (unsigned)(h)->sizeof_size                                            \
     + (unsigned)(h)->sizeof_addr                        \
                                                                                                             \
                                                                \
     + (unsigned)(h)->sizeof_size                                                \
     + (unsigned)(h)->sizeof_addr                                  \
                                                                                                             \
                                                                                 \
     + (unsigned)(h)->sizeof_size                                           \
     + (unsigned)(h)->sizeof_size                           \
     + (unsigned)(h)->sizeof_size                                 \
     + (unsigned)(h)->sizeof_size                                       \
     + (unsigned)(h)->sizeof_size                                           \
     + (unsigned)(h)->sizeof_size                                       \
     + (unsigned)(h)->sizeof_size                                           \
     + (unsigned)(h)->sizeof_size                                       \
                                                                                                             \
                                                              \
     + H5HF_DTABLE_INFO_SIZE(h)                               \
    )
#define H5HF_MAN_ABS_DIRECT_OVERHEAD(h)                                                                      \
    (                                                                           \
     H5HF_METADATA_PREFIX_SIZE(h->checksum_dblocks)                                                          \
                                                                                                             \
                              \
     + (unsigned)(h)->sizeof_addr                              \
     + (unsigned)(h)->heap_off_size                                    \
    )
#define H5HF_MAN_INDIRECT_CHILD_DIR_ENTRY_SIZE(h)                                                            \
    (((h)->filter_len > 0 ? ((unsigned)(h)->sizeof_addr + (unsigned)(h)->sizeof_size + (unsigned)4)          \
                          :                       \
          (unsigned)(h)->sizeof_addr)                    \
    )
#define H5HF_MAN_INDIRECT_SIZE(h, r)                                                                         \
    (                                                                           \
     H5HF_METADATA_PREFIX_SIZE(true)                                                                         \
                                                                                                             \
                            \
     + (unsigned)(h)->sizeof_addr                              \
     + (unsigned)(h)->heap_off_size                                    \
     + (MIN(r, (h)->man_dtable.max_direct_rows) * (h)->man_dtable.cparam.width *                             \
        H5HF_MAN_INDIRECT_CHILD_DIR_ENTRY_SIZE(h))                   \
     + (((r > (h)->man_dtable.max_direct_rows) ? (r - (h)->man_dtable.max_direct_rows) : 0) *                \
        (h)->man_dtable.cparam.width * (h)->sizeof_addr)           \
    )
#define H5HF_SIZEOF_OFFSET_BITS(b) (((b) + 7) / 8)
#define H5HF_SIZEOF_OFFSET_LEN(l)  H5HF_SIZEOF_OFFSET_BITS(H5VM_log2_of2((unsigned)(l)))
#define H5HF_ID_VERS_CURR 0x00
#define H5HF_ID_VERS_MASK 0xC0
#define H5HF_ID_TYPE_MAN      0x00
#define H5HF_ID_TYPE_HUGE     0x10
#define H5HF_ID_TYPE_TINY     0x20
#define H5HF_ID_TYPE_RESERVED 0x30
#define H5HF_ID_TYPE_MASK     0x30
#define H5HF_MAN_ID_ENCODE(i, h, o, l)                                                                       \
    *(i) = H5HF_ID_VERS_CURR | H5HF_ID_TYPE_MAN;                                                             \
    (i)++;                                                                                                   \
    UINT64ENCODE_VAR((i), (o), (h)->heap_off_size);                                                          \
    UINT64ENCODE_VAR((i), (l), (h)->heap_len_size)
#define H5HF_MAN_ID_DECODE(i, h, f, o, l)                                                                    \
    do {                                                                                                     \
        f = *(uint8_t *)i++;                                                                                 \
        UINT64DECODE_VAR((i), (o), (h)->heap_off_size);                                                      \
        UINT64DECODE_VAR((i), (l), (h)->heap_len_size);                                                      \
    } while (0)
#define H5HF_FSPACE_SECT_SINGLE     0
#define H5HF_FSPACE_SECT_FIRST_ROW  1
#define H5HF_FSPACE_SECT_NORMAL_ROW 2
#define H5HF_FSPACE_SECT_INDIRECT   3
#define H5HF_OP_MODIFY 0x0001
#define H5HF_OP_FLAGS  (H5HF_OP_MODIFY)
#define H5HF_ROOT_IBLOCK_PINNED    0x01
#define H5HF_ROOT_IBLOCK_PROTECTED 0x02
typedef struct H5HF_dtable_t {
    H5HF_dtable_cparam_t cparam;
    haddr_t table_addr;
    unsigned curr_root_rows;
    unsigned max_root_rows;
    unsigned max_direct_rows;
    unsigned start_bits;
    unsigned max_direct_bits;
    unsigned max_dir_blk_off_size;
    unsigned first_row_bits;
    hsize_t  num_id_first_row;
    hsize_t *row_block_size;
    hsize_t *row_block_off;
    hsize_t *row_tot_dblock_free;
    size_t *row_max_dblock_free;
} H5HF_dtable_t;
typedef struct H5HF_freelist_t H5HF_freelist_t;
typedef struct H5HF_block_loc_t {
    unsigned row;
    unsigned col;
    unsigned entry;
    H5HF_indirect_t         *context;
    struct H5HF_block_loc_t *up;
} H5HF_block_loc_t;
typedef struct H5HF_block_iter_t {
    bool              ready;
    H5HF_block_loc_t *curr;
} H5HF_block_iter_t;
typedef struct H5HF_free_section_t {
    H5FS_section_info_t sect_info;
    union {
        struct {
            H5HF_indirect_t *parent;
            unsigned         par_entry;
        } single;
        struct {
            struct H5HF_free_section_t *under;
            unsigned                    row;
            unsigned                    col;
            unsigned                    num_entries;
            bool checked_out;
        } row;
        struct {
            union {
                H5HF_indirect_t *iblock;
                hsize_t          iblock_off;
            } u;
            unsigned row;
            unsigned col;
            unsigned num_entries;
            struct H5HF_free_section_t *parent;
            unsigned                    par_entry;
            hsize_t                     span_size;
            unsigned iblock_entries;
            unsigned rc;
            unsigned dir_nrows;
            struct H5HF_free_section_t **dir_rows;
            unsigned                     indir_nents;
            struct H5HF_free_section_t *
                *indir_ents;
        } indirect;
    } u;
} H5HF_free_section_t;
struct H5HF_hdr_t {
    H5AC_info_t cache_info;
    unsigned id_len;
    unsigned filter_len;
    bool debug_objs;
    bool write_once;
    bool huge_ids_wrapped;
    bool checksum_dblocks;
    H5HF_dtable_t man_dtable;
    hsize_t total_man_free;
    haddr_t fs_addr;
    uint32_t max_man_size;
    hsize_t  huge_next_id;
    haddr_t  huge_bt2_addr;
    H5O_pline_t pline;
    size_t      pline_root_direct_size;
    unsigned    pline_root_direct_filter_mask;
    hsize_t man_size;
    hsize_t man_alloc_size;
    hsize_t man_iter_off;
    hsize_t man_nobjs;
    hsize_t huge_size;
    hsize_t huge_nobjs;
    hsize_t tiny_size;
    hsize_t tiny_nobjs;
    size_t                  rc;
    haddr_t                 heap_addr;
    size_t                  heap_size;
    unsigned                mode;
    H5F_t                  *f;
    size_t                  file_rc;
    bool                    pending_delete;
    uint8_t                 sizeof_size;
    uint8_t                 sizeof_addr;
    struct H5HF_indirect_t *root_iblock;
    unsigned root_iblock_flags;
    H5FS_t  *fspace;
    H5HF_block_iter_t next_block;
    H5B2_t           *huge_bt2;
    hsize_t           huge_max_id;
    uint8_t           huge_id_size;
    bool huge_ids_direct;
    size_t tiny_max_len;
    bool   tiny_len_extended;
    uint8_t heap_off_size;
    uint8_t heap_len_size;
    bool    checked_filters;
};
typedef struct H5HF_indirect_ent_t {
    haddr_t addr;
} H5HF_indirect_ent_t;
typedef struct H5HF_indirect_filt_ent_t {
    size_t   size;
    unsigned filter_mask;
} H5HF_indirect_filt_ent_t;
struct H5HF_indirect_t {
    H5AC_info_t cache_info;
    size_t                  rc;
    H5HF_hdr_t             *hdr;
    struct H5HF_indirect_t *parent;
    void                   *fd_parent;
    unsigned                 par_entry;
    haddr_t                  addr;
    size_t                   size;
    unsigned                 nrows;
    unsigned                 max_rows;
    unsigned                 nchildren;
    unsigned                 max_child;
    struct H5HF_indirect_t **child_iblocks;
    bool                     removed_from_cache;
    hsize_t                   block_off;
    H5HF_indirect_ent_t      *ents;
    H5HF_indirect_filt_ent_t *filt_ents;
};
typedef struct H5HF_direct_t {
    H5AC_info_t cache_info;
    H5HF_hdr_t      *hdr;
    H5HF_indirect_t *parent;
    void            *fd_parent;
    unsigned par_entry;
    size_t   size;
    hsize_t  file_size;
    uint8_t *blk;
    uint8_t *write_buf;
    size_t write_size;
    hsize_t block_off;
} H5HF_direct_t;
struct H5HF_t {
    H5HF_hdr_t *hdr;
    H5F_t      *f;
};
typedef struct H5HF_parent_t {
    H5HF_hdr_t      *hdr;
    H5HF_indirect_t *iblock;
    unsigned         entry;
} H5HF_parent_t;
typedef struct H5HF_huge_bt2_indir_rec_t {
    haddr_t addr;
    hsize_t len;
    hsize_t id;
} H5HF_huge_bt2_indir_rec_t;
typedef struct H5HF_huge_bt2_filt_indir_rec_t {
    haddr_t  addr;
    hsize_t  len;
    unsigned filter_mask;
    hsize_t  obj_size;
    hsize_t  id;
} H5HF_huge_bt2_filt_indir_rec_t;
typedef struct H5HF_huge_bt2_dir_rec_t {
    haddr_t addr;
    hsize_t len;
} H5HF_huge_bt2_dir_rec_t;
typedef struct H5HF_huge_bt2_filt_dir_rec_t {
    haddr_t  addr;
    hsize_t  len;
    unsigned filter_mask;
    hsize_t  obj_size;
} H5HF_huge_bt2_filt_dir_rec_t;
typedef struct {
    H5HF_hdr_t *hdr;
} H5HF_sect_add_ud_t;
typedef struct {
    H5HF_hdr_t *hdr;
    hsize_t     obj_len;
} H5HF_huge_remove_ud_t;
typedef struct H5HF_hdr_cache_ud_t {
    H5F_t *f;
} H5HF_hdr_cache_ud_t;
typedef struct H5HF_iblock_cache_ud_t {
    H5HF_parent_t  *par_info;
    H5F_t          *f;
    const unsigned *nrows;
} H5HF_iblock_cache_ud_t;
typedef struct H5HF_dblock_cache_ud_t {
    H5HF_parent_t par_info;
    H5F_t        *f;
    size_t        odi_size;
    size_t dblock_size;
    unsigned filter_mask;
    uint8_t *dblk;
    htri_t decompressed;
} H5HF_dblock_cache_ud_t;
H5_DLLVAR const H5B2_class_t H5HF_HUGE_BT2_INDIR[1];
H5_DLLVAR const H5B2_class_t H5HF_HUGE_BT2_FILT_INDIR[1];
H5_DLLVAR const H5B2_class_t H5HF_HUGE_BT2_DIR[1];
H5_DLLVAR const H5B2_class_t H5HF_HUGE_BT2_FILT_DIR[1];
H5_DLLVAR H5FS_section_class_t H5HF_FSPACE_SECT_CLS_SINGLE[1];
H5_DLLVAR H5FS_section_class_t H5HF_FSPACE_SECT_CLS_FIRST_ROW[1];
H5_DLLVAR H5FS_section_class_t H5HF_FSPACE_SECT_CLS_NORMAL_ROW[1];
H5_DLLVAR H5FS_section_class_t H5HF_FSPACE_SECT_CLS_INDIRECT[1];
H5FL_EXTERN(H5HF_indirect_t);
H5FL_SEQ_EXTERN(H5HF_indirect_ent_t);
H5FL_SEQ_EXTERN(H5HF_indirect_filt_ent_t);
typedef H5HF_indirect_t *H5HF_indirect_ptr_t;
H5FL_SEQ_EXTERN(H5HF_indirect_ptr_t);
H5FL_EXTERN(H5HF_direct_t);
H5FL_BLK_EXTERN(direct_block);
H5_DLL herr_t   H5HF__dtable_init(H5HF_dtable_t *dtable);
H5_DLL herr_t   H5HF__dtable_dest(H5HF_dtable_t *dtable);
H5_DLL herr_t   H5HF__dtable_lookup(const H5HF_dtable_t *dtable, hsize_t off, unsigned *row, unsigned *col);
H5_DLL unsigned H5HF__dtable_size_to_row(const H5HF_dtable_t *dtable, size_t block_size);
H5_DLL unsigned H5HF__dtable_size_to_rows(const H5HF_dtable_t *dtable, hsize_t size);
H5_DLL hsize_t  H5HF__dtable_span_size(const H5HF_dtable_t *dtable, unsigned start_row, unsigned start_col,
                                       unsigned num_entries);
H5_DLL H5HF_hdr_t *H5HF__hdr_alloc(H5F_t *f);
H5_DLL haddr_t     H5HF__hdr_create(H5F_t *f, const H5HF_create_t *cparam);
H5_DLL H5HF_hdr_t *H5HF__hdr_protect(H5F_t *f, haddr_t addr, unsigned flags);
H5_DLL herr_t      H5HF__hdr_finish_init_phase1(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_finish_init_phase2(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_finish_init(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_incr(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_decr(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_fuse_incr(H5HF_hdr_t *hdr);
H5_DLL size_t      H5HF__hdr_fuse_decr(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_dirty(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_adj_free(H5HF_hdr_t *hdr, ssize_t amt);
H5_DLL herr_t      H5HF__hdr_adjust_heap(H5HF_hdr_t *hdr, hsize_t new_size, hssize_t extra_free);
H5_DLL herr_t      H5HF__hdr_inc_alloc(H5HF_hdr_t *hdr, size_t alloc_size);
H5_DLL herr_t      H5HF__hdr_start_iter(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock, hsize_t curr_off,
                                        unsigned curr_entry);
H5_DLL herr_t      H5HF__hdr_skip_blocks(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock, unsigned start_entry,
                                         unsigned nentries);
H5_DLL herr_t      H5HF__hdr_update_iter(H5HF_hdr_t *hdr, size_t min_dblock_size);
H5_DLL herr_t      H5HF__hdr_inc_iter(H5HF_hdr_t *hdr, hsize_t adv_size, unsigned nentries);
H5_DLL herr_t      H5HF__hdr_reverse_iter(H5HF_hdr_t *hdr, haddr_t dblock_addr);
H5_DLL herr_t      H5HF__hdr_reset_iter(H5HF_hdr_t *hdr, hsize_t curr_off);
H5_DLL herr_t      H5HF__hdr_empty(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_free(H5HF_hdr_t *hdr);
H5_DLL herr_t      H5HF__hdr_delete(H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__iblock_incr(H5HF_indirect_t *iblock);
H5_DLL herr_t H5HF__iblock_decr(H5HF_indirect_t *iblock);
H5_DLL herr_t H5HF__iblock_dirty(H5HF_indirect_t *iblock);
H5_DLL herr_t H5HF__man_iblock_root_create(H5HF_hdr_t *hdr, size_t min_dblock_size);
H5_DLL herr_t H5HF__man_iblock_root_double(H5HF_hdr_t *hdr, size_t min_dblock_size);
H5_DLL herr_t H5HF__man_iblock_alloc_row(H5HF_hdr_t *hdr, H5HF_free_section_t **sec_node);
H5_DLL herr_t H5HF__man_iblock_create(H5HF_hdr_t *hdr, H5HF_indirect_t *par_iblock, unsigned par_entry,
                                      unsigned nrows, unsigned max_rows, haddr_t *addr_p);
H5_DLL H5HF_indirect_t *H5HF__man_iblock_protect(H5HF_hdr_t *hdr, haddr_t iblock_addr, unsigned iblock_nrows,
                                                 H5HF_indirect_t *par_iblock, unsigned par_entry,
                                                 bool must_protect, unsigned flags, bool *did_protect);
H5_DLL herr_t H5HF__man_iblock_unprotect(H5HF_indirect_t *iblock, unsigned cache_flags, bool did_protect);
H5_DLL herr_t H5HF__man_iblock_attach(H5HF_indirect_t *iblock, unsigned entry, haddr_t dblock_addr);
H5_DLL herr_t H5HF__man_iblock_detach(H5HF_indirect_t *iblock, unsigned entry);
H5_DLL herr_t H5HF__man_iblock_entry_addr(H5HF_indirect_t *iblock, unsigned entry, haddr_t *child_addr);
H5_DLL herr_t H5HF__man_iblock_delete(H5HF_hdr_t *hdr, haddr_t iblock_addr, unsigned iblock_nrows,
                                      H5HF_indirect_t *par_iblock, unsigned par_entry);
H5_DLL herr_t H5HF__man_iblock_size(H5F_t *f, H5HF_hdr_t *hdr, haddr_t iblock_addr, unsigned nrows,
                                    H5HF_indirect_t *par_iblock, unsigned par_entry,
                                    hsize_t *heap_size);
H5_DLL herr_t H5HF__man_iblock_parent_info(const H5HF_hdr_t *hdr, hsize_t block_off,
                                           hsize_t *ret_par_block_off, unsigned *ret_entry);
H5_DLL herr_t H5HF__man_iblock_dest(H5HF_indirect_t *iblock);
H5_DLL herr_t H5HF__man_dblock_new(H5HF_hdr_t *fh, size_t request, H5HF_free_section_t **ret_sec_node);
H5_DLL herr_t H5HF__man_dblock_create(H5HF_hdr_t *hdr, H5HF_indirect_t *par_iblock, unsigned par_entry,
                                      haddr_t *addr_p, H5HF_free_section_t **ret_sec_node);
H5_DLL herr_t H5HF__man_dblock_destroy(H5HF_hdr_t *hdr, H5HF_direct_t *dblock, haddr_t dblock_addr,
                                       bool *parent_removed);
H5_DLL H5HF_direct_t *H5HF__man_dblock_protect(H5HF_hdr_t *hdr, haddr_t dblock_addr, size_t dblock_size,
                                               H5HF_indirect_t *par_iblock, unsigned par_entry,
                                               unsigned flags);
H5_DLL herr_t         H5HF__man_dblock_locate(H5HF_hdr_t *hdr, hsize_t obj_off, H5HF_indirect_t **par_iblock,
                                              unsigned *par_entry, bool *par_did_protect, unsigned flags);
H5_DLL herr_t         H5HF__man_dblock_delete(H5F_t *f, haddr_t dblock_addr, hsize_t dblock_size);
H5_DLL herr_t         H5HF__man_dblock_dest(H5HF_direct_t *dblock);
H5_DLL herr_t H5HF__man_insert(H5HF_hdr_t *fh, size_t obj_size, const void *obj, void *id);
H5_DLL herr_t H5HF__man_get_obj_len(H5HF_hdr_t *hdr, const uint8_t *id, size_t *obj_len_p);
H5_DLL void   H5HF__man_get_obj_off(const H5HF_hdr_t *hdr, const uint8_t *id, hsize_t *obj_off_p);
H5_DLL herr_t H5HF__man_read(H5HF_hdr_t *fh, const uint8_t *id, void *obj);
H5_DLL herr_t H5HF__man_write(H5HF_hdr_t *hdr, const uint8_t *id, const void *obj);
H5_DLL herr_t H5HF__man_op(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data);
H5_DLL herr_t H5HF__man_remove(H5HF_hdr_t *hdr, const uint8_t *id);
H5_DLL herr_t H5HF__huge_init(H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__huge_insert(H5HF_hdr_t *hdr, size_t obj_size, void *obj, void *id);
H5_DLL herr_t H5HF__huge_get_obj_len(H5HF_hdr_t *hdr, const uint8_t *id, size_t *obj_len_p);
H5_DLL herr_t H5HF__huge_get_obj_off(H5HF_hdr_t *hdr, const uint8_t *id, hsize_t *obj_off_p);
H5_DLL herr_t H5HF__huge_read(H5HF_hdr_t *fh, const uint8_t *id, void *obj);
H5_DLL herr_t H5HF__huge_write(H5HF_hdr_t *hdr, const uint8_t *id, const void *obj);
H5_DLL herr_t H5HF__huge_op(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data);
H5_DLL herr_t H5HF__huge_remove(H5HF_hdr_t *fh, const uint8_t *id);
H5_DLL herr_t H5HF__huge_term(H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__huge_delete(H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__huge_bt2_indir_found(const void *nrecord, void *op_data);
H5_DLL herr_t H5HF__huge_bt2_indir_remove(const void *nrecord, void *op_data);
H5_DLL herr_t H5HF__huge_bt2_filt_indir_found(const void *nrecord, void *op_data);
H5_DLL herr_t H5HF__huge_bt2_filt_indir_remove(const void *nrecord, void *op_data);
H5_DLL herr_t H5HF__huge_bt2_dir_remove(const void *nrecord, void *op_data);
H5_DLL herr_t H5HF__huge_bt2_filt_dir_found(const void *nrecord, void *op_data);
H5_DLL herr_t H5HF__huge_bt2_filt_dir_remove(const void *nrecord, void *op_data);
H5_DLL herr_t H5HF__tiny_init(H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__tiny_insert(H5HF_hdr_t *hdr, size_t obj_size, const void *obj, void *id);
H5_DLL herr_t H5HF__tiny_get_obj_len(H5HF_hdr_t *hdr, const uint8_t *id, size_t *obj_len_p);
H5_DLL herr_t H5HF__tiny_read(H5HF_hdr_t *fh, const uint8_t *id, void *obj);
H5_DLL herr_t H5HF__tiny_op(H5HF_hdr_t *hdr, const uint8_t *id, H5HF_operator_t op, void *op_data);
H5_DLL herr_t H5HF__tiny_remove(H5HF_hdr_t *fh, const uint8_t *id);
H5_DLL herr_t H5HF__man_iter_init(H5HF_block_iter_t *biter);
H5_DLL herr_t H5HF__man_iter_start_offset(H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, hsize_t offset);
H5_DLL herr_t H5HF__man_iter_start_entry(H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, H5HF_indirect_t *iblock,
                                         unsigned start_entry);
H5_DLL herr_t H5HF__man_iter_set_entry(const H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, unsigned entry);
H5_DLL herr_t H5HF__man_iter_next(H5HF_hdr_t *hdr, H5HF_block_iter_t *biter, unsigned nentries);
H5_DLL herr_t H5HF__man_iter_up(H5HF_block_iter_t *biter);
H5_DLL herr_t H5HF__man_iter_down(H5HF_block_iter_t *biter, H5HF_indirect_t *iblock);
H5_DLL herr_t H5HF__man_iter_reset(H5HF_block_iter_t *biter);
H5_DLL herr_t H5HF__man_iter_curr(H5HF_block_iter_t *biter, unsigned *row, unsigned *col, unsigned *entry,
                                  H5HF_indirect_t **block);
H5_DLL bool   H5HF__man_iter_ready(H5HF_block_iter_t *biter);
H5_DLL herr_t H5HF__space_start(H5HF_hdr_t *hdr, bool may_create);
H5_DLL herr_t H5HF__space_add(H5HF_hdr_t *hdr, H5HF_free_section_t *node, unsigned flags);
H5_DLL htri_t H5HF__space_find(H5HF_hdr_t *hdr, hsize_t request, H5HF_free_section_t **node);
H5_DLL herr_t H5HF__space_revert_root(const H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__space_create_root(const H5HF_hdr_t *hdr, H5HF_indirect_t *root_iblock);
H5_DLL herr_t H5HF__space_size(H5HF_hdr_t *hdr, hsize_t *fs_size);
H5_DLL herr_t H5HF__space_remove(H5HF_hdr_t *hdr, H5HF_free_section_t *node);
H5_DLL herr_t H5HF__space_close(H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__space_delete(H5HF_hdr_t *hdr);
H5_DLL herr_t H5HF__space_sect_change_class(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, uint16_t new_class);
H5_DLL H5HF_free_section_t *H5HF__sect_single_new(hsize_t sect_off, size_t sect_size, H5HF_indirect_t *parent,
                                                  unsigned par_entry);
H5_DLL herr_t               H5HF__sect_single_revive(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);
H5_DLL herr_t               H5HF__sect_single_dblock_info(H5HF_hdr_t *hdr, const H5HF_free_section_t *sect,
                                                          haddr_t *dblock_addr, size_t *dblock_size);
H5_DLL herr_t               H5HF__sect_single_reduce(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, size_t amt);
H5_DLL herr_t               H5HF__sect_row_revive(H5HF_hdr_t *hdr, H5HF_free_section_t *sect);
H5_DLL herr_t           H5HF__sect_row_reduce(H5HF_hdr_t *hdr, H5HF_free_section_t *sect, unsigned *entry_p);
H5_DLL H5HF_indirect_t *H5HF__sect_row_get_iblock(H5HF_free_section_t *sect);
H5_DLL herr_t H5HF__sect_indirect_add(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock, unsigned start_entry,
                                      unsigned nentries);
H5_DLL herr_t H5HF__sect_single_free(H5FS_section_info_t *sect);
H5_DLL herr_t H5HF__op_read(const void *obj, size_t obj_len, void *op_data);
H5_DLL herr_t H5HF__op_write(const void *obj, size_t obj_len, void *op_data);
#ifdef H5HF_TESTING
H5_DLL herr_t   H5HF_get_cparam_test(const H5HF_t *fh, H5HF_create_t *cparam);
H5_DLL int      H5HF_cmp_cparam_test(const H5HF_create_t *cparam1, const H5HF_create_t *cparam2);
H5_DLL unsigned H5HF_get_max_root_rows(const H5HF_t *fh);
H5_DLL unsigned H5HF_get_dtable_width_test(const H5HF_t *fh);
H5_DLL unsigned H5HF_get_dtable_max_drows_test(const H5HF_t *fh);
H5_DLL unsigned H5HF_get_iblock_max_drows_test(const H5HF_t *fh, unsigned pos);
H5_DLL hsize_t  H5HF_get_dblock_size_test(const H5HF_t *fh, unsigned row);
H5_DLL hsize_t  H5HF_get_dblock_free_test(const H5HF_t *fh, unsigned row);
H5_DLL herr_t   H5HF_get_id_off_test(const H5HF_t *fh, const void *id, hsize_t *obj_off);
H5_DLL herr_t   H5HF_get_id_type_test(const void *id, unsigned char *obj_type);
H5_DLL herr_t   H5HF_get_tiny_info_test(const H5HF_t *fh, size_t *max_len, bool *len_extended);
H5_DLL herr_t   H5HF_get_huge_info_test(const H5HF_t *fh, hsize_t *next_id, bool *ids_direct);
#endif
#endif
