library(testthat)
library(gwsem)
library(MASS)

suppressWarnings(RNGversion("3.5"))
set.seed(1)

dir <- system.file("extdata", package = "gwsem")
tdir <- tempdir()

pheno <- read.table(file.path(dir, "example.psam"),
                    stringsAsFactors = FALSE,header=TRUE, comment.char="")
colnames(pheno)[1] <- "FID"

numIndicators <- 7
loadings <- rbeta(numIndicators, 4, 3)
resid <- rbeta(numIndicators, 4, 3)^2
indicators <- pheno$phenotype %*% t(loadings) +
  mvrnorm(nrow(pheno), mu=rep(0, numIndicators), Sigma=diag(numIndicators))
colnames(indicators) <- paste0("i", 1:numIndicators)
pheno <- cbind(pheno, indicators)
origPheno <- pheno
pheno$i1 <- cut(pheno$i1, c(-Inf, quantile(pheno$i1, 1:2/3), Inf), ordered_result = TRUE)
pheno$i2 <- cut(pheno$i2, c(-Inf, quantile(pheno$i2, .5), Inf), ordered_result = TRUE)

numCovariate <- 2
for (cx in 1:numCovariate) {
  pheno[[paste0("covar", cx)]] <- rnorm(nrow(pheno))
}

# -----

m1 <- buildOneFac(pheno, paste0("i", 1:numIndicators),
                  covariates = paste0("covar",1:numCovariate), exogenous=TRUE)
expect_equivalent(m1$M$labels[1,'covar1'], 'data.covar1')
GWAS(m1, file.path(dir,"example.pgen"),
     file.path(tdir,"out.log"))

pgen <- read.table(file.path(tdir,"out.log"), stringsAsFactors = FALSE, header=TRUE,
                   sep="\t", check.names=FALSE, quote="", comment.char="")

mask <- (pgen$catch1 == "" & pgen$statusCode=="OK" & !is.na(pgen$snp2FSE))
pgen <- pgen[mask,]

mask <- (abs(pgen$snp2F) < 2.6*mad(pgen$snp2F) & (abs(pgen$snp2F / pgen$snp2FSE) > .5))
pgen <- pgen[mask,]

cvNames <- paste(rep(paste0("covar",1:numCovariate), each = numIndicators),
      paste0("i", 1:numIndicators), sep = "2")
expect_equivalent(colMeans(pgen[,cvNames]), rep(0, length(cvNames)), tolerance=.1)

pgen <- loadResults(file.path(tdir,"out.log"), "snp2F")
expect_error(plot(pgen, y=1),
             "plot does not accept a y= argument")

# -----

m2 <- buildOneFac(pheno, paste0("i", 1:numIndicators),
                  covariates = paste0("covar",1:numCovariate))
expect_equivalent(m2$M$labels[1,'covar1'], 'covar1_mean')
GWAS(m2,
     file.path(dir,"example.pgen"),
     file.path(tdir,"out.log"))

pgen2 <- loadResults(file.path(tdir,"out.log"), "snp2F")

expect_equal(cor(pgen$Z, pgen2$Z, use = "pairwise"), 1, tolerance=.2)

# ----- compare OneFacRes exo vs endo covariates

m1 <- buildOneFacRes(pheno, paste0("i", 1:numIndicators),
                  covariates = paste0("covar",1:numCovariate), exogenous=TRUE)
expect_equivalent(m1$M$labels[1,'covar1'], 'data.covar1')
GWAS(m1, file.path(dir,"example.pgen"),
     file.path(tdir,"outx.log"))

m2 <- buildOneFacRes(pheno, paste0("i", 1:numIndicators),
                     covariates = paste0("covar",1:numCovariate))
expect_equivalent(m2$M$labels[1,'covar1'], 'covar1_mean')
GWAS(m2, file.path(dir,"example.pgen"),
     file.path(tdir,"out.log"))

for (ind in paste0("snp2i", 1:numIndicators)) {
  m1o <- loadResults(file.path(tdir,"outx.log"), ind)
  m2o <- loadResults(file.path(tdir,"out.log"), ind)
  # Wow, this tolerance is really terrible TODO
  expect_equal(cor(m1o$Z, m2o$Z, use="complete.obs"), 1, tolerance=1.2)
}

# ----- compare TwoFac exo vs endo covariates

m1 <- buildTwoFac(pheno,
                  paste0("i", 1:(numIndicators-1)),
                  paste0("i", 2:numIndicators),
                  covariates = paste0("covar",1:numCovariate),
                  exogenous=TRUE)
expect_equivalent(m1$M$labels[1,'covar1'], 'data.covar1')
GWAS(m1, file.path(dir,"example.pgen"),
     file.path(tdir,"outx.log"))

m2 <- buildTwoFac(pheno,
                  paste0("i", 1:(numIndicators-1)),
                  paste0("i", 2:numIndicators),
                  covariates = paste0("covar",1:numCovariate))
expect_equivalent(m2$M$labels[1,'covar1'], 'covar1_mean')
GWAS(m2, file.path(dir,"example.pgen"),
     file.path(tdir,"out.log"))

for (ind in paste0("snp2F", 1:2)) {
  m1o <- loadResults(file.path(tdir,"outx.log"), ind)
  m2o <- loadResults(file.path(tdir,"out.log"), ind)
  # Wow, this tolerance is really terrible TODO
  expect_equal(cor(m1o$Z, m2o$Z, use="complete.obs"), 1, tolerance=1.2)
}

# ----- test continuous endogenous covariates

lastFit <- GWAS(buildTwoFac(pheno,
                            paste0("i", 1:(numIndicators-1)),
                            paste0("i", 2:numIndicators),
                 covariates = paste0("covar",1:numCovariate)), SNP=1,
     file.path(dir,"example.pgen"),
     file.path(tdir,"out.log"))

expect_equal(lastFit$A$labels['F1','covar1'], "covar12F1")
expect_equal(lastFit$A$labels['i2','F1'], "F1_lambda_i2")

# ----- test ordinal endogenous covariates

lastFit <- GWAS(buildOneFac(pheno, paste0("i", 2:numIndicators),
                 covariates = 'i1'), SNP=1,
     file.path(dir,"example.pgen"),
     file.path(tdir,"out.log"))

expect_true(!lastFit$M$free[,'i1'])
expect_equal(lastFit$S$values['i1','i1'], 1)
expect_equal(lastFit$A$labels['F','i1'], 'i12F')
expect_equal(lastFit$A$values['F','i1'], .67, tolerance=.01)
