% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modify_columns.R
\name{cols_merge_uncert}
\alias{cols_merge_uncert}
\title{Merge columns to a value-with-uncertainty column}
\usage{
cols_merge_uncert(data, col_val, col_uncert, sep = " +/- ", autohide = TRUE)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{col_val}{A single column name that contains the base values. This is
the column where values will be mutated.}

\item{col_uncert}{Either one or two column names that contain the uncertainty
values. The most common case involves supplying a single column with
uncertainties; these values will be combined with those in \code{col_val}. Less
commonly, lower and upper uncertainty bounds may be different. For that
case two columns (representing lower and upper uncertainty values away from
\code{col_val}, respectively) should be provided. Since we often don't want the
uncertainty value columns in the output table, we can automatically hide
any \code{col_uncert} columns through the \code{autohide} option.}

\item{sep}{The separator text that contains the uncertainty mark for a single
uncertainty value. The default value of \code{" +/- "} indicates that an
appropriate plus/minus mark will be used depending on the output context.
Should you want this special symbol to be taken literally, it can be
supplied within the \code{\link[=I]{I()}} function.}

\item{autohide}{An option to automatically hide any columns specified in
\code{col_uncert}. Any columns with their state changed to 'hidden' will behave
the same as before, they just won't be displayed in the finalized table.
By default, this is set to \code{TRUE}.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
The \code{cols_merge_uncert()} function is a specialized variant of the
\code{\link[=cols_merge]{cols_merge()}} function. It takes as input a base value column (\code{col_val})
and either: (1) a single uncertainty column, or (2) two columns representing
lower and upper uncertainty bounds. These columns will be essentially merged
in a single column (that of \code{col_val}). What results is a column with values
and associated uncertainties (e.g., \verb{12.0 ± 0.1}), and any columns specified
in \code{col_uncert} are hidden from appearing the output table.
}
\details{
This function could be somewhat replicated using \code{\link[=cols_merge]{cols_merge()}} in the case
where a single column is supplied for \code{col_uncert}, however,
\code{cols_merge_uncert()} employs the following specialized semantics for \code{NA}
handling:
\enumerate{
\item \code{NA}s in \code{col_val} result in missing values for the merged column (e.g.,
\code{NA} + \code{0.1} = \code{NA})
\item \code{NA}s in \code{col_uncert} (but not \code{col_val}) result in base values only for
the merged column (e.g., \code{12.0} + \code{NA} = \code{12.0})
\item \code{NA}s both \code{col_val} and \code{col_uncert} result in missing values for the
merged column (e.g., \code{NA} + \code{NA} = \code{NA})
}

Any resulting \code{NA} values in the \code{col_val} column following the merge
operation can be easily formatted using the \code{\link[=sub_missing]{sub_missing()}} function.

This function is part of a set of four column-merging functions. The other
two are the general \code{\link[=cols_merge]{cols_merge()}} function and the specialized
\code{\link[=cols_merge_range]{cols_merge_range()}} and \code{\link[=cols_merge_n_pct]{cols_merge_n_pct()}} functions. These functions
operate similarly, where the non-target columns can be optionally hidden from
the output table through the \code{hide_columns} or \code{autohide} options.
}
\section{Examples}{


Use \code{\link{exibble}} to create a \strong{gt} table, keeping only the \code{currency} and
\code{num} columns. Merge columns into one with a base value and uncertainty
(after formatting the \code{num} column) using the \code{cols_merge_uncert()} function.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble \%>\%
  dplyr::select(currency, num) \%>\%
  dplyr::slice(1:7) \%>\%
  gt() \%>\%
  fmt_number(
    columns = num,
    decimals = 3,
    use_seps = FALSE
  ) \%>\%
  cols_merge_uncert(
    col_val = currency,
    col_uncert = num
  ) \%>\%
  cols_label(currency = "value + uncert.")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_merge_uncert_1.png" alt="This image of a table was generated from the first code example in the `cols_merge_uncert()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

4-9
}

\seealso{
Other column modification functions: 
\code{\link{cols_align}()},
\code{\link{cols_hide}()},
\code{\link{cols_label}()},
\code{\link{cols_merge_n_pct}()},
\code{\link{cols_merge_range}()},
\code{\link{cols_merge}()},
\code{\link{cols_move_to_end}()},
\code{\link{cols_move_to_start}()},
\code{\link{cols_move}()},
\code{\link{cols_unhide}()},
\code{\link{cols_width}()}
}
\concept{column modification functions}
