% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/collapse_groups.R
\name{collapse_groups}
\alias{collapse_groups}
\title{Collapse groups with categorical, numerical, ID, and size balancing}
\usage{
collapse_groups(
  data,
  n,
  group_cols,
  cat_cols = NULL,
  cat_levels = NULL,
  num_cols = NULL,
  id_cols = NULL,
  balance_size = TRUE,
  auto_tune = FALSE,
  weights = NULL,
  method = "balance",
  group_aggregation_fn = mean,
  num_new_group_cols = 1,
  unique_new_group_cols_only = TRUE,
  max_iters = 5,
  extreme_pairing_levels = 1,
  combine_method = "avg_standardized",
  col_name = ".coll_groups",
  parallel = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{data}{\code{data.frame}. Can be \emph{grouped}, in which case
the function is applied group-wise.}

\item{n}{Number of new groups.

When \code{`num_new_group_cols` > 1}, \code{`n`} can also be a vector
with one \code{`n`} per new group column. This allows trying multiple \code{`n`}
settings at a time. Note that the generated group columns are not guaranteed
to be in the order of \code{`n`}.}

\item{group_cols}{Names of factors in \code{`data`} for identifying the \emph{existing} groups
that should be collapsed.

Multiple names are treated as in \code{\link[dplyr:group_by]{dplyr::group_by()}}
(i.e., a hierarchy of groups), where each leaf group within each parent group is
considered a unique group to be collapsed.
Parent groups are not considered during collapsing, why leaf groups from different
parent groups can be collapsed together.

\strong{Note}: Do not confuse these group columns with potential columns that \code{`data`} is grouped by.
\code{`group_cols`} identifies the groups to be collapsed. When \code{`data`} is
grouped with \code{\link[dplyr:group_by]{dplyr::group_by()}}, the function is
applied separately to each of those subsets.}

\item{cat_cols}{Names of categorical columns to balance the average frequency
of one or more levels of.}

\item{cat_levels}{Names of the levels in the \code{`cat_cols`} columns to balance the average frequencies
of. When \code{`NULL`} (default), all levels are balanced.
Can be weights indicating the balancing importance of each level (within each column).

The weights are automatically scaled to sum to \code{1}.

Can be \code{".minority"} or \code{".majority"}, in which case the minority/majority level
are found and used.

\subsection{When \code{`cat_cols`} has single column name:}{

Either a \code{vector} with level names or a named \code{numeric vector} with weights:

E.g. \code{c("dog", "pidgeon", "mouse")} or \code{c("dog" = 5, "pidgeon" = 1, "mouse" = 3)}
}

\subsection{When \code{`cat_cols`} has multiple column names:}{

A named \code{list} with \code{vector}s for each column name in \code{`cat_cols`}.
When not providing a \code{vector} for a \code{`cat_cols`}
column, all levels are balanced in that column.

E.g. \code{list("col1" = c("dog" = 5, "pidgeon" = 1, "mouse" = 3),
 "col2" = c("hydrated", "dehydrated"))}.
}}

\item{num_cols}{Names of numerical columns to balance between groups.}

\item{id_cols}{Names of factor columns with IDs to balance the counts of between groups.

E.g. useful to get a similar number of participants in each group.}

\item{balance_size}{Whether to balance the size of the collapsed groups. (logical)}

\item{auto_tune}{Whether to create a larger set of collapsed group columns
from all combinations of the balancing dimensions and select the
overall most balanced group column(s).

This tends to create much more balanced collapsed group columns.

Can be slow, why we recommend enabling parallelization (see \code{`parallel`}).}

\item{weights}{Named \code{vector} with balancing importance weights for each of
the balancing columns. Besides the columns in \code{`cat_cols`}, \code{`num_cols`}, and \code{`id_cols`},
the \emph{size} balancing weight can be given as \code{"size"}.

The weights are automatically scaled to sum to \code{1}.

Dimensions that are \emph{not} given a weight is automatically given the weight \code{1}.

E.g. \code{c("size" = 1, "cat" = 1, "num1" = 4, "num2" = 7, "id" = 2)}.}

\item{method}{\code{"balance"}, \code{"ascending"}, or \code{"descending"}:

After calculating a \emph{combined balancing column} from each of the balancing columns (see \code{Details >> Balancing columns}):
\itemize{
\item \code{"balance"} balances the combined balancing column between the groups.
\item \code{"ascending"} orders the combined balancing column and groups from the lowest to highest value.
\item \code{"descending"} orders the combined balancing column and groups from the highest to lowest value.
}}

\item{group_aggregation_fn}{Function for aggregating values in the \code{`num_cols`} columns
for each group in \code{`group_cols`}.

Default is \code{mean()}, where the average value(s) are balanced across the new groups.

When using \code{sum()}, the groups will have similar sums across the new groups.

\strong{N.B.} Only used when \code{`num_cols`} is specified.}

\item{num_new_group_cols}{Number of group columns to create.

When \code{`num_new_group_cols` > 1}, columns are named
with a combination of \code{`col_name`} and \code{"_1"}, \code{"_2"}, etc.
E.g. \eqn{".coll_groups_1"}, \eqn{".coll_groups_2"}, ...

\strong{N.B.} When \code{`unique_new_group_cols_only`} is \code{`TRUE`},
we may end up with fewer columns than specified, see \code{`max_iters`}.}

\item{unique_new_group_cols_only}{Whether to only return unique new group columns.

As the number of column comparisons can be quite time consuming,
we recommend enabling parallelization. See \code{`parallel`}.

\strong{N.B.} We can end up with fewer columns than specified in
\code{`num_new_group_cols`}, see \code{`max_iters`}.

\strong{N.B.} Only used when \code{`num_new_group_cols` > 1}.}

\item{max_iters}{Maximum number of attempts at reaching
\code{`num_new_group_cols`} \emph{unique} new group columns.

When only keeping unique new group columns, we risk having fewer columns than expected.
Hence, we repeatedly create the missing columns and remove those that are not unique.
This is done until we have \code{`num_new_group_cols`} unique group columns
or we have attempted \code{`max_iters`} times.

In some cases, it is not possible to create \code{`num_new_group_cols`}
unique combinations of the dataset.
\code{`max_iters`} specifies when to stop trying.
Note that we can end up with fewer columns than specified in \code{`num_new_group_cols`}.

\strong{N.B.} Only used when \code{`num_new_group_cols` > 1}.}

\item{extreme_pairing_levels}{How many levels of extreme pairing to do
when balancing the groups by the combined balancing column (see \code{Details}).

\strong{Extreme pairing}: Rows/pairs are ordered as smallest, largest,
second smallest, second largest, etc. If \code{extreme_pairing_levels > 1},
this is done "recursively" on the extreme pairs.

\strong{N.B.} Larger values work best with large datasets. If set too high,
the result might not be stochastic. Always check if an increase
actually makes the groups more balanced.}

\item{combine_method}{Method to combine the balancing columns by.
One of \code{"avg_standardized"} or \code{"avg_min_max_scaled"}.

For each balancing column (all columns in \emph{\code{num_cols}}, \emph{\code{cat_cols}},
and \emph{\code{id_cols}}, plus \emph{size}), we calculate a normalized, numeric group summary column, which indicates the
"size" of each group in that dimension. These are then combined to a single
\emph{combined balancing column}.

The three steps are:
\enumerate{
\item Calculate a numeric representation of the balance for each column.
E.g. the number of unique levels within each group of an ID column
(see \code{Details > Balancing columns} for more on this).
\item Normalize each column separately with standardization (\code{"avg_standardized"}; Default) or MinMax scaling
to the [0, 1] range (\code{"avg_min_max_scaled"}).
\item Average the columns \emph{rowwise} to get a single column with one value per group. The averaging
is weighted by \code{`weights`}, which is useful when one of the dimensions is
more important to get a good balance of.
}

\code{`combine_method`} chooses whether to use standardization or MinMax scaling in step 2.}

\item{col_name}{Name of the new group column. When creating multiple new group columns
(\code{`num_new_group_cols`>1}), this is the prefix for the names, which will
be suffixed with an underscore and a number (_1, _2, _3, etc.).}

\item{parallel}{Whether to parallelize the group column comparisons
when \code{`unique_new_group_cols_only`} is \code{`TRUE`}.

Especially highly recommended when \code{`auto_tune`} is enabled.

Requires a registered parallel backend.
Like \code{doParallel::registerDoParallel}.}

\item{verbose}{Whether to print information about the process.
May make the function slightly slower.

N.B. Currently only used during auto-tuning.}
}
\value{
\code{data.frame} with one or more new grouping factors.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Collapses a set of groups into a smaller set of groups.

\emph{Attempts} to balance the new groups by specified numerical columns,
categorical columns,
level counts in ID columns,
and/or the number of rows (size).

\strong{Note}: The more of these you balance at a time,
the less balanced each of them may become. While, \emph{on average},
the balancing work better than without, this is
\strong{not guaranteed on every run}. Enabling \code{`auto_tune`} can yield a
much better overall balance than without in most contexts.
This generates a larger set of group columns using all combinations of the
balancing columns and selects the most balanced group column(s).
This is slower and we recommend enabling parallelization (see \code{`parallel`}).

While this balancing algorithm will not be \emph{optimal} in all cases,
it allows balancing a \strong{large} number of columns at once. Especially
with auto-tuning enabled, this can be very powerful.

\strong{Tip}: Check the balances of the new groups with
\code{\link[groupdata2:summarize_balances]{summarize_balances()}} and
\code{\link[groupdata2:ranked_balances]{ranked_balances()}}.

\strong{Note}: The categorical and ID balancing algorithms are different to those
in \code{\link[groupdata2:fold]{fold()}} and
\code{\link[groupdata2:partition]{partition()}}.
}
\details{
The goal of \code{collapse_groups()} is to combine existing groups
to a lower number of groups while (optionally) balancing one or more
\emph{numeric}, \emph{categorical} and/or \emph{ID} columns, along with the group
\emph{size}.

For each of these columns (and size), we calculate a normalized, numeric \emph{"balancing column"}
that when balanced between the groups lead to its original column being balanced as well.

To balance multiple columns at once, we combine their balancing columns with
weighted averaging (see \code{`combine_method`} and \code{`weights`}) to a single
\emph{combined balancing column}.

Finally, we create groups where this combined balancing column is balanced between the groups,
using the numerical balancing in \code{\link[groupdata2:fold]{fold()}}.

\subsection{Auto-tuning}{

This strategy is not guaranteed to produce balanced groups in all contexts,
e.g. when the balancing columns cancel out. To increase the probability of
balanced groups, we can produce multiple group columns with all combinations
of the balancing columns and select the overall most balanced group column(s).
We refer to this as auto-tuning (see \code{`auto_tune`}).

We find the overall most balanced group column by ranking the across-group
standard deviations for each of the balancing columns, as found with
\code{\link[groupdata2:summarize_balances]{summarize_balances()}}.

\strong{Example} of finding the overall most balanced group column(s):

Given a group column with the following average \emph{age} per group: \code{`c(16, 18, 25, 21)`},
the standard deviation hereof (\code{3.92}) is a measure of how balanced the \emph{age}
column is. Another group column can thus have a lower/higher standard deviation
and be considered more/less balanced.

We find the rankings of these standard deviations for all the balancing columns
and average them (again weighted by \code{`weights`}). We select the group column(s) with the,
on average, highest rank (i.e. lowest standard deviations).

}\subsection{Checking balances}{

We highly recommend using
\code{\link[groupdata2:summarize_balances]{summarize_balances()}}
and \code{\link[groupdata2:ranked_balances]{ranked_balances()}} to
check how balanced the created groups are on the various dimensions.
When applying \code{\link[groupdata2:ranked_balances]{ranked_balances()}}
to the output of \code{\link[groupdata2:summarize_balances]{summarize_balances()}},
we get a \code{data.frame} with the standard deviations
for each balancing dimension (lower means more balanced),
ordered by the average rank (see \code{Examples}).

}\subsection{Balancing columns}{

The following describes the creation of the balancing columns
for each of the supported column types:

\subsection{cat_cols}{
For each column in \code{`cat_cols`}:
\itemize{
\item \strong{Count each level} within each group. This creates a \code{data.frame} with
one count column per level, with one row per group.
\item \strong{Standardize} the count columns.
\item \strong{Average} the standardized counts rowwise to create one combined column representing
the balance of the levels for each group. When \code{cat_levels} contains weights for each of the levels,
we apply weighted averaging.
}

\strong{Example}: Consider a factor column with the levels \code{c("A", "B", "C")}.
We count each level per group, \strong{n}ormalize the counts and combine them with weighted averaging:

\tabular{rrrrrrrrrr}{
\strong{Group} \tab \strong{A} \tab
\strong{B} \tab \strong{C} \tab
\strong{ -> } \tab \strong{nA} \tab
\strong{nB} \tab \strong{nC} \tab
\strong{ -> } \tab \strong{Combined}\cr
1 \tab 5 \tab 57 \tab 1 \tab | \tab 0.24 \tab 0.55 \tab -0.77 \tab | \tab 0.007  \cr
2 \tab 7 \tab 69 \tab 2 \tab | \tab 0.93 \tab 0.64 \tab -0.77 \tab | \tab 0.267 \cr
3 \tab 2 \tab 34 \tab 14\tab | \tab -1.42\tab 0.29 \tab 1.34  \tab | \tab 0.07 \cr
4 \tab 5 \tab 0 \tab 4  \tab | \tab 0.24 \tab -1.48\tab 0.19  \tab | \tab -0.35 \cr
... \tab ... \tab ... \tab ... \tab | \tab ... \tab ... \tab ... \tab | \tab ... }

}

\subsection{id_cols}{
For each column in \code{`id_cols`}:
\itemize{
\item \strong{Count} the unique IDs (levels) within each group.
(Note: The same ID can be counted in multiple groups.)
}
}

\subsection{num_cols}{
For each column in \code{`num_cols`}:
\itemize{
\item \strong{Aggregate} the numeric columns by group using the \code{`group_aggregation_fn`}.
}
}

\subsection{size}{
\itemize{
\item \strong{Count} the number of rows per group.
}
}

\subsection{Combining balancing columns}{
\itemize{
\item Apply standardization or MinMax scaling to each of the balancing columns (see \code{`combine_method`}).
\item Perform weighted averaging to get a single balancing column (see \code{`weights`}).
}

\strong{Example}: We apply standardization and perform weighted averaging:

\tabular{rrrrrrrrrrrr}{
\strong{Group} \tab \strong{Size} \tab
\strong{Num} \tab \strong{Cat} \tab
\strong{ID} \tab \strong{->} \tab
\strong{nSize} \tab \strong{nNum}
\tab \strong{nCat} \tab \strong{nID} \tab
\strong{->} \tab \strong{Combined}\cr
1 \tab 34 \tab 1.3 \tab 0.007 \tab 3 \tab | \tab -0.33 \tab -0.82 \tab 0.03 \tab -0.46\tab | \tab -0.395  \cr
2 \tab 23 \tab 4.6 \tab 0.267 \tab 4 \tab | \tab -1.12 \tab 0.34  \tab 1.04 \tab 0.0  \tab | \tab 0.065 \cr
3 \tab 56 \tab 7.2 \tab 0.07  \tab 7 \tab | \tab 1.27  \tab 1.26  \tab 0.28 \tab 1.39 \tab | \tab 1.05 \cr
4 \tab 41 \tab 1.4 \tab -0.35 \tab 2 \tab | \tab 0.18  \tab -0.79 \tab -1.35\tab -0.93\tab | \tab -0.723 \cr
... \tab ... \tab ... \tab ... \tab ... \tab |
\tab ... \tab ... \tab ... \tab ... \tab | \tab ... }

}

}\subsection{Creating the groups}{

Finally, we get to the group creation. There are three methods for creating groups based on the
combined balancing column: \code{"balance"} (default), \code{"ascending"}, and \code{"descending"}.

\subsection{\code{method} is "balance"}{
To create groups that are balanced by the combined balancing column, we use the numerical balancing
in \code{\link[groupdata2:fold]{fold()}}.

The following describes the numerical balancing in broad terms:

\enumerate{
\item Rows are shuffled.
\strong{Note} that this will only affect rows with the same value in the combined balancing column.
\item Extreme pairing 1: Rows are ordered as \emph{smallest, largest, second smallest, second largest}, etc.
Each small+large pair get an \emph{extreme-group} identifier. (See \code{\link[rearrr:pair_extremes]{rearrr::pair_extremes()}})
\item If \code{`extreme_pairing_levels` > 1}: These extreme-group identifiers are reordered as \emph{smallest,
largest, second smallest, second largest}, etc., by the \code{sum} of the combined balancing column in the represented rows.
These pairs (of pairs) get a new set of extreme-group identifiers, and the process is repeated
\code{`extreme_pairing_levels`-2} times. Note that the extreme-group identifiers at the last level will represent
\code{2^`extreme_pairing_levels`} rows, why you should be careful when choosing a larger setting.
\item The extreme-group identifiers from the last pairing are randomly divided into the final groups
and these final identifiers are transferred to the original rows.
}

\strong{N.B.} When doing extreme pairing of an unequal number of rows,
the row with the smallest value is placed in a group by itself, and the order is instead:
(smallest), \emph{(second smallest, largest), (third smallest, second largest)}, etc.

A similar approach with \emph{extreme triplets} (i.e. smallest, closest to median, largest,
second smallest, second closest to median, second largest, etc.) may also be utilized in some scenarios.
(See \code{\link[rearrr:triplet_extremes]{rearrr::triplet_extremes()}})

\strong{Example}: We order the \code{data.frame} by smallest \emph{"Num"} value,
largest \emph{"Num"} value, second smallest, and so on.
We \emph{could} further (when \code{`extreme_pairing_levels` > 1})
find the sum of \emph{"Num"} for each pair and perform extreme pairing on the pairs.
Finally, we group the \code{data.frame}:

\tabular{rrrrrrrrrrrr}{
\strong{Group} \tab \strong{Num} \tab
\strong{->} \tab
\strong{Group} \tab \strong{Num} \tab
\strong{Pair} \tab \strong{->} \tab
\strong{New group}\cr
1 \tab -0.395\tab | \tab 5 \tab -1.23 \tab 1 \tab | \tab 3 \cr
2 \tab 0.065 \tab | \tab 3 \tab 1.05  \tab 1 \tab | \tab 3 \cr
3 \tab 1.05  \tab | \tab 4 \tab -0.723\tab 2 \tab | \tab 1 \cr
4 \tab -0.723\tab | \tab 2 \tab 0.065 \tab 2 \tab | \tab 1 \cr
5 \tab -1.23 \tab | \tab 1 \tab -0.395\tab 3 \tab | \tab 2 \cr
6 \tab -0.15 \tab | \tab 6 \tab -0.15 \tab 3 \tab | \tab 2 \cr
... \tab ... \tab | \tab ... \tab ... \tab ... \tab | \tab ... }

}

\subsection{\code{method} is "ascending" or "descending"}{
These methods order the data by the combined balancing column and
creates groups such that the sums get increasingly larger (\code{`ascending`})
or smaller (\code{`descending`}). This will in turn lead to a \emph{pattern} of
increasing/decreasing sums in the balancing columns (e.g. increasing/decreasing counts
of the categorical levels, counts of IDs, number of rows and sums of numeric columns).
}

}
}
\examples{
# Attach packages
library(groupdata2)
library(dplyr)

# Set seed
xpectr::set_test_seed(42)

# Create data frame
df <- data.frame(
  "participant" = factor(rep(1:20, 3)),
  "age" = rep(sample(c(1:100), 20), 3),
  "answer" = factor(sample(c("a", "b", "c", "d"), 60, replace = TRUE)),
  "score" = sample(c(1:100), 20 * 3)
)
df <- df \%>\% dplyr::arrange(participant)
df$session <- rep(c("1", "2", "3"), 20)

# Sample rows to get unequal sizes per participant
df <- dplyr::sample_n(df, size = 53)

# Create the initial groups (to be collapsed)
df <- fold(
  data = df,
  k = 8,
  method = "n_dist",
  id_col = "participant"
)

# Ungroup the data frame
# Otherwise `collapse_groups()` would be
# applied to each fold separately!
df <- dplyr::ungroup(df)

# NOTE: Make sure to check the examples with `auto_tune`
# in the end, as this is where the magic lies

# Collapse to 3 groups with size balancing
# Creates new `.coll_groups` column
df_coll <- collapse_groups(
  data = df,
  n = 3,
  group_cols = ".folds",
  balance_size = TRUE # enabled by default
)

# Check balances
(coll_summary <- summarize_balances(
  data = df_coll,
  group_cols = ".coll_groups",
  cat_cols = 'answer',
  num_cols = c('score', 'age'),
  id_cols = 'participant'
))

# Get ranked balances
# NOTE: When we only have a single new group column
# we don't get ranks - but this is good to use
# when comparing multiple group columns!
# The scores are standard deviations across groups
ranked_balances(coll_summary)

# Collapse to 3 groups with size + *categorical* balancing
# We create 2 new `.coll_groups_1/2` columns
df_coll <- collapse_groups(
  data = df,
  n = 3,
  group_cols = ".folds",
  cat_cols = "answer",
  balance_size = TRUE,
  num_new_group_cols = 2
)

# Check balances
# To simplify the output, we only find the
# balance of the `answer` column
(coll_summary <- summarize_balances(
  data = df_coll,
  group_cols = paste0(".coll_groups_", 1:2),
  cat_cols = 'answer'
))

# Get ranked balances
# All scores are standard deviations across groups or (average) ranks
# Rows are ranked by most to least balanced
# (i.e. lowest average SD rank)
ranked_balances(coll_summary)

# Collapse to 3 groups with size + categorical + *numerical* balancing
# We create 2 new `.coll_groups_1/2` columns
df_coll <- collapse_groups(
  data = df,
  n = 3,
  group_cols = ".folds",
  cat_cols = "answer",
  num_cols = "score",
  balance_size = TRUE,
  num_new_group_cols = 2
)

# Check balances
(coll_summary <- summarize_balances(
  data = df_coll,
  group_cols = paste0(".coll_groups_", 1:2),
  cat_cols = 'answer',
  num_cols = 'score'
))

# Get ranked balances
# All scores are standard deviations across groups or (average) ranks
ranked_balances(coll_summary)

# Collapse to 3 groups with size and *ID* balancing
# We create 2 new `.coll_groups_1/2` columns
df_coll <- collapse_groups(
  data = df,
  n = 3,
  group_cols = ".folds",
  id_cols = "participant",
  balance_size = TRUE,
  num_new_group_cols = 2
)

# Check balances
# To simplify the output, we only find the
# balance of the `participant` column
(coll_summary <- summarize_balances(
  data = df_coll,
  group_cols = paste0(".coll_groups_", 1:2),
  id_cols = 'participant'
))

# Get ranked balances
# All scores are standard deviations across groups or (average) ranks
ranked_balances(coll_summary)

###################
#### Auto-tune ####

# As you might have seen, the balancing does not always
# perform as optimal as we might want or need
# To get a better balance, we can enable `auto_tune`
# which will create a larger set of collapsings
# and select the most balanced new group columns
# While it is not required, we recommend
# enabling parallelization

\dontrun{
# Uncomment for parallelization
# library(doParallel)
# doParallel::registerDoParallel(7) # use 7 cores

# Collapse to 3 groups with lots of balancing
# We enable `auto_tune` to get a more balanced set of columns
# We create 10 new `.coll_groups_1/2/...` columns
df_coll <- collapse_groups(
  data = df,
  n = 3,
  group_cols = ".folds",
  cat_cols = "answer",
  num_cols = "score",
  id_cols = "participant",
  balance_size = TRUE,
  num_new_group_cols = 10,
  auto_tune = TRUE,
  parallel = FALSE # Set to TRUE for parallelization!
)

# Check balances
# To simplify the output, we only find the
# balance of the `participant` column
(coll_summary <- summarize_balances(
  data = df_coll,
  group_cols = paste0(".coll_groups_", 1:10),
  cat_cols = "answer",
  num_cols = "score",
  id_cols = 'participant'
))

# Get ranked balances
# All scores are standard deviations across groups or (average) ranks
ranked_balances(coll_summary)

# Now we can choose the .coll_groups_* column(s)
# that we favor the balance of
# and move on with our lives!
}


}
\seealso{
\code{\link[groupdata2:fold]{fold()}} for creating balanced folds/groups.

\code{\link[groupdata2:partition]{partition()}} for creating balanced partitions.

Other grouping functions: 
\code{\link{all_groups_identical}()},
\code{\link{collapse_groups_by}},
\code{\link{fold}()},
\code{\link{group_factor}()},
\code{\link{group}()},
\code{\link{partition}()},
\code{\link{splt}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{grouping functions}
