% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GOC.R
\name{GOC}
\alias{GOC}
\alias{GOC,mpg-method}
\title{Produce a grains of connectivity model at multiple scales (patch-based or lattice GOC)}
\usage{
GOC(x, ...)

\S4method{GOC}{mpg}(
  x,
  nThresh = NULL,
  doThresh = NULL,
  weight = "lcpPerimWeight",
  verbose = 0,
  ...
)
}
\arguments{
\item{x}{A \code{mpg} object produced by \code{\link{MPG}}.
For lattice GOC \code{MPG} must be run with patch set as an integer value.}

\item{...}{Additional arguments (not used).}

\item{nThresh}{Optional. An integer giving the number of thresholds (or scales)
at which to create GOC models. Thresholds are selected to produce
a maximum number of unique grains (i.e., models).
\code{nThresh} thresholds are also approximately evenly spread
between 0 and the threshold at which all patches or focal points
on the landscape are connected. This is a simple way to get
a representative subset of all possible GOC models.
Provide either \code{nThresh} or \code{doThresh} not both.}

\item{doThresh}{Optional. A vector giving the link thresholds at which to create GOC models.
Use \code{\link{threshold}} to identify thresholds of interest.
Provide either \code{nThresh} or \code{doThresh} not both.}

\item{weight}{A string giving the link weight or attribute to use for threshold.
\code{"lcpPerimWeight"} uses the accumulated resistance or least-cost
path distance from the perimeters of patches as the link weight.
\code{"eucPerimWeight"} use the Euclidean distance from the
perimeters of patches as the link weight.}

\item{verbose}{Set \code{verbose=0} for no progress information to console.}
}
\value{
A \code{\link[=goc-class]{goc}} object.
}
\description{
Produce a grains of connectivity (GOC) model at multiple scales (resistance thresholds)
by scalar analysis.
Patch-based or lattice GOC modelling can be done with this function.
}
\details{
Grain or scalar analysis of connectivity may be appropriate for a variety of purposes, not
limited to visualization and improving connectivity estimates for highly-mobile organisms.
See Galpern \emph{et al.} (2012), Galpern & Manseau (2013a, 2013b) for applications
and review of these capabilities.
}
\note{
Researchers should consider whether the use of a patch-based GOC or a lattice
GOC model is appropriate based on the patch-dependency of the organism under study.
Patch-based models make most sense when animals are restricted to, or dependent on,
a resource patch.
Lattice models can be used as a generalized and functional approach to scaling
resistance surfaces.

See \code{\link{MPG}} for warning related to areal measurements.
}
\examples{
## Load raster landscape
tiny <- raster::raster(system.file("extdata/tiny.asc", package = "grainscape"))

## Create a resistance surface from a raster using an is-becomes reclassification
tinyCost <- raster::reclassify(tiny, rcl = cbind(c(1, 2, 3, 4), c(1, 5, 10, 12)))

## Produce a patch-based MPG where patches are resistance features=1
tinyPatchMPG <- MPG(cost = tinyCost, patch = tinyCost == 1)

## Extract a representative subset of 5 grains of connectivity
tinyPatchGOC <- GOC(tinyPatchMPG, nThresh = 5)

## Examine the properties of the GOC graph of grain 3 of 5
graphdf(grain(tinyPatchGOC, whichThresh = 3))

## Extract grains of connectivity
## representation of the finest grain and three others
## by giving thresholds in link weights (doThresh)
tinyPatchGOC <- GOC(tinyPatchMPG, doThresh = c(0, 20, 40))
}
\references{
Fall, A., M.-J. Fortin, M. Manseau, D. O'Brien. (2007) Spatial graphs:
Principles and applications for habitat connectivity. Ecosystems 10:448:461.

Galpern, P., M. Manseau. (2013a) Finding the functional grain: comparing methods
for scaling resistance surfaces. Landscape Ecology 28:1269-1291.

Galpern, P., M. Manseau. (2013b) Modelling the influence of landscape connectivity
on animal distribution: a functional grain approach. Ecography 36:1004-1016.

Galpern, P., M. Manseau, A. Fall. (2011) Patch-based graphs of landscape connectivity:
a guide to construction, analysis, and application for conservation.
Biological Conservation 144:44-55.

Galpern, P., M. Manseau, P.J. Wilson. (2012) Grains of connectivity: analysis
at multiple spatial scales in landscape genetics. Molecular Ecology 21:3996-4009.
}
\seealso{
\code{\link{MPG}}, \code{\link{grain}},
         \code{\link{distance}}, \code{\link{point}}
}
\author{
Paul Galpern
}
