% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateGMVAR.R
\name{simulateGMVAR}
\alias{simulateGMVAR}
\title{Simulate from GMVAR process}
\usage{
simulateGMVAR(
  gmvar,
  nsimu,
  init_values = NULL,
  ntimes = 1,
  drop = TRUE,
  seed = NULL,
  girf_pars = NULL
)
}
\arguments{
\item{gmvar}{an object of class \code{'gmvar'} created with \code{fitGMVAR} or \code{GMVAR}.}

\item{nsimu}{number of observations to be simulated.}

\item{init_values}{a size \eqn{(pxd)} matrix specifying the initial values to be used in the simulation, where
d is the number of time series in the system.
The \strong{last} row will be used as initial values for the first lag, the second last row for second lag etc. If not
specified, initial values will be drawn from the stationary distribution of the process.}

\item{ntimes}{how many sets of simulations should be performed?}

\item{drop}{if \code{TRUE} (default) then the components of the returned list are coerced to lower dimension if \code{ntimes==1}, i.e.,
\code{$sample} and \code{$mixing_weights} will be matrices, and \code{$component} will be vector.}

\item{seed}{set seed for the random number generator?}

\item{girf_pars}{This argument is used internally in the estimation of generalized impulse response functions (see \code{?GIRF}). You
should ignore it.}
}
\value{
If \code{drop==TRUE} and \code{ntimes==1} (default): \code{$sample}, \code{$component}, and \code{$mixing_weights} are matrices.
  Otherwise, returns a list with...
  \describe{
    \item{\code{$sample}}{a size (\code{nsimu}\eqn{ x d x }\code{ntimes}) array containing the samples: the dimension \code{[t, , ]} is
     the time index, the dimension \code{[, d, ]} indicates the marginal time series, and the dimension \code{[, , i]} indicates
     the i:th set of simulations.}
    \item{\code{$component}}{a size (\code{nsimu}\eqn{ x }\code{ntimes}) matrix containing the information from which mixture component each
     value was generated from.}
    \item{\code{$mixing_weights}}{a size (\code{nsimu}\eqn{ x M x }\code{ntimes}) array containing the mixing weights corresponding to the
     sample: the dimension \code{[t, , ]} is the time index, the dimension \code{[, m, ]} indicates the regime, and the dimension
     \code{[, , i]} indicates the i:th set of simulations.}
  }
}
\description{
\code{simulateGMVAR} simulates observations from a GMVAR process.
}
\details{
The argument \code{ntimes} is intended for forecasting: a GMVAR process can be forecasted by simulating its possible future values.
 One can easily perform a large number simulations and calculate the sample quantiles from the simulated values to obtain prediction
 intervals (see the forecasting example).
}
\examples{
 \donttest{
 # These examples use the data 'eurusd' which comes with the
 # package, but in a scaled form.
 data <- cbind(10*eurusd[,1], 100*eurusd[,2])
 colnames(data) <- colnames(eurusd)

 # GMVAR(1,2), d=2 process, initial values from the stationary
 # distribution
 params122 <- c(0.623, -0.129, 0.959, 0.089, -0.006, 1.006, 1.746,
  0.804, 5.804, 3.245, 7.913, 0.952, -0.037, -0.019, 0.943, 6.926,
  3.982, 12.135, 0.789)
 mod122 <- GMVAR(p=1, M=2, d=2, params=params122)
 set.seed(1)
 sim122 <- simulateGMVAR(mod122, nsimu=500)
 plot.ts(sim122$sample)
 ts.plot(sim122$mixing_weights, col=c("blue", "red"), lty=2)
 plot(sim122$component, type="l")

 # Structural GMVAR(2, 2), d=2 model identified with sign-constraints:
 params222s <- c(-11.964, 155.024, 11.636, 124.988, 1.314, 0.145, 0.094, 1.292,
   -0.389, -0.07, -0.109, -0.281, 1.248, 0.077, -0.04, 1.266, -0.272, -0.074,
   0.034, -0.313, 0.903, 0.718, -0.324, 2.079, 7.00, 1.44, 0.742)
 W_222 <- matrix(c(1, 1, -1, 1), nrow=2, byrow=FALSE)
 mod222s <- GMVAR(data, p=2, M=2, params=params222s, parametrization="mean",
   structural_pars=list(W=W_222))
 sim222s <- simulateGMVAR(mod222s, nsimu=100)
 plot.ts(sim222s$sample)

 ## FORECASTING EXAMPLE ##
 # Forecast 5-steps-ahead, 10000 sets of simulations with initial
 # values from the data:
 # GMVAR(2,2), d=2 model with mean-parametrization:
 params222 <- c(-11.904, 154.684, 1.314, 0.145, 0.094, 1.292, -0.389,
  -0.070, -0.109, -0.281, 0.920, -0.025, 4.839, 11.633, 124.983, 1.248,
   0.077, -0.040, 1.266, -0.272, -0.074, 0.034, -0.313, 5.855, 3.570,
   9.838, 0.740)
 mod222 <- GMVAR(data, p=2, M=2, params=params222, parametrization="mean")
 sim222 <- simulateGMVAR(mod222, nsimu=5, ntimes=10000)

 # Point forecast + 95\% prediction intervals:
 apply(sim222$sample, 1:2, quantile, probs=c(0.025, 0.5, 0.972))

 # Similar forecast for the mixing weights:
 apply(sim222$mixing_weights, 1:2, quantile, probs=c(0.025, 0.5, 0.972))


 # GMVAR(2,2), d=2 model with AR parameters restricted to be
 # the same for both regimes, custom inital values:
 C_mat <- rbind(diag(2*2^2), diag(2*2^2))
 params222c <- c(1.031, 2.356, 1.786, 3.000, 1.250, 0.060, 0.036,
  1.335, -0.290, -0.083, -0.047, -0.356, 0.934, -0.152, 5.201, 5.883,
  3.560, 9.799, 0.368)
 mod222c <- GMVAR(data, p=2, M=2, params=params222c, constraints=C_mat)
 sim222c <- simulateGMVAR(mod222c, nsimu=100,
              init_values=matrix(c(30, 30, 80, 80), nrow=2))
 plot.ts(sim222c$sample)
 ts.plot(sim222c$mixing_weights, col=c("blue", "red"), lty=2)
 plot(sim222c$component, type="l")
 }
}
\references{
\itemize{
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Virolainen S. 2020. Structural Gaussian mixture vector autoregressive model. Unpublished working
     paper, available as arXiv:2007.04713.
 }
}
\seealso{
\code{\link{fitGMVAR}}, \code{\link{GMVAR}}, \code{\link{diagnostic_plot}}, \code{\link{predict.gmvar}},
 \code{\link{profile_logliks}}, \code{\link{quantile_residual_tests}}, \code{\link{GIRF}}
}
