#' @import graphics
#'
#' @title plot method for class 'gmvarpred' objects
#'
#' @description \code{plot.gmvarpred} is plot method for gmvarpred objects.
#'
#' @param x object of class \code{'gmvarpred'} generated by \code{predict.gmvar}.
#' @param nt a positive integer specifying the number of observations to be plotted
#'   along with the prediction. Default is \code{round(nrow(data)*0.2)}.
#' @param add_grid should grid be added to the plots?
#' @param ... arguments passed to \code{grid} which plots grid to the figure.
#' @details This method is used plot forecasts of GMVAR processes
#' @inherit in_paramspace_int references
#' @export

plot.gmvarpred <- function(x, ..., nt, add_grid=TRUE) {
  old_par <- par(no.readonly=TRUE)
  on.exit(par(old_par))
  gmvarpred <- x
  data <- as.ts(gmvarpred$gmvar$data)
  n_obs <- nrow(data)
  d <- ncol(data)
  q <- gmvarpred$q
  if(missing(nt)) {
    nt <- round(nrow(data)*0.2)
  } else {
    stopifnot(nt > 0 & nt %% 1 == 0)
    if(nt > nrow(data)) {
      warning("nt > nrow(data), using nt = nrow(data)")
      nt <- nrow(data)
    }
  }
  par(mfrow=c(d, 1), mar=c(2.5, 2.5, 2.1, 1))
  make_ts <- function(x) ts(rbind(data[n_obs,], x), start=time(data)[n_obs], frequency=frequency(data))
  ts_pred <- make_ts(gmvarpred$pred)
  ts_dat <- ts(data[(n_obs - nt + 1):n_obs,], start=time(data)[n_obs - nt + 1], frequency=frequency(data))
  t0 <- time(ts_pred)
  ts_names <- attributes(data)$dimnames[[2]]

  if(gmvarpred$pi_type == "none") {
    all_val <- lapply(1:d, function(j) c(ts_dat[,j], ts_pred[,j]))
  } else {
    all_val <- lapply(1:d, function(j) c(ts_dat[,j], ts_pred[,j], simplify2array(gmvarpred$pred_ints, higher=TRUE)[, j, ]))
  }
  plot_pred <- function(j) {
    ts.plot(ts_dat[,j], ts_pred[,j], gpars=list(col=c("black", "blue"), lty=1:2,
                                                ylim=c(round(min(all_val[[j]])) - 1,
                                                       round(max(all_val[[j]])) + 1), main=ts_names[j]))
    if(add_grid) grid(...)
  }

  if(gmvarpred$pi_type == "two-sided") {
    ts_lowers <- lapply(1:(length(q)/2), function(i1) make_ts(gmvarpred$pred_ints[[i1]]))
    ts_uppers <- lapply((length(q)/2 + 1):length(q), function(i1) make_ts(gmvarpred$pred_ints[[i1]]))

  } else if(gmvarpred$pi_type == "upper") {
    ts_min <- vapply(1:d, function(j) rep(round(min(all_val[[j]])) - 3, times=nrow(ts_pred)), numeric(nrow(ts_pred)))
    ts_lowers <- lapply(1:length(q), function(i1) make_ts(ts_min)[-1,])
    ts_uppers <- lapply(1:length(q), function(i1) make_ts(gmvarpred$pred_ints[[i1]]))

  } else if(gmvarpred$pi_type == "lower") {
    ts_max <- vapply(1:d, function(j) rep(round(max(all_val[[j]])) + 3, times=nrow(ts_pred)), numeric(nrow(ts_pred)))
    ts_lowers <- lapply(1:length(q), function(i1) make_ts(gmvarpred$pred_ints[[i1]]))
    ts_uppers <- lapply(1:length(q), function(i1) make_ts(ts_max)[-1,])
  }

  for(i1 in 1:d) {
    plot_pred(i1)
    if(gmvarpred$pi_type %in% c("two-sided", "upper", "lower")) {
      for(i2 in 1:length(gmvarpred$pi)) {
        polygon(x=c(t0, rev(t0)), y=c(ts_lowers[[i2]][,i1], rev(ts_pred[,i1])), col=grDevices::rgb(0, 0, 1, 0.2), border=NA)
        polygon(x=c(t0, rev(t0)), y=c(ts_uppers[[i2]][,i1], rev(ts_pred[,i1])), col=grDevices::rgb(0, 0, 1, 0.2), border=NA)
      }
    }
  }
  invisible(gmvarpred)
}

#' @import graphics
#' @describeIn GMVAR plot method for class 'gmvar'
#' @param x object of class \code{'gmvar'} generated by \code{fitGMVAR} or \code{GMVAR}.
#' @param ... currectly not used.
#' @export

plot.gmvar <- function(x, ...) {
  gmvar <- x
  check_null_data(gmvar)
  data <- as.ts(gmvar$data)
  n_obs <- nrow(data)
  p <- gmvar$model$p
  M <- gmvar$model$M
  d <- ncol(data)
  ts_mw <- ts(rbind(matrix(NA, nrow=p, ncol=M), gmvar$mixing_weights),
              start=start(data), frequency=frequency(data)) # First p observations are starting values

  old_par <- par(no.readonly=TRUE)
  on.exit(par(old_par))
  graphics::par(mfrow=c(2, 1), mar=c(2.5, 2.5, 2.1, 1))
  colpal_ts <- grDevices::colorRampPalette(c("darkgreen", "darkblue", "darkmagenta", "red3"))(d)
  colpal_mw <- grDevices::colorRampPalette(c("blue", "turquoise1", "green", "red"))(M)
  names_ts <- colnames(data)
  names_mw <- paste0("mix.comp.", 1:M)
  draw_legend <- function(nams, cols) {
    legend("topleft", legend=nams, bty="n", col=cols, lty=1, lwd=2, text.font=2, cex=0.6, x.intersp=0.5, y.intersp=1)
  }

  ts.plot(data, gpars=list(main="Time series", col=colpal_ts, lty=1:d))
  draw_legend(names_ts, cols=colpal_ts)
  ts.plot(ts_mw, gpars=list(main="Mixing weights", ylim=c(0, 1), col=colpal_mw, lty=2))
  draw_legend(names_mw, cols=colpal_mw)
}



#' @import graphics
#' @describeIn quantile_residual_tests Plot p-values of the autocorrelation and conditional
#'  heteroskedasticity tests.
#' @inheritParams print.qrtest
#' @export

plot.qrtest <- function(x, ...) {
  old_par <- par(no.readonly=TRUE)
  on.exit(par(old_par))
  qrtest <- x
  par(mfrow=c(1, 2), mar=c(5.1, 3.1, 3.1, 1.1))

  plot_pvalues <- function(which_ones) { # ac_res, ch_res
    res <- qrtest[[which(names(qrtest) == which_ones)]]
    pvals <- res$test_results$p_val
    seq_pvals <- seq_along(pvals)
    plot(pvals, ylim=c(0, 1), xlim=c(min(seq_pvals) - 0.2, max(seq_pvals) + 0.2), ylab="", xlab="lags",
         main=ifelse(which_ones == "ac_res", "Autocorrelation", "Cond. h.skedasticity"),
         xaxt="n", yaxt="n", pch=16, col="blue")
    axis(side=1, at=seq_pvals, labels=res$test_results$lags)
    levels <- c(0.01, 0.05, 0.10, seq(from=0.20, to=1.00, by=0.20))
    axis(side=2, at=levels, las=1, cex.axis=0.8)
    abline(h=0, lwd=2)
    abline(h=c(0.01, 0.05, 0.10, 1.00), lty=2, col=c(rep("red", 3), "green4"))
    segments(x0=seq_pvals, y0=0, y1=pvals, x1=seq_pvals, ...)
    points(pvals)
  }

  plot_pvalues("ac_res")
  plot_pvalues("ch_res")
}

