#' @title Function factory for value formatting
#'
#' @description \code{format_valuef} is a function factory for
#'   formatting values with certain number of digits.
#'
#' @param digits number of digits to use
#' @return returns a function that takes an atomic vector as argument
#'   and returns it formatted to character with \code{digits} decimals.

format_valuef <- function(digits) {
  function(x) format(round(x, digits), nsmall=digits)
}


#' @describeIn GMVAR print method
#' @inheritParams plot.gmvar
#' @param digits number of digits to be printed
#' @param summary_print if set to \code{TRUE} then the print
#'   will include log-likelihood and information criteria values.
#' @export

print.gmvar <- function(x, ..., digits=2, summary_print=FALSE) {
  gmvar <- x
  stopifnot(digits >= 0 & digits%%1 == 0)
  format_value <- format_valuef(digits)
  p <- gmvar$model$p
  M <- gmvar$model$M
  d <- gmvar$model$d
  IC <- gmvar$IC
  constraints <- gmvar$model$constraints
  all_mu <- round(get_regime_means(gmvar), digits)
  params <- gmvar$params
  if(gmvar$model$parametrization == "mean") {
    params <- change_parametrization(p=p, M=M, d=d, params=params, constraints=constraints,
                                     change_to="intercept")
  }
  pars <- reform_constrained_pars(p=p, M=M, d=d, params=params, constraints=constraints)
  all_phi0 <- pick_phi0(p=p, M=M, d=d, params=pars)
  all_A <- pick_allA(p=p, M=M, d=d, params=pars)
  all_Omega <- pick_Omegas(p=p, M=M, d=d, params=pars)
  alphas <- pick_alphas(p=p, M=M, d=d, params=pars)
  cat("Model:\n")
  cat(paste0("p = ", p, ", M = ", M, ","),
      ifelse(gmvar$model$conditional, "conditional,", "exact,"),
      ifelse(gmvar$model$parametrization=="mean", "mean parametrization,", "intercept parametrization,"),
      ifelse(is.null(constraints), "no constraints", "linear constraints employed"), "\n")
  cat("\n")

  if(summary_print == TRUE) {
    all_boldA_eigens <- get_boldA_eigens(gmvar)
    cat(paste("log-likelihood:", format_value(gmvar$loglik)), "\n")
    cat(paste("AIC: ", format_value(IC$AIC)), "\n")
    cat(paste("HQIC:", format_value(IC$HQIC)), "\n")
    cat(paste("BIC: ", format_value(IC$BIC)), "\n\n")
  }

  plus <- c("+", rep(" ", d-1))
  Y <- paste0("Y", 1:d)
  tmp_names <- paste0("tmp", 1:(p*(d + 2) + d + 2))

  for(m in seq_len(M)) {
    count <- 1
    cat(paste("Regime", m), "\n")
    if(summary_print == TRUE) cat(paste("Modulus of 'bold A' eigenvalues: ", paste0(format_value(all_boldA_eigens[[m]]), collapse=", ")),"\n")
    cat(paste("Mixing weight:", format_value(alphas[m])), "\n")
    cat("Regime means:", paste0(format_value(all_mu[,m]), collapse=", "), "\n\n")
    df <- data.frame(Y=Y,
                     eq=c("=", rep(" ", d-1)),
                     eq=rep("[", d),
                     phi0=format_value(all_phi0[, m, drop=FALSE]),
                     eq=rep("]", d),
                     plus)
    for(i1 in seq_len(p)) {
      Amp_colnames <- c(paste0("A", i1), tmp_names[count:(count + d - 1 - 1)]); count <- count + d - 1
      df[, tmp_names[count]] <- rep("[", d); count <- count + 1
      df[, Amp_colnames] <- format_value(all_A[, ,i1 , m])
      df[, tmp_names[count]] <- rep("]", d); count <- count + 1
      df[, tmp_names[count]] <- paste0(Y, ".l", i1); count <- count + 1
      df <- cbind(df, plus)
    }
    df[, tmp_names[p*(d + 2) + 1]] <- rep("[", d)
    df[, c("Omega", tmp_names[(p*(d + 2) + 2):(p*(d + 2) + d)])] <- format_value(all_Omega[, , m])
    df[, tmp_names[p*(d + 2) + d + 1]] <- rep("]", d)
    df[, "1/2"] <- rep(" ", d)
    df[, tmp_names[p*(d + 2) + d + 2]] <- paste0("eps", 1:d)
    names_to_omit <- unlist(lapply(c("plus", "eq", tmp_names), function(nam) grep(nam, colnames(df))))
    colnames(df)[names_to_omit] <- " "
    print(df)
    cat("\n")
    if(summary_print == TRUE) {
      cat("Error term correlation matrix:\n")
      print(cov2cor(all_Omega[, , m]), digits=digits)
      cat("\n")
    }
  }
  invisible(gmvar)
}


#' @title Summary print method from objects of class 'gmvarsum'
#'
#' @description \code{print.gmvarsum} is a print method for object \code{'gmvarsum'} generated
#'   by \code{summary.gmvar()}.
#'
#' @param x object of class 'gmvarsum' generated by \code{summary.gmvar()}.
#' @param ... currectly not used.
#' @examples
#' # This example uses the data 'eurusd' which comes with the
#' # package, but in a scaled form.
#' data <- cbind(10*eurusd[,1], 100*eurusd[,2])
#' colnames(data) <- colnames(eurusd)
#'
#' # GMVAR(2,2), d=2 model
#' params222 <- c(-11.904, 154.684, 1.314, 0.145, 0.094, 1.292, -0.389,
#'  -0.070, -0.109, -0.281, 0.920, -0.025, 4.839, 11.633, 124.983, 1.248,
#'   0.077, -0.040, 1.266, -0.272, -0.074, 0.034, -0.313, 5.855, 3.570,
#'   9.838, 0.740)
#' mod222 <- GMVAR(data, p=2, M=2, params=params222, parametrization="mean")
#' sumry222 <- summary(mod222)
#' print(sumry222)
#' @export

print.gmvarsum <- function(x, ...) {
  gmvarsum <- x
  print.gmvar(gmvarsum$gmvar, summary_print=TRUE)
  if(!is.null(gmvarsum$qrtest)) {
    cat("_____________________________________\n")
    cat("Quantile residual tests based on data\n\n")
    print.qrtest(gmvarsum$qrtest)
  }
  invisible(gmvarsum)
}


#' @title print method for class 'gmvarpred' objects
#'
#' @description \code{print.gmvarpred} is print method for object generated
#'  by \code{predict.gmvar()}.
#'
#' @inheritParams plot.gmvarpred
#' @param digits number of decimals to print
#' @param ... currectly not used.
#' @examples
#' # This example uses the data 'eurusd' which comes with the
#' # package, but in a scaled form.
#' data <- cbind(10*eurusd[,1], 100*eurusd[,2])
#' colnames(data) <- colnames(eurusd)
#'
#' # GMVAR(2,2), d=2 model
#' params222 <- c(-11.904, 154.684, 1.314, 0.145, 0.094, 1.292, -0.389,
#'  -0.070, -0.109, -0.281, 0.920, -0.025, 4.839, 11.633, 124.983, 1.248,
#'   0.077, -0.040, 1.266, -0.272, -0.074, 0.034, -0.313, 5.855, 3.570,
#'   9.838, 0.740)
#' mod222 <- GMVAR(data, p=2, M=2, params=params222, parametrization="mean")
#' pred222 <- predict(mod222, n_ahead=3, plot_res=FALSE)
#' print(pred222)
#' print(pred222, digits=3)
#' @export

print.gmvarpred <- function(x, ..., digits=2) {
  gmvarpred <- x
  stopifnot(digits >= 0 & digits%%1 == 0)
  format_value <- format_valuef(digits)

  if(gmvarpred$pred_type == "cond_mean") {
    cat("One-step-ahead prediction by exact conditional mean, no confidence intervals\n")
    cat("Forecast:", paste0(format_value(gmvarpred$pred), collapse=", "), "\n")

  } else if(gmvarpred$ci_type == "none") {
    cat(paste0("Prediction by ", gmvarpred$pred_type, ", no confidence intervals"), "\n")
    cat(paste0("Forecast ", gmvarpred$n_ahead, " steps ahead, based on ", gmvarpred$n_simu, " simulations\n"))
    print(gmvarpred$pred)

  } else {
    cat(paste0("Prediction by ", gmvarpred$pred_type, ", ", gmvarpred$ci_type,
               " confidence intervals with levels ", paste(gmvarpred$ci, collapse=", "), ""), "\n")
    cat(paste0("Forecast ", gmvarpred$n_ahead, " steps ahead, based on ", gmvarpred$n_simu, " simulations\n"))

    cat("\n")
    q <- gmvarpred$q
    conf_ints <- gmvarpred$conf_ints
    pred <- gmvarpred$pred
    pred_type <- gmvarpred$pred_type
    for(i1 in seq_len(gmvarpred$gmvar$model$d)) {
      cat(paste0("Component ", i1, ":"), "\n")
      df <- as.data.frame(lapply(1:length(conf_ints), function(i2) format_value(conf_ints[[i2]][,i1])))
      names(df) <- q
      df[, pred_type] <- format_value(pred[,i1])
      if(gmvarpred$ci_type == "two-sided") {
        new_order <- as.character(c(q[1:(length(q)/2)], pred_type, q[(length(q)/2 + 1):length(q)]))
      } else if(gmvarpred$ci_type == "upper") {
        new_order <- as.character(c(pred_type, q))
      } else {
        new_order <- names(df)
      }
      print(df[, new_order])
      cat("\n")
    }
  }
  invisible(gmvarpred)
}


#' @describeIn quantile_residual_tests print method for class 'qrtest'
#' @inheritParams print.gmvarpred
#' @param x object of class \code{'qrtest'} generated by the function \code{quantile_residual_tests()}.
#' @param ... currectly not used.
#' @export

print.qrtest <- function(x, ..., digits=3) {
  qrtest <- x
  format_value <- format_valuef(digits)
  format_lag <- format_valuef(0)
  cat(paste("Normality test p_value:", format_value(qrtest$norm_res$p_val)), "\n\n")

  cat("Autocorrelation tests:\nlags | p_value\n")
  for(i1 in seq_along(qrtest$ac_res$test_results$lags)) {
    if(qrtest$ac_res$test_results$lags[i1] < 10) {
      cat(" ", format_lag(qrtest$ac_res$test_results$lags[i1]), " | ", format_value(qrtest$ac_res$test_results$p_val[i1]), "\n")
    } else {
      cat(" ", format_lag(qrtest$ac_res$test_results$lags[i1]), "| ", format_value(qrtest$ac_res$test_results$p_val[i1]), "\n")
    }
  }
  cat("\nConditional hetetoskedasticity tests:\nlags | p_value\n")
  for(i1 in seq_along(qrtest$ch_res$test_results$lags)) {
    if(qrtest$ch_res$test_results$lags[i1] < 10) {
      cat(" ", format_lag(qrtest$ch_res$test_results$lags[i1]), " | ", format_value(qrtest$ch_res$test_results$p_val[i1]), "\n")
    } else {
      cat(" ", format_lag(qrtest$ch_res$test_results$lags[i1]), "| ", format_value(qrtest$ch_res$test_results$p_val[i1]), "\n")
    }
  }
  invisible(qrtest)
}

