% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{dbeta_}
\alias{dbeta_}
\title{Density of the Beta Distribution (gamma, delta+1 Parameterization)}
\usage{
dbeta_(x, gamma, delta, log_prob = FALSE)
}
\arguments{
\item{x}{Vector of quantiles (values between 0 and 1).}

\item{gamma}{First shape parameter (\code{shape1}), \eqn{\gamma > 0}. Can be a
scalar or a vector. Default: 1.0.}

\item{delta}{Second shape parameter is \code{delta + 1} (\code{shape2}), requires
\eqn{\delta \ge 0} so that \code{shape2 >= 1}. Can be a scalar or a vector.
Default: 0.0 (leading to \code{shape2 = 1}).}

\item{log_prob}{Logical; if \code{TRUE}, the logarithm of the density is
returned (\eqn{\log(f(x))}). Default: \code{FALSE}.}
}
\value{
A vector of density values (\eqn{f(x)}) or log-density values
(\eqn{\log(f(x))}). The length of the result is determined by the recycling
rule applied to the arguments (\code{x}, \code{gamma}, \code{delta}).
Returns \code{0} (or \code{-Inf} if \code{log_prob = TRUE}) for \code{x}
outside the interval (0, 1), or \code{NaN} if parameters are invalid
(e.g., \code{gamma <= 0}, \code{delta < 0}).
}
\description{
Computes the probability density function (PDF) for the standard Beta
distribution, using a parameterization common in generalized distribution
families. The distribution is parameterized by \code{gamma} (\eqn{\gamma}) and
\code{delta} (\eqn{\delta}), corresponding to the standard Beta distribution
with shape parameters \code{shape1 = gamma} and \code{shape2 = delta + 1}.
The distribution is defined on the interval (0, 1).
}
\details{
The probability density function (PDF) calculated by this function corresponds
to a standard Beta distribution \eqn{Beta(\gamma, \delta+1)}:
\deqn{
f(x; \gamma, \delta) = \frac{x^{\gamma-1} (1-x)^{(\delta+1)-1}}{B(\gamma, \delta+1)} = \frac{x^{\gamma-1} (1-x)^{\delta}}{B(\gamma, \delta+1)}
}
for \eqn{0 < x < 1}, where \eqn{B(a,b)} is the Beta function
(\code{\link[base]{beta}}).

This specific parameterization arises as a special case of the five-parameter
Generalized Kumaraswamy (GKw) distribution (\code{\link{dgkw}}) obtained
by setting the parameters \eqn{\alpha = 1}, \eqn{\beta = 1}, and \eqn{\lambda = 1}.
It is therefore equivalent to the McDonald (Mc)/Beta Power distribution
(\code{\link{dmc}}) with \eqn{\lambda = 1}.

Note the difference in the second parameter compared to \code{\link[stats]{dbeta}},
where \code{dbeta(x, shape1, shape2)} uses \code{shape2} directly. Here,
\code{shape1 = gamma} and \code{shape2 = delta + 1}.
}
\examples{
\donttest{
# Example values
x_vals <- c(0.2, 0.5, 0.8)
gamma_par <- 2.0 # Corresponds to shape1
delta_par <- 3.0 # Corresponds to shape2 - 1
shape1 <- gamma_par
shape2 <- delta_par + 1

# Calculate density using dbeta_
densities <- dbeta_(x_vals, gamma_par, delta_par)
print(densities)

# Compare with stats::dbeta
densities_stats <- stats::dbeta(x_vals, shape1 = shape1, shape2 = shape2)
print(paste("Max difference vs stats::dbeta:", max(abs(densities - densities_stats))))

# Compare with dgkw setting alpha=1, beta=1, lambda=1
densities_gkw <- dgkw(x_vals, alpha = 1.0, beta = 1.0, gamma = gamma_par,
                      delta = delta_par, lambda = 1.0)
print(paste("Max difference vs dgkw:", max(abs(densities - densities_gkw))))

# Compare with dmc setting lambda=1
densities_mc <- dmc(x_vals, gamma = gamma_par, delta = delta_par, lambda = 1.0)
print(paste("Max difference vs dmc:", max(abs(densities - densities_mc))))

# Calculate log-density
log_densities <- dbeta_(x_vals, gamma_par, delta_par, log_prob = TRUE)
print(log_densities)
print(stats::dbeta(x_vals, shape1 = shape1, shape2 = shape2, log = TRUE))

# Plot the density
curve_x <- seq(0.001, 0.999, length.out = 200)
curve_y <- dbeta_(curve_x, gamma = 2, delta = 3) # Beta(2, 4)
plot(curve_x, curve_y, type = "l", main = "Beta(2, 4) Density via dbeta_",
     xlab = "x", ylab = "f(x)", col = "blue")
curve(stats::dbeta(x, 2, 4), add=TRUE, col="red", lty=2)
legend("topright", legend=c("dbeta_(gamma=2, delta=3)", "stats::dbeta(shape1=2, shape2=4)"),
       col=c("blue", "red"), lty=c(1,2), bty="n")

}

}
\references{
Johnson, N. L., Kotz, S., & Balakrishnan, N. (1995). \emph{Continuous Univariate
Distributions, Volume 2} (2nd ed.). Wiley.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},
}
\seealso{
\code{\link[stats]{dbeta}} (standard R implementation),
\code{\link{dgkw}} (parent distribution density),
\code{\link{dmc}} (McDonald/Beta Power density),
\code{pbeta_}, \code{qbeta_}, \code{rbeta_} (other functions for this parameterization, if they exist).
}
\author{
Lopes, J. E.
}
\keyword{beta}
\keyword{density}
\keyword{distribution}
