tar_git_add <- function(files, repo, spinner = TRUE) {
  processx::run(
    command = tar_git_binary(),
    args = c("add", files),
    wd = repo,
    echo = FALSE,
    spinner = spinner
  )
}

tar_git_branch_checkout <- function(branch, repo, force) {
  args <- c("checkout", if_any(force, "--force", character(0)), branch)
  processx::run(
    command = tar_git_binary(),
    args = args,
    wd = repo,
    echo = FALSE
  )
}

tar_git_branch_create <- function(branch, repo) {
  processx::run(
    command = tar_git_binary(),
    args = c("branch", branch),
    wd = repo,
    echo = FALSE
  )
}

tar_git_commit <- function(message, repo, spinner = TRUE) {
  processx::run(
    command = tar_git_binary(),
    args = c("commit", "--message", message),
    wd = repo,
    echo = FALSE,
    spinner = spinner
  )
}

tar_git_commit_all <- function(message, repo, spinner = TRUE) {
  processx::run(
    command = tar_git_binary(),
    args = c("commit", "--all", "--message", message),
    wd = repo,
    echo = FALSE,
    spinner = spinner
  )
}

tar_git_init_repo <- function(path) {
  processx::run(command = tar_git_binary(), args = "init", wd = path)
}

tar_git_pack_refs <- function(repo, spinner = TRUE) {
  processx::run(
    command = tar_git_binary(),
    args = c("pack-refs", "--all"),
    wd = repo,
    echo = FALSE,
    spinner = spinner
  )
}

tar_git_reset_hard <- function(repo, spinner = TRUE) {
  processx::run(
    command = tar_git_binary(),
    args = c("reset", "--hard", "HEAD"),
    wd = repo,
    echo = FALSE,
    spinner = spinner
  )
}

# Get a data snapshot branch name from a code commit hash.
# The branch name refers to the code commit with "code="
# followed by the commit hash.
tar_git_branch_snapshot <- function(commit) {
  sprintf("code=%s", commit)
}

# Get a code commit hash from a data snapshot branch name.
tar_git_commit_code <- function(branch) {
  gsub(pattern = "^code=", replacement = "", x = branch)
}

tar_git_binary <- function() {
  out <- Sys.getenv("TAR_GIT", unset = Sys.which("git"))
  msg <- paste(
    "no existing Git installation found.",
    "Install from https://git-scm.com/downloads.",
    "If you already installed Git",
    "set the TAR_GIT environment variable to the path of the",
    "Git executable. Functions usethis::edit_r_environ() and",
    "Sys.setenv() can help."
  )
  targets::tar_assert_nzchar(out, msg = msg)
  targets::tar_assert_path(out, msg = msg)
  out
}

tar_git_repo_exists <- function(repo) {
  file.exists(file.path(repo, ".git"))
}

tar_git_gitignore_stash <- function(repo) {
  from <- file.path(repo, ".gitignore")
  to <- file.path(repo, ".gittargets_gitignore")
  if (file.exists(from)) {
    file.rename(from = from, to = to)
    writeLines(tar_git_gitignore_lines(), from)
  }
  invisible()
}

tar_git_gitignore_restore <- function(repo) {
  from <- file.path(repo, ".gittargets_gitignore")
  to <- file.path(repo, ".gitignore")
  has_lines <- file.exists(to) &&
    identical(readLines(to), tar_git_gitignore_lines())
  restore <- file.exists(from) && (!file.exists(to) || has_lines)
  if (restore) {
    file.rename(from = from, to = to)
  }
  invisible()
}

tar_git_gitignore_unstash <- function(repo) {
  from <- file.path(repo, ".gittargets_gitignore")
  to <- file.path(repo, ".gitignore")
  if (file.exists(from)) {
    file.rename(from = from, to = to)
  }
  invisible()
}

tar_git_gitignore_lines <- function() {
  c(
    "# Generated by gittargets: do not edit by hand",
    ".gitignore",
    ".gittargets_gitignore"
  )
}

tar_git_stub_path <- function(repo) {
  file.path(repo, ".gittargets")
}

tar_git_stub_write <- function(repo) {
  path <- tar_git_stub_path(repo)
  uuid <- uuid::UUIDgenerate(use.time = NA, n = 1L)
  writeLines(uuid, path)
}
